/*
 * Copyright 2000-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.services.security;

import java.util.HashMap;
import java.util.Iterator;

import junit.framework.Test;
import junit.framework.TestSuite;

import org.apache.jetspeed.om.security.GroupRole;
import org.apache.jetspeed.om.security.JetspeedRoleFactory;
import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.StringUtils;
import org.apache.turbine.util.TurbineConfig;

/**
 * Unit test for RoleManagement interface
 * 
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: TestRoleManagement.java,v 1.1 2004/04/07 22:02:43 jford Exp $
 */

public class TestRoleManagement extends JetspeedTestCase {    

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestRoleManagement( String name ) {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[]) 
    {
        junit.awtui.TestRunner.main( new String[] { TestRoleManagement.class.getName() } );
    }
 
    public void setup() 
    {
        //System.out.println("Setup: Testing Turbine Role Management");         
    }

    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite() 
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestRoleManagement.class );
    }

    /**
     * Tests getRoles method
     * @throws Exception
     */

    public void testGetRoles() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;
        HashMap map = new HashMap();

        try
        {
            Iterator roles = service.getRoles();
            while (roles.hasNext())
            {
                role = (Role)roles.next();
                map.put(role.getName(), role);
            }
            assertTrue(map.get("user") != null);
            assertTrue(map.get("admin") != null);
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getRoles Test OK ");

    }

    /**
     * Tests getRoles method
     * @throws Exception
     */

    public void testGetRolesForUser() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;
        HashMap map = new HashMap();

        try
        {
            Iterator roles = service.getRoles("turbine");
            while (roles.hasNext())
            {
            	GroupRole gr = (GroupRole) roles.next();
                role = gr.getRole();
                map.put(role.getName(), role);
                System.out.println("role = " + role.getName());
            }
            assertTrue(map.get("user") != null);
            assertTrue(map.get("admin") == null);

            map.clear();
            roles = service.getRoles("admin");            
            while (roles.hasNext())
            {
                GroupRole gr = (GroupRole)roles.next();
                role = gr.getRole();
                map.put(role.getName(), role);
                System.out.println("role = " + role.getName());
            }
            assertTrue(map.get("user") != null);
            assertTrue(map.get("admin") != null);
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getRoles Test OK ");

    }

    /**
     * Tests addRole method 
     * @throws Exception
     */

    public void testAddRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            role = JetspeedRoleFactory.getInstance();
            role.setName("bogus");
            service.addRole(role);
            System.out.println("new role id = " + role.getId());
            assertTrue(role.getId() != null);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            role = JetspeedRoleFactory.getInstance();
            role.setName("bogus");
            service.addRole(role);
            fail("Should've thrown a dup key exception on role");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);           
        }

        System.out.println("Completed addRole Test OK ");

    }

    /**
     * Tests getRemoveRole method 
     * @throws Exception
     */

    public void testRemoveRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            service.removeRole("bogus");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.removeRole("catchmeifyoucan");
            fail("Should've thrown a not found exception on role");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);
        }

        System.out.println("Completed addRole Test OK ");

    }

    /**
     * Tests getRole method
     * @throws Exception
     */

    public void testGetRole() throws Exception 
    {
        RoleManagement service = getService();

        try
        {
            Role role = service.getRole("user");
            System.out.println("*** role nm = " + role.getName());
            System.out.println("*** role id = " + role.getId());
            assertTrue(role.getName().equals("user"));
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getRole Test OK ");

    }

    /**
     * Tests saveRole method 
     * @throws Exception
     */

    public void testSaveRole() throws Exception 
    {
        RoleManagement service = getService();

        try
        {
            Role role = service.getRole("user");
            service.saveRole(role);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed saveRole Test OK ");

    }

    /**
     * Tests grantRole method 
     * @throws Exception
     */
    public void testGrantRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            service.grantRole("turbine", "admin");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.grantRole("baduser", "admin");
            fail("Should've thrown a bad user exception on grant");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);           
        }
        try
        {
            service.grantRole("turbine", "badrole");
            fail("Should've thrown a bad role exception on grant");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);           
        }

        System.out.println("Completed grantRole Test OK ");

    }

    /**
     * Tests revokeRole method 
     * @throws Exception
     */
    public void testRevokeRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            service.revokeRole("turbine", "admin");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.revokeRole("baduser", "admin");
            fail("Should've thrown a bad user exception on revoke");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);           
        }

        System.out.println("Completed revokeRole Test OK ");

    }

    /**
     * Tests hasRole method 
     * @throws Exception
     */
    public void testHasRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            boolean has = service.hasRole("admin", "admin");
            assertTrue(true == has);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            boolean has = service.hasRole("turbine", "admin");
            assertTrue(false == has);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed hasRole Test OK ");

    }

  /*
    Configuration object to run Turbine outside a servlet container
    ( uses turbine.properties )
    */
    private static TurbineConfig config = null;
    
    /**
    Sets up TurbineConfig using the system property:
    <pre>turbine.properties</pre>
    */
    static
    {
        try
        {
            config = new TurbineConfig( "webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }

    private static RoleManagement getService()
    {
        return (RoleManagement)TurbineServices
                .getInstance()
                .getService(RoleManagement.SERVICE_NAME);
    }

}





