/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.services.security.registry;

// Java imports
import java.util.Iterator;
import java.util.Vector;

import junit.awtui.TestRunner;
import junit.framework.Test;
import junit.framework.TestSuite;

import org.apache.jetspeed.om.BaseSecurityReference;
import org.apache.jetspeed.om.SecurityReference;
import org.apache.jetspeed.om.profile.Entry;
import org.apache.jetspeed.om.profile.psml.PsmlEntry;
import org.apache.jetspeed.om.registry.RegistryEntry;
import org.apache.jetspeed.om.registry.SecurityAccess;
import org.apache.jetspeed.om.registry.SecurityAllow;
import org.apache.jetspeed.om.registry.SecurityEntry;
import org.apache.jetspeed.om.registry.base.BaseSecurityAccess;
import org.apache.jetspeed.om.registry.base.BaseSecurityAllow;
import org.apache.jetspeed.om.registry.base.BaseSecurityEntry;
import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.services.JetspeedPortalAccessController;
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.Registry;
import org.apache.jetspeed.services.resources.JetspeedResources;
import org.apache.jetspeed.services.security.JetspeedGroupManagement;
import org.apache.jetspeed.services.security.JetspeedRoleManagement;
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.turbine.util.StringUtils;
import org.apache.turbine.util.TurbineConfig;

/**
 * TestAccessController
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id: TestAccessController.java,v 1.1 2004/04/07 22:02:43 jford Exp $
 */
public class TestAccessController extends JetspeedTestCase
{
 
    private static String ADMIN_PORTLET = "GlobalAdminPortlet"; // Portlet accessable by Admin user, role = admin
    private static SecurityReference adminSecurityRef = new BaseSecurityReference();
    private static String ALL_PORTLET = "HelloVelocity";  // Portlet accessable by Anonymous user
    private static SecurityReference defaultSecurityRef = new BaseSecurityReference();
    private static String TEST_GROUP = "Jetspeed";
    private static String TEST_SECURITY_PAGE = "SecurityTest";
    private static String USER_PORTLET = "SkinBrowser"; // Portlet accessable by general user, role = user
    private static String USERANON_PORTLET = "Welcome";  // Portlet viewable by Anonymous user, all by role=user
    private static SecurityReference userSecurityRef = new BaseSecurityReference();
    private static SecurityReference userAllAnonViewSecurityRef = new BaseSecurityReference();

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestAccessController( String name )
    {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[])
    {
        TestRunner.main( new String[]
        { TestAccessController.class.getName() } );
    }
    
    public void setup()
    {
        System.out.println("Setup: Testing categories of Profiler Service");
    }
    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite()
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestAccessController.class );
    }
    
    public void testVerifyEnvironment() throws Exception
    {
        assertEquals( "Using TurbineAccessController",
        "org.apache.jetspeed.services.security.registry.RegistryAccessController",
        JetspeedResources.getString("services.PortalAccessController.classname"));

        Registry.addEntry(Registry.SECURITY, (RegistryEntry) createSecurityEntry( "admin_only", null, "admin", null, "*"));
        assertNotNull( "Getting admin_only security " , Registry.getEntry( Registry.SECURITY, "admin_only"));
        Registry.addEntry(Registry.SECURITY, (RegistryEntry) createSecurityEntry( "user_only", null, "user", null, "*"));
        assertNotNull( "Getting user_only security " , Registry.getEntry( Registry.SECURITY, "user_only"));
        Registry.addEntry(Registry.SECURITY, (RegistryEntry) createSecurityEntry( "wide_open", null, null, null, "*"));
        assertNotNull( "Getting wide_open security " , Registry.getEntry( Registry.SECURITY, "wide_open"));

        /*
         * Create a security entry that looks look like the following
         *
         * <security-entry name="all_users-view_anon">
         *   <access action="*">
         *     <allow-if role="user"/>
         *   </access>
         *   <access action="view">
         *     <allow-if role="guest"/>
         *   </access>
         * </security-entry>
         */
        Registry.addEntry(Registry.SECURITY, (RegistryEntry) createSecurityEntry( "all_users-view_anon", null, "user", null, "*"));
        assertNotNull( "Getting all_users-view_anon security " , Registry.getEntry( Registry.SECURITY, "all_users-view_anon"));
        SecurityEntry secEntry = (SecurityEntry) Registry.getEntry( Registry.SECURITY, "all_users-view_anon");
        Vector accessVector = secEntry.getAccesses();
        assertEquals( "Getting number of accesses for all_users-view_anon", 1, accessVector.size());
        BaseSecurityAllow allowElement = new BaseSecurityAllow();
        allowElement.setRole("guest");
        Vector allowVector = new Vector();
        allowVector.addElement(allowElement);
        BaseSecurityAccess accessElement  = new BaseSecurityAccess();
        accessElement.setAction("view");
        accessElement.setAllows( allowVector );
        accessVector.addElement(accessElement);
        secEntry.setAccesses(accessVector);
        assertEquals( "Getting number of accesses for all_users-view_anon", 2, secEntry.getAccesses().size());
        
        // Verify users and their groups
        assertNotNull( "Getting admin user", JetspeedSecurity.getUser("admin"));
        assertTrue( "Admin user has Admin role", JetspeedRoleManagement.hasRole("admin","admin"));
        assertTrue( "Admin user has User role", JetspeedRoleManagement.hasRole("admin","user"));
        assertNotNull( "Getting turbine user", JetspeedSecurity.getUser("turbine"));
        assertTrue( "Turbine user does not have Admin role", !JetspeedRoleManagement.hasRole("turbine","admin"));
        assertTrue( "Turbine user has User role", JetspeedRoleManagement.hasRole("turbine","user"));
        assertNotNull( "Getting anonymous user", JetspeedSecurity.getAnonymousUser());
        assertTrue( "anonymous user does not have Admin role", !JetspeedRoleManagement.hasRole(JetspeedSecurity.getAnonymousUser().getUserName(),"admin"));
        assertTrue( "anonymous user does not have User role", !JetspeedRoleManagement.hasRole(JetspeedSecurity.getAnonymousUser().getUserName(),"user"));
        assertTrue( "anonymous user does not have Guest role", JetspeedRoleManagement.hasRole(JetspeedSecurity.getAnonymousUser().getUserName(),"guest"));
        
        assertNotNull( "adminSecurityRef", adminSecurityRef);
        adminSecurityRef.setParent("admin_only");
        assertNotNull( "Getting security for " + adminSecurityRef.getParent(), Registry.getEntry( Registry.SECURITY, adminSecurityRef.getParent()));

        assertNotNull( "userSecurityRef", userSecurityRef);
        userSecurityRef.setParent("user_only");
        assertNotNull( "Getting security for " + userSecurityRef.getParent(), Registry.getEntry( Registry.SECURITY, userSecurityRef.getParent()));

        assertNotNull( "defaultSecurityRef", defaultSecurityRef);
        defaultSecurityRef.setParent("wide_open");
        assertNotNull( "Getting security for " + defaultSecurityRef.getParent(), Registry.getEntry( Registry.SECURITY, defaultSecurityRef.getParent()));

        assertNotNull( "userAllAnonViewSecurityRef", userAllAnonViewSecurityRef);
        userAllAnonViewSecurityRef.setParent("all_users-view_anon");
        assertNotNull( "Getting security for " + userAllAnonViewSecurityRef.getParent(), Registry.getEntry( Registry.SECURITY, defaultSecurityRef.getParent()));

    }
        
    public void testRequiredActions() throws Exception
    {
        JetspeedUser adminUser = (JetspeedUser) JetspeedSecurity.getUser("admin");
        assertNotNull( "Getting admin user", adminUser);
        adminUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser turbineUser = (JetspeedUser) JetspeedSecurity.getUser("turbine");
        assertNotNull( "Getting turbine user", turbineUser);
        turbineUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser anonymousUser =  (JetspeedUser) JetspeedSecurity.getAnonymousUser();
        assertNotNull( "Getting anonymous user", anonymousUser);
        Entry adminEntry = createEntry( ADMIN_PORTLET, "ST_01.admin", adminSecurityRef);
        Entry userEntry = createEntry( USER_PORTLET, "ST_01.user", userSecurityRef);
        Entry allEntry = createEntry( ALL_PORTLET, "ST_01.all", defaultSecurityRef);
        Entry userAnonEntry = createEntry( USERANON_PORTLET, "ST_01.userAnon", userAllAnonViewSecurityRef);
        
        assertEquals( "Admin user has view access to " + ADMIN_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, adminEntry, "view"));
        assertEquals( "Turbine user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( turbineUser, adminEntry, "view"));
        assertEquals( "Anonymous user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, adminEntry, "view"));
        
        assertEquals( "Admin user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, userEntry, "view"));
        assertEquals( "Turbine user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, userEntry, "view"));
        assertEquals( "Anonymous user DOES NOT have view access to " + USER_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, userEntry, "view"));
        
        assertEquals( "Admin user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, allEntry, "view"));
        assertEquals( "Turbine user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, allEntry, "view"));
        assertEquals( "Anonymous user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( anonymousUser, allEntry, "view"));

        assertEquals( "Admin user has view access to " + USERANON_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, userAnonEntry, "view"));
        assertEquals( "Admin user has maximize access to " + USERANON_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, userAnonEntry, "maximize"));
        assertEquals( "Anonymous user has view access to " + USERANON_PORTLET, true, JetspeedPortalAccessController.checkPermission( anonymousUser, userAnonEntry, "view"));
        assertEquals( "Anonymous user has maximize access to " + USERANON_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, userAnonEntry, "maximize"));
    }

	public void testRolesAndGroups() throws Exception
	{
		/*
		 * Create a security entry that looks look like the following
		 *
		 * <security-entry name="powerusers_all-anon_view">
		 *   <access action="*">
		 *     <allow-if group="apache" role="admin"/>
		 *   </access>
		 * </security-entry>
		 */
		
		assertEquals( "Using RegistryAccessController",
		"org.apache.jetspeed.services.security.registry.RegistryAccessController",
		JetspeedResources.getString("services.PortalAccessController.classname"));

		SecurityEntry paav = createSecurityEntry( "powerusers_all-anon_view", "apache", "admin", null, "*");
		Registry.addEntry(Registry.SECURITY, (RegistryEntry) paav);
		assertNotNull( "Getting powerusers_all-anon_view" , Registry.getEntry( Registry.SECURITY, "powerusers_all-anon_view"));
		
		SecurityEntry secEntry = (SecurityEntry) Registry.getEntry( Registry.SECURITY, "powerusers_all-anon_view");
		Vector accessVector = secEntry.getAccesses();
		
		for (Iterator it = accessVector.iterator(); it.hasNext();)
		{
			SecurityAccess access = (SecurityAccess) it.next(); 
			System.out.println("Action:" + access.getAction().toString());
			
			Vector allAllows = access.getAllAllows();
			for (Iterator it1 = allAllows.iterator(); it1.hasNext();)
			{
				SecurityAllow allow = (SecurityAllow) it1.next(); 
				System.out.println("Allow group: " + allow.getGroup() + ", role: " + allow.getRole() + ", user: " + allow.getUser());				
			}
		}		
		
		SecurityReference secRef = new BaseSecurityReference();
		secRef.setParent("powerusers_all-anon_view");
		Entry adminEntry = createEntry(ADMIN_PORTLET, "ST_01.apache.admin", secRef);
				
		JetspeedUser turbineUser = (JetspeedUser) JetspeedSecurity.getUser("turbine");
		assertNotNull( "Getting turbine user", turbineUser);
		turbineUser.setHasLoggedIn(Boolean.TRUE);

		try
		{
			JetspeedGroupManagement.joinGroup("turbine", "apache", "admin");
		}
		catch (Exception e)
		{
			e.printStackTrace();			
		}
		
		assertEquals( "Turbine user has view access to " + ADMIN_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, adminEntry, "view"));

		try
		{
			JetspeedGroupManagement.unjoinGroup("turbine", "apache", "admin");
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}

		assertEquals( "Turbine user has no view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( turbineUser, adminEntry, "view"));		
	}

	/*
     * Setup Turbine environment
     */
    
    /*
     * Configuration object to run Turbine outside a servlet container
     * ( uses turbine.properties )
     */
    private static TurbineConfig config = null;
    
    /*
     * Sets up TurbineConfig using the system property:
     * <pre>turbine.properties</pre>
     */
    static
    {
        try
        {
            config = new TurbineConfig( "webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }
    
    private PsmlEntry createEntry(String parent, String entryId, SecurityReference security)
    {
        PsmlEntry entry = new PsmlEntry();
        entry.setParent( parent);
        if (entryId != null)
            entry.setId( entryId);
        if (security != null)
            entry.setSecurityRef( security);
        return entry;
    }

    private SecurityEntry createSecurityEntry( String name, String group, String role, String user, String action)
    {
        Vector allowVector = null;
        if (role != null || group != null || user != null)
        {
            BaseSecurityAllow allowElement = new BaseSecurityAllow();
            allowElement.setRole(role);
            allowElement.setGroup(group);
            allowElement.setUser(user);
            allowVector = new Vector();
            allowVector.addElement(allowElement);
        }
        
        BaseSecurityAccess accessElement  = new BaseSecurityAccess();
        accessElement.setAction(action);
        accessElement.setAllows( allowVector );
        Vector accessVector = new Vector();
        accessVector.addElement(accessElement);
        
        BaseSecurityEntry securityEntry = new BaseSecurityEntry();
        securityEntry.setName(name);
        securityEntry.setAccesses( accessVector);
        return securityEntry;
    }
    
}
