<?php
include_once '../lib/cumas_config.php';
include_once '../lib/libutil';
include_once '../lib/libsendmail';

define("LOG_DEFAULT", LOG_LOCAL4);

/**
 *設定ファイル読み込み処理
 */
try {
    $config = new CuMAS_Config();
} catch (CuMAS_Exception $e) {
    $e->log(LOG_DEFAULT, __FILE__);
    exit;
}
$logFacility = DgCommon_set_logfacility($config->syslogfascility);

/**
 *DB操作
 */

/**
 * task_tabテーブルからta_postが現在時刻よりも古いレコードを取得する 
 */
function getAllReserveTask($pdo)
{
    $nowdate = date(DB_TIMESTAMP_FMT);
    $sql = "SELECT * FROM task_tab where ta_post < ?";    
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array($nowdate));
    $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
    return $result;
}

/**
 * task_tabテーブルからta_postが現在時刻よりも古いレコードを取得する
 */
function getAllActiveUsers($pdo)
{
    $sql = "SELECT us_id, us_name,us_mail,us_active" 
           ." FROM user_tab WHERE us_active = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array(TRUE));

    $result = $stmt->fetchAll(PDO::FETCH_ASSOC);

    return $result;
}

/**
 * user_tabからus_idを基にしてユーザの情報を取得
 */
function getAllUsers($pdo, $us_id)
{
    $sql = "SELECT us_id, us_name,us_mail,us_active"
           ." FROM user_tab";

    $stmt = $pdo->prepare($sql);
    $stmt->execute(array($us_id));
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

    return $results;
}

function mkKeyValueUsers($allUsers)
{
    $arrUsers = array();

    foreach ($allUsers as $users) {
        $arrUsers[$users["us_id"]] = array(
            $users["us_name"],
            $users["us_mail"],
        ); 
    }

    return $arrUsers;
}

/**
 * user_tabからus_idによってus_mailを取得する
 */
function getUserMailById($datausers, $id)
{
    foreach ($datausers as $user) {
        if ($user["us_id"] === $id) {
            return $user["us_mail"];
	}	    
    }
    return "";
}

/**
 * 引数からカテゴリIDを取得
 */
function getCategoryOfNewJob($pdo, $ca_ident)
{
    $sql = "SELECT * from category_tab where ca_id = ? AND ca_active = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array($ca_ident, True));
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result;
}

/**
 * mail_tabに登録
 */
function insertMailTab($pdo, $data)
{
    $sql = ("INSERT INTO mail_tab"
         .  " (ma_message_id,ma_reference_id,ma_date,ma_from_addr,ma_subject)"
         .  " VALUES"
         .  " (?,?,?,?,?) "
         .  " RETURNING ma_id");

    $stmt = $pdo->prepare($sql);
    $stmt->execute(array(
                        $data['messageId'],
                        $data['references'],
                        $data['date'],
                        $data['fromAddress'],
                        $data['subject'] ?: 'no subject'));

    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result['ma_id'];
}

/**
 * contact_tabに登録
 */
function insertContactTab($pdo, $contactData)
{
    $sql = "INSERT INTO contact_tab("
         . " co_us_id"
         . ",co_inquiry"
         . ",co_lastupdate"
         . ",co_ma_id"
         . ",ca_id"
         . ",co_comment"
         . " )"
         . " VALUES"
         . " (?,?,?,?,?,?)"
         . " RETURNING co_id";

    $stmt = $pdo->prepare($sql);
    $stmt->execute(array(
        $contactData["co_us_id"],
        $contactData["co_inquiry"],
        $contactData["co_lastupdate"],
        $contactData["co_ma_id"],
        $contactData["ca_id"],
        $contactData["co_comment"],
    ));

    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $co_id = $result['co_id'];

    $sql = "UPDATE contact_tab SET co_parent = $co_id WHERE co_id = $co_id";
    $pdo->query($sql);
    return $co_id;
}

/**
 * contact_mail_tabに登録
 */
function insertContactMailTab($pdo, $co_id, $ma_id)
{
    $sql = "INSERT INTO contact_mail_tab "
         . "(co_id,ma_id) VALUES (?,?)";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array($co_id, $ma_id));
}

/**
 * contact_tabのco_us_idを更新
 */
function updateCoUsId($pdo, $co_id, $us_id)
{
    $sql = "UPDATE contact_tab SET co_us_id = $us_id WHERE co_id = $co_id";
    $stmt = $pdo->query($sql);
}

/**
 * 毎週モードの場合、task_tabを更新する
 */
function updateTaskTabWeekMode($pdo, $ta_id, $ta_post)
{
    $sql = "UPDATE task_tab SET ta_post = ? WHERE ta_id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array($ta_post, $ta_id));
}

/**
 * ta_idを基にしてtask_tabからタスクを削除する
 */
function deleteTaskTab($pdo, $ta_id)
{
    $sql = "DELETE FROM task_tab WHERE ta_id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array($ta_id));
}

function updateEveryMonthMode($next_num,  $ta_post_parts,
                        $cur_year, $ta_post_m, $taskdata, $pdo)
{
    if (($ta_post_m + $next_num) > 12) {
        $next_month = $ta_post_m + $next_num - 12;
        $year = $cur_year + 1;
    } else {
        $next_month = $ta_post_m + $next_num;
        $year = $cur_year;
    }

    /* 日付の妥当性をチェック */
    if (checkdate($next_month, $taskdata["ta_repday"], $year)) {
	$ta_post_new = $year . "-"
                         . $next_month . "-"
                         . $taskdata["ta_repday"];
        /* 時刻は元のまま付ける */
        $ta_post_new = date(DB_TA_DATE_FMT, strtotime($ta_post_new))
                         . " " . $ta_post_parts[1];

	/* task_tabを更新 */ 
	updateTaskTabWeekMode($pdo, $taskdata["ta_id"],
                                  $ta_post_new);
        return true;
    }

    return false;
}

/**
 * task_tabの更新を行う
 */
function updateTaskTab($pdo, $taskdata)
{
    /* 繰り返さないモード */
    if ($taskdata["ta_repmode"] === 0) {
        /* 現在処理している実行タスクをDBから削除する */
        deleteTaskTab($pdo, $taskdata["ta_id"]);

    /* 毎週 */
    } else if ($taskdata["ta_repmode"] === 1) {
        $afterdays_post = 7;	     
        $ut_ta_post = strtotime($taskdata["ta_post"]);
        $ut_ta_post = strtotime("+$afterdays_post day", $ut_ta_post);
        $ta_post_new = date(DB_TA_POST_FMT, $ut_ta_post);

	/* task_tabの更新 */
	updateTaskTabWeekMode($pdo, $taskdata["ta_id"], $ta_post_new);

    /* 毎月 */
    } else if ($taskdata["ta_repmode"] === 2) {

        /* スペースで分析 */
        $ta_post_parts = explode(" ", $taskdata["ta_post"]);

        /* task_tab.ta_postの現在の年を取得 */
        $cur_year = date('Y',  strtotime($taskdata["ta_post"]));

        /* task_tab.ta_postの現在の月を取得 */
        $ta_post_m =  date('n', strtotime($taskdata["ta_post"]));

        /* 次月 */
        $ret = updateEveryMonthMode(1, $ta_post_parts, $cur_year,
                                    $ta_post_m, $taskdata, $pdo);
        if ($ret === false) {
            /* 次の次月 */
            $ret = updateEveryMonthMode(2, $ta_post_parts, $cur_year,
                                        $ta_post_m, $taskdata, $pdo);
        }

    /* 毎月末  */
    } else if ($taskdata["ta_repmode"] === 3) {

        /* スペースで分析 */
        $ta_post_parts = explode(" ", $taskdata["ta_post"]);

        /* 翌月末を採用 */
        $ta_post_new = date('Y-m-d',
            strtotime('last day of next month' . $taskdata["ta_post"]));
 
        $ta_post_new .= " " . $ta_post_parts[1]; 
	
        /* task_tabを更新 */ 
	updateTaskTabWeekMode($pdo, $taskdata["ta_id"], $ta_post_new);
    }
}

/*
 * main処理
 */

/* 引数のチェックを行わない */

/* 他のプログラムが起動していないかどうかチェック */
$ret= check_other_process($argv);
if (!$ret) {
    // 引数の数が0個であるとき、メールを登録せずに終了
    Cumas_Exception::log_s(
        $logFacility,
        __FILE__,
        "Multiple startup."
    );
    exit(1);
}

//DB登録
try {
    //DB接続
    $dsn = "pgsql:host=$config->dbserver;"
         . "port=$config->dbport;dbname=$config->dbname";

    $pdo = new PDO($dsn, $config->dbuser, $config->dbpasswd);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    //トランザクション開始
    $pdo->beginTransaction();
    $stmt = $pdo->query(
          "LOCK TABLE contact_tab,mail_tab,contact_mail_tab,"
	 ."category_tab,task_tab IN ACCESS EXCLUSIVE MODE"
     );

    /* データベースを検索 */
    // ta_postが現在時刻よりも古い
    $reserveTasks = getAllReserveTask($pdo);

    /* タスク作成対象を存在しない場合 */
    if (count($reserveTasks) === 0) {
        empty($db) ?: $db->rollBack();
        /* 静かに終了 */
        /*
        Cumas_Exception::log_s(
            $logFacility,
            __FILE__,
            "Task creation target does not exist."
        );
        */
        exit(0);
    }

    /* user_tabテーブルからus_activeがtrueである */
    $activeUsers = getAllActiveUsers($pdo);
    if (count($reserveTasks) === 0) {
        Cumas_Exception::log_s(
            $logFacility,
            __FILE__,
            "Active User does not exist."
        );
    }

    /* 配列を調整 */
    $activeUsers = mkKeyValueUsers($activeUsers);

    foreach ($reserveTasks as $task) {

        /* アクティブカテゴリを取得 */
        $categoryOfTask = getCategoryOfNewJob($pdo, $task["ta_category"]);

        if (empty($categoryOfTask)) {
            /* アクティブのテゴリ識別名がDBに存在しない */
            $logmsg = sprintf("No such active category identification name (%s)", $task['ta_category']);
            Cumas_Exception::log_s(
                $logFacility,
                __FILE__,
                $logmsg
            );
            continue;
        }

        /* 登録者は非アクティブの場合、該当タスクを作成しない */
        if (!isset($activeUsers[$task["ta_registuser"]])) {
            Cumas_Exception::log_s(
                $logFacility,
                __FILE__,
               sprintf( "Not create job because user is disabled (user_id = %s, task_id = %s)", 
                 $task["ta_registuser"],
                 $task['ta_id']
               )
            );
            continue;
        }

	/* 登録データの編集 */
        $mailData['fromAddress'] = $activeUsers[$task["ta_registuser"]][1];
        $mailData['date']        = date("Y-m-j H:i:s",
                                         strtotime($task["ta_post"])) ?: null;
        $mailData['subject']     = $task["ta_subject"];
        $mailData['body']        = $task["ta_body"];
        $mailData['references']  = "";
        $mailData['messageId']   = mkMsgID();
        $mailData['registuser']  = $task["ta_registuser"]; 
        $mailData['body']        = $task["ta_body"]; 
        $mailData['comment']     = $task["ta_comment"]; 

	/* mail_tabに登録 */
        $ma_id = insertMailTab($pdo, $mailData);

        if ($task["ta_user"] === 0) {
	    $contactData["co_us_id"]  = null;
        } else {
	    $contactData["co_us_id"]  = $task["ta_user"];
        }

	$contactData["co_inquiry"]    = $task["ta_post"];
	$contactData["co_lastupdate"] = date(DB_TIMESTAMP_FMT);
	$contactData["co_comment"]    = $task["ta_comment"];
	$contactData["co_ma_id"]      = $ma_id;
	$contactData["ca_id"]         = $task["ta_category"];
	$contactData["co_limit"]      = null;
	$contactData["co_comment"]    = $task["ta_comment"];

        $contactData["co_child_no"] = 0;
        $contactData["ca_name"] = $categoryOfTask["ca_name"];

	/* contact_tabに情報を登録 */
        $co_id= insertContactTab($pdo, $contactData);

	/* contact_mail_tabに情報を登録 */
        insertContactMailTab($pdo, $co_id, $ma_id);
        
        $contactData["co_id"] = $co_id;

        /* 本文をファイルに保存 */
        $dirname = sprintf("%02d", $ma_id % 100);
        $dirname = "{$config->mailsavedir}/{$dirname}";

        if (!file_exists($dirname)) {
            $result = mkdir($dirname, 0700);
            if ($result === false) {
                throw new CuMAS_Exception(
                    "Failed to create mail directory ($dirname)");
            }
        }

	/* 本文を格納するファイルをオープン */
        $fp = fopen("{$dirname}/{$ma_id}", "w");
        if ($fp === false) {
            throw new CuMAS_Exception(
                "Failed to open file ({$dirname}/{$ma_id})");
        }

        $result = fwrite($fp, $mailData['body']);
        fclose($fp);
        if ($result === false) {
            throw new CuMAS_Exception(
                "Failed to write mail text ({$dirname}/{$ma_id})");
        }

	/* task_tab更新 */
	updateTaskTab($pdo, $task);

        $ret = chmod("{$dirname}/{$ma_id}", 0400);
        /* 失敗した場合 */
        if ($ret === false) {
            throw new CuMAS_Exception(
                "Failed to chmod maildir ({$dirname}/{$ma_id})");
        }

        $pdo->commit();

        /* メール通知 */
        sendmailTask($logFacility, $mailData, $contactData, $activeUsers);
    }

} catch (PDOException $e) {
    empty($db) ?: $db->rollBack();
    $ce = new CuMAS_Exception($e->getMessage());
    $ce->log($logFacility, __FILE__);
    exit(1);
} catch (CuMAS_Exception $ce) {
    empty($db) ?: $db->rollBack();
    $ce->log($logFacility, __FILE__);
    exit(1);
}

exit(0);
