<?php
include_once '../lib/cumas_config.php';
define("LOG_DEFAULT", LOG_LOCAL4);

/**
 *設定ファイル読み込み処理
 */
try {
    $config = new CuMAS_Config();
} catch (CuMAS_Exception $e) {
    $e->log(LOG_DEFAULT, __FILE__);
    exit;
}
$logFacility = DgCommon_set_logfacility($config->syslogfascility);

/**
 *メールデータ解析
 */

function getMailData($rawMail)
{
    $resource = mailparse_msg_create();
    $ret = mailparse_msg_parse($resource, $rawMail);
    if ($ret === false) {
        mailparse_msg_free($resource);
        return false;
    }
    $structure = mailparse_msg_get_structure($resource);
    if ($structure === false) {
        mailparse_msg_free($resource);
        return false;
    }

    // マルチパートをケアしながら、メールの本文を探す
    // まず、text/plainを探す
    foreach ($structure as $part_id) {
        $parts = mailparse_msg_get_part_data(mailparse_msg_get_part($resource, $part_id));
        if ($parts === false) {
            mailparse_msg_free($resource);
            return false;
        }
        if ($part_id == 1) {
            $mail['headers'] = $parts['headers'];
        }
        if (!isset($parts['content-type'])) {
            continue;
        }
        if (strpos($parts['content-type'], "text/plain") === false) {
            continue;
        }
        if (isset($parts['content-disposition']) && strpos($parts['content-disposition'], "attachment") !== false) {
            continue;
        }

        $start = $parts['starting-pos-body'];
        $end   = $parts['ending-pos-body'];
        $mail['body']  = substr($rawMail, $start, $end - $start);
        $mail['encoding']  = $parts['transfer-encoding'];

        mailparse_msg_free($resource);
        return $mail;
    }

    // text/plainをもたないメールだった場合、text/htmlを探す
    foreach ($structure as $part_id) {
        $parts = mailparse_msg_get_part_data(mailparse_msg_get_part($resource, $part_id));
        if (!isset($parts['content-type'])) {
            continue;
        }
        if (strpos($parts['content-type'], "text/html") === false) {
            continue;
        }
        if (isset($parts['content-disposition']) && strpos($parts['content-disposition'], "attachment") !== false) {
            continue;
        }
        $start = $parts['starting-pos-body'];
        $end   = $parts['ending-pos-body'];
        $mail['body']  = substr($rawMail, $start, $end - $start);
        $mail['encoding'] = $parts['transfer-encoding'];

        mailparse_msg_free($resource);
        return $mail;
    }
    $mail['body'] = "";
    $mail['encoding'] = "";

    mailparse_msg_free($resource);
    return $mail;
}

/**
 *DB操作
 */

//引数からカテゴリIDを取得
function getCategoryIdOfNewJob($pdo, $ca_ident)
{
    $sql = "SELECT ca_id from category_tab where ca_ident = ?";    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$ca_ident]);

    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    return $result['ca_id'];
}

//mail_tabに登録
function insertMailTab($pdo, $data)
{
    $sql = ("INSERT INTO mail_tab"
         .  " (ma_message_id,ma_reference_id,ma_date,ma_from_addr,ma_subject) "
         .  " VALUES"
         .  " (?,?,?,?,?) "
         .  " RETURNING ma_id");

    $stmt = $pdo->prepare($sql);
    $stmt->execute(array(
                        $data['messageId'],
                        $data['references'],
                        $data['date'],
                        $data['fromAddress'],
                        $data['subject'] ?: 'no subject'));

    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result['ma_id'];
}

//contact_tabに登録
function insertContactTab($pdo, $us_id, $ma_id, $ca_id)
{
    $nowtime = date("Y-m-j H:i:s", time());

    $sql = "INSERT INTO contact_tab"
         . " (co_us_id,co_inquiry,co_lastupdate,co_ma_id,ca_id)"
         . " VALUES"
         . " (?,?,?,?,?)"
         . " RETURNING co_id";

    $stmt = $pdo->prepare($sql);
    $stmt->execute(array(
                        $us_id,
                        $nowtime,
                        $nowtime,
                        $ma_id,
                        $ca_id,
                    ));

    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $co_id = $result['co_id'];

    $sql = "UPDATE contact_tab SET co_parent = $co_id WHERE co_id = $co_id";
    $pdo->query($sql);
    return $co_id;
}

//contact_mail_tabに登録
function insertContactMailTab($pdo, $co_id, $ma_id)
{
    $sql = "INSERT INTO contact_mail_tab "
         . "(co_id,ma_id) VALUES (?,?)";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array($co_id, $ma_id));
}

//contact_tabのco_us_idを更新
function updateCoUsId($pdo, $co_id, $us_id)
{
    $sql = "UPDATE contact_tab SET co_us_id = $us_id WHERE co_id = $co_id";
    $stmt = $pdo->query($sql);
}

//新規ジョブか既存ジョブかの判定
function searchJobIdByMsgId($pdo, $config, $msgId, $ca_id)
{
    $sql_incomplete = implode(
        " or ",
        array_map(
            function ($s) {
                return "co_status = $s";
            },
            $config->incomplete
        )
    );

    $sql = "SELECT contact_tab.co_id, co_us_id"
         . " FROM contact_tab"
         . " JOIN contact_mail_tab ON contact_tab.co_id = contact_mail_tab.co_id"
         . " JOIN mail_tab ON mail_tab.ma_id= contact_mail_tab.ma_id"
         . " WHERE ma_message_id = ?"
         . " AND (ca_id = ?)"
         . " AND ({$sql_incomplete})";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([$msgId, $ca_id]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($result === false) {
        return null;
    }
    return $result;
}

//担当者の確認
function searchUsIdByFromAddress($pdo, $fromaddr)
{
    $sql = "SELECT us_id FROM user_tab WHERE us_mail = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array($fromaddr));
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($result === false) {
        return null;
    }
    return $result['us_id'];
}
/**
 *メールデータ登録先振り分け
 */

function isNewJob($pdo, $config, $data, $ca_id)
{
    $getData = array(
                    'co_id' => "",
                    'us_id' => null,
                    'flag'  => 0);       // 0->既存ジョブ(担当者あり)
                                         // 1->既存ジョブ(担当者無し)
                                         // 2->新規ジョブ

    //references有無確認
    if (empty($data['references'])) {
        $getData['us_id'] = searchUsIdByFromAddress($pdo, $data['fromAddress']);
        $getData['flag'] = 2;
        return $getData;
    }

    //referencesが存在する場合
    $result = searchJobIdByMsgId($pdo, $config, $data['references'], $ca_id);

    //co_idがない（元ジョブがない）
    if ($result === null) {
        $getData['us_id'] = searchUsIdByFromAddress($pdo, $data['fromAddress']);
        $getData['flag'] = 2;
        return $getData;
    }
    $getData['co_id'] = $result['co_id'];

    //co_us_idがない（担当者が未定）
    if ($result['co_us_id'] === null) {
        $getData['us_id'] = searchUsIdByFromAddress($pdo, $data['fromAddress']);

        //us_idがある(登録されているメールアドレス)
        if ($getData['us_id'] !== null) {
            $getData['flag'] = 1;
        }
    }
    return $getData;
}

/*
 * main処理
 */

//引数のチェック
if ($argc <= 1) {
    // 引数の数が0個であるとき、メールを登録せずに終了
    Cumas_Exception::log_s(
        $logFacility,
        __FILE__,
        "Category identification name must be specified as an argument."
    );
    exit(1);
}

//標準入力の取得
$rawMail = file_get_contents("php://stdin");

$mail_data = getMailData($rawMail);
if ($mail_data === false) {
    $mail_data = array(
                'headers' => array(
                'fromAddress' => "",
                'references'  => "",
                'messageId'   => "",
                'date'        => "",
                'subject'     => "no subject"),
                'body'        => $rawMail,
                'encoding'    => "");
    Cumas_Exception::log_s(
        $logFacility,
        __FILE__,
        "Invalid Format Mail Data"
    );

} else {
    //メールデータの整形
    $addr = mailparse_rfc822_parse_addresses($mail_data['headers']['from']);
    $fromaddress = filter_var($addr[0]['address'], FILTER_VALIDATE_EMAIL);
    if ($fromaddress === false) {
        $mailData['fromAddress'] = $mail_data['headers']['from'];
        Cumas_Exception::log_s(
            $logFacility,
            __FILE__,
            "Invalid Format fromAddress"
        );
    } else {
        $mailData['fromAddress'] = $fromaddress;
    }

    if (isset($mail_data['headers']['references'])) {
        $tmpreferences = preg_split("/\s+/", $mail_data['headers']['references'], null, PREG_SPLIT_NO_EMPTY);
        $mailData['references']  = trim(array_pop($tmpreferences), "<>");
    } else {
        $mailData['references']  = "";
    }

    $mailData['messageId'] = trim($mail_data['headers']['message-id'], " <>");
    $mailData['date']      = date("Y-m-j H:i:s", strtotime($mail_data['headers']['date'])) ?: null;

    // *note* mb_decode_mimeheaderがUTF-8 <-> ISO-2022-JPをうまく出来ない
    $defEnc = mb_internal_encoding();
    mb_internal_encoding('EUC-JP');
    $mailData['subject'] = mb_convert_encoding(
        mb_decode_mimeheader($mail_data['headers']['subject']),
        'UTF-8',
        'ISO-2022-JP,UTF-8,EUC-JP,SJIS'
    );
    mb_internal_encoding($defEnc);


    // Content-Transfer-Encodingがquoted-printable, base64のときデコード
    if ($mail_data['encoding'] == "quoted-printable") {
        $body = quoted_printable_decode($mail_data['body']);
    } else if ($mail_data['encoding'] == "base64") {
        $body = base64_decode($mail_data['body']);
    } else {
        $body = $mail_data['body'];
    }

    // 文字コードの順番はこれじゃないとだめ
    $mailData['body'] = mb_convert_encoding(
        $body,
        'UTF-8',
        'ISO-2022-JP,UTF-8,EUC-JP,SJIS'
    );

} 


//DB登録
try {
    //DB接続
    $dsn = "pgsql:host=$config->dbserver;"
         . "port=$config->dbport;dbname=$config->dbname";

    $pdo = new PDO($dsn, $config->dbuser, $config->dbpasswd);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    //トランザクション開始
    $pdo->beginTransaction();
    $stmt = $pdo->query("LOCK TABLE contact_tab,mail_tab,contact_mail_tab,category_tab IN ACCESS EXCLUSIVE MODE");

    // 引数の値のチェック
    $ca_id_ofNewJob = getCategoryIdOfNewJob($pdo, $argv[1]);
    if (empty($ca_id_ofNewJob)) {
        // 引数で指定したカテゴリ識別名がDBに存在しない
        throw new CuMAS_Exception("No such category identification name ("
                                  . "$argv[1])."
                                 );
    }

    //mail_tabに登録
    $ma_id = insertMailTab($pdo, $mailData);

    //各処理の決定
    $regiData = isNewJob($pdo, $config, $mailData, $ca_id_ofNewJob);

    //新規ジョブ
    if ($regiData['flag'] === 2) {
        $regiData['co_id'] = insertContactTab($pdo, $regiData['us_id'], $ma_id, $ca_id_ofNewJob);
    }
    //既存ジョブ（担当者有り）
    if ($regiData['flag'] === 1) {
        updateCoUsId($pdo, $regiData['co_id'], $regiData['us_id']);
    }
    insertContactMailTab($pdo, $regiData['co_id'], $ma_id);

    //本文をファイルに保存
    $dirname = sprintf("%02d", $ma_id % 100);
    $dirname = "{$config->mailsavedir}/{$dirname}";

    if (!file_exists($dirname)) {
        $result = mkdir($dirname, 0700);
        if ($result === false) {
            throw new CuMAS_Exception("Failed to create mail directory ($dirname)");
        }
    }

    $fp = fopen("{$dirname}/{$ma_id}", "w");
    if ($fp === false) {
        throw new CuMAS_Exception("Failed to open file ({$dirname}/{$ma_id})");
    }
    $result = fwrite($fp, $mailData['body']);
    if ($result === false) {
        throw new CuMAS_Exception("Failed to write mail text ({$dirname}/{$ma_id})");
    }

    fclose($fp);

    chmod("{$dirname}/{$ma_id}", 0400);
    $pdo->commit();

} catch (PDOException $e) {
    empty($db) ?: $db->rollBack();
    $ce = new CuMAS_Exception($e->getMessage());
    $ce->log($logFacility, __FILE__);
    exit(1);
} catch (CuMAS_Exception $ce) {
    empty($db) ?: $db->rollBack();
    $ce->log($logFacility, __FILE__);
    exit(1);
}

exit(0);
