/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "AxisTicsOptionWidget.h"
#include "AxisTics.h"

#include <QtCore/QCoreApplication>
#include <QtGui/QCheckBox>
#include <QtGui/QComboBox>
#include <QtGui/QLabel>
#include <QtGui/QLineEdit>
#include <QtGui/QStackedWidget>
#include <QtGui/QVBoxLayout>

AxisTicsOptionWidget::AxisTicsOptionWidget(QWidget *parent)
: AxisOptionWidget(parent), axisTics(0)
{
	setWindowTitle(tr("Tics"));
}

AxisTicsOptionWidget::~AxisTicsOptionWidget()
{
	delete axisTics;
}

void AxisTicsOptionWidget::construct()
{
	setWhatsThis(tr("set &lt;<i>axis</i>&gt;tics"));

	AxisOptionWidget::construct();
	for (int i = 0; !AxisTics::AXIS[i].isEmpty(); i++) {
		axisComboBox->addItem(AxisTics::AXIS[i]);
	}
}

void AxisTicsOptionWidget::init()
{
	delete axisTics;
	axisTics = new AxisTics;
	setWidget();
}

void AxisTicsOptionWidget::apply()
{
	AxisOptionWidget::apply();
	emit executed(axisTics->command());
}

void AxisTicsOptionWidget::constructWidget(QWidget *widget)
{
	onCheckBox = new QCheckBox;
	onCheckBox->setText(tr("Ac&tivation"));
	connect(onCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	locationComboBox = new QComboBox;
	for (int i = 0; AxisTics::LOCATION[i][0]; i++) {
		locationComboBox->addItem(QCoreApplication::translate(
					"AxisTics", AxisTics::LOCATION[i]));
	}
	connect(locationComboBox, SIGNAL(activated(int)),
			this, SIGNAL(modified()));

	mirrorCheckBox = new QCheckBox;
	mirrorCheckBox->setText(tr("&Mirroring"));
	connect(mirrorCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	rotationCheckBox = new QCheckBox;
	rotationCheckBox->setText(tr("&Rotation"));
	connect(rotationCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	startPosLineEdit = new QLineEdit;
	connect(startPosLineEdit, SIGNAL(textChanged(const QString &)),
			this, SIGNAL(modified()));

	QLabel *startPosLabel = new QLabel;
	startPosLabel->setText(tr("&Start:"));
	startPosLabel->setBuddy(startPosLineEdit);

	incPosLineEdit = new QLineEdit;
	connect(incPosLineEdit, SIGNAL(textChanged(const QString &)),
			this, SIGNAL(modified()));

	QLabel *incPosLabel = new QLabel;
	incPosLabel->setText(tr("&Increment:"));
	incPosLabel->setBuddy(incPosLineEdit);

	endPosLineEdit = new QLineEdit;
	connect(endPosLineEdit, SIGNAL(textChanged(const QString &)),
			this, SIGNAL(modified()));

	QLabel *endPosLabel = new QLabel;
	endPosLabel->setText(tr("&End:"));
	endPosLabel->setBuddy(endPosLineEdit);

	QGridLayout *sieLayout = new QGridLayout;
	sieLayout->addWidget(startPosLabel, 0, 0);
	sieLayout->addWidget(startPosLineEdit, 0, 1);
	sieLayout->addItem(new QSpacerItem(40, 20, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	sieLayout->addWidget(incPosLabel, 1, 0);
	sieLayout->addWidget(incPosLineEdit, 1, 1);
	sieLayout->addWidget(endPosLabel, 2, 0);
	sieLayout->addWidget(endPosLineEdit, 2, 1);

	QWidget *sieWidget = new QWidget;
	sieWidget->setLayout(sieLayout);

	labelsPosLineEdit = new QLineEdit;
	connect(labelsPosLineEdit, SIGNAL(textChanged(const QString &)),
			this, SIGNAL(modified()));

	QVBoxLayout *labelsPosLayout = new QVBoxLayout;
	labelsPosLayout->addWidget(labelsPosLineEdit);
	labelsPosLayout->addStretch();

	QWidget *labelsPosWidget = new QWidget;
	labelsPosWidget->setLayout(labelsPosLayout);

	QStackedWidget *positionStackedWidget = new QStackedWidget;
	positionStackedWidget->addWidget(new QWidget);
	positionStackedWidget->addWidget(sieWidget);
	positionStackedWidget->addWidget(labelsPosWidget);

	positionComboBox = new QComboBox;
	positionComboBox->addItem(tr("Auto"));
	positionComboBox->addItem(tr("Start/Increment/End"));
	positionComboBox->addItem(tr("Labels/Positions"));
	connect(positionComboBox, SIGNAL(activated(int)),
			positionStackedWidget, SLOT(setCurrentIndex(int)));
	connect(positionComboBox, SIGNAL(activated(int)),
			this, SIGNAL(modified()));

	// create and assemble layouts

	QVBoxLayout *posLayout = new QVBoxLayout;
	posLayout->addWidget(positionComboBox);
	posLayout->addWidget(positionStackedWidget);

	QWidget *posWidget = new QWidget;
	posWidget->setLayout(posLayout);

	QGridLayout *othersLayout = new QGridLayout;
	othersLayout->addWidget(locationComboBox, 0, 0);
	othersLayout->addItem(new QSpacerItem(40, 20, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 1);
	othersLayout->addWidget(mirrorCheckBox, 1, 0);
	othersLayout->addWidget(rotationCheckBox, 2, 0);
	othersLayout->addItem(new QSpacerItem(20, 40,
				QSizePolicy::Minimum, QSizePolicy::Expanding), 3, 0);

	QWidget *othersWidget = new QWidget;
	othersWidget->setLayout(othersLayout);

	ticsTabWidget = new QTabWidget;
	ticsTabWidget->addTab(posWidget, tr("&Position"));
	ticsTabWidget->addTab(othersWidget, tr("O&thers"));
	connect(onCheckBox, SIGNAL(toggled(bool)),
			ticsTabWidget, SLOT(setEnabled(bool)));

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(onCheckBox);
	mainLayout->addWidget(ticsTabWidget);
	mainLayout->addStretch();

	widget->setLayout(mainLayout);
}


void AxisTicsOptionWidget::setOption(int axis)
{
	axisTics->setFlag(axis, onCheckBox->isChecked());
	axisTics->setMirror(axis, mirrorCheckBox->isChecked());
	axisTics->setBorder(axis, !locationComboBox->currentIndex());
	axisTics->setRotate(axis, rotationCheckBox->isChecked());

	switch (positionComboBox->currentIndex()) {
	case 0:
		axisTics->setAutofreq(axis);
		break;
	case 1:
		axisTics->setIncrement(axis, incPosLineEdit->text(),
				startPosLineEdit->text(),
				endPosLineEdit->text());
		break;
	case 2:
		axisTics->setPosition(axis, labelsPosLineEdit->text().split(","));
	}
}


void AxisTicsOptionWidget::setAxisWidget(int axis)
{
	onCheckBox->setChecked(axisTics->flag(axis));

	ticsTabWidget->setEnabled(axisTics->flag(axis));

	mirrorCheckBox->setChecked(axisTics->mirrors(axis));

	locationComboBox->setCurrentIndex(!axisTics->isBorder(axis));

	rotationCheckBox->setChecked(axisTics->rotates(axis));

	if (axisTics->isAutofreq(axis)) {
		positionComboBox->setCurrentIndex(0);
	} else if (axisTics->isIncrement(axis)) {
		positionComboBox->setCurrentIndex(1);

		startPosLineEdit->setText(axisTics->start(axis));
		incPosLineEdit->setText(axisTics->increment(axis));
		endPosLineEdit->setText(axisTics->end(axis));
	} else if (axisTics->isPosition(axis)) {
		positionComboBox->setCurrentIndex(2);

		labelsPosLineEdit->setText(axisTics->position(axis).join(","));
	}
}
