/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "PlotOptionDialog.h"
#include "Plot.h"
#include "PlotModifierWidget.h"
#include "PlotOthersWidget.h"
#include "PlotStyleWidget.h"
#include "PlotTitleWidget.h"

#include <QtCore/QCoreApplication>
#include <QtGui/QListWidget>
#include <QtGui/QPushButton>
#include <QtGui/QSplitter>
#include <QtGui/QStackedWidget>
#include <QtGui/QVBoxLayout>

PlotOptionDialog::PlotOptionDialog(QWidget *parent)
: QDialog(parent)
{
	setModal(true);

	plotOthersWidget = new PlotOthersWidget;
	plotTitleWidget = new PlotTitleWidget;
	plotStyleWidget = new PlotStyleWidget;
	plotModifierWidget = new PlotModifierWidget;

	pagesStackedWidget = new QStackedWidget;
	pagesStackedWidget->addWidget(plotModifierWidget);
	pagesStackedWidget->addWidget(plotTitleWidget);
	pagesStackedWidget->addWidget(plotStyleWidget);
	pagesStackedWidget->addWidget(plotOthersWidget);

	contentsListWidget = new QListWidget;
	for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
		QWidget *widget = pagesStackedWidget->widget(i);
		contentsListWidget->addItem(widget->windowTitle());
		contentsListWidget->item(i)->setWhatsThis(widget->whatsThis());
	}
	contentsListWidget->setCurrentRow(0);
	connect(contentsListWidget,
			SIGNAL(currentItemChanged(QListWidgetItem *, QListWidgetItem *)),
			this, SLOT(changePage(QListWidgetItem *, QListWidgetItem *)));

	QSplitter *centerSplitter = new QSplitter;
	centerSplitter->addWidget(contentsListWidget);
	centerSplitter->addWidget(pagesStackedWidget);

	QPushButton *plotPushButton = new QPushButton;
	plotPushButton->setText(tr("&2D Plot Test"));
	plotPushButton->setAutoDefault(false);
	plotPushButton->setDefault(false);
	connect(plotPushButton, SIGNAL(clicked()), this, SLOT(plot()));

	QPushButton *splotPushButton = new QPushButton;
	splotPushButton->setAutoDefault(false);
	splotPushButton->setDefault(false);
	splotPushButton->setText(tr("&3D Plot Test"));
	connect(splotPushButton, SIGNAL(clicked()), this, SLOT(splot()));

	QPushButton *acceptPushButton = new QPushButton;
	acceptPushButton->setText(tr("&OK"));
	acceptPushButton->setAutoDefault(true);
	acceptPushButton->setDefault(true);
	connect(acceptPushButton, SIGNAL(clicked()), this, SLOT(accept()));

	QPushButton *applyPushButton = new QPushButton;
	applyPushButton->setText(tr("&Apply"));
	connect(applyPushButton, SIGNAL(clicked()), this, SLOT(apply()));

	QPushButton *cancelPushButton = new QPushButton;
	cancelPushButton->setText(tr("&Cancel"));
	connect(cancelPushButton, SIGNAL(clicked()), this, SLOT(reject()));

	// ------------------------ create layouts
	// ----------- assemble layouts and widgets

	QHBoxLayout *buttonLayout = new QHBoxLayout;
	buttonLayout->addStretch();
	buttonLayout->addWidget(plotPushButton);
	buttonLayout->addWidget(splotPushButton);
	buttonLayout->addWidget(acceptPushButton);
	buttonLayout->addWidget(applyPushButton);
	buttonLayout->addWidget(cancelPushButton);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(centerSplitter);
	mainLayout->addLayout(buttonLayout);

	setLayout(mainLayout);
}

void PlotOptionDialog::setPlot(Plot &plot)
{
	plot_ = &plot;
	bool isFunction = plot.isFunction();
	setWindowTitle(tr("Options of %1 %2")
			.arg(isFunction ? tr("Function") : tr("Data File"))
			.arg(isFunction ? plot.function() : plot.dataFile())
			+ " - " + QCoreApplication::applicationName());
	plotOthersWidget->setPlotAxis(plot.axis());
	plotTitleWidget->setPlotTitle(plot.titleSpecification());
	plotStyleWidget->setPlotStyle(plot.style());
	if (plot.isFunction()) {
		plotModifierWidget->setEnabled(false);
	} else {
		plotModifierWidget->setPlotModifier(plot.dataFileModifier());
	}
}

void PlotOptionDialog::accept()
{
	apply();
	QDialog::accept();
}

void PlotOptionDialog::apply()
{
	setPlotOption(*plot_);
}

void PlotOptionDialog::plot()
{
	Plot plot = *plot_;
	setPlotOption(plot);
	emit executed("plot " + plot.command());
}

// TODO use Splot
void PlotOptionDialog::splot()
{
	Plot plot = *plot_;
	setPlotOption(plot);
	emit executed("splot " + plot.command());
}

void PlotOptionDialog::setPlotOption(Plot &plot)
{
	plot.setAxis(plotOthersWidget->plotAxis());
	plot.setTitleSpecification(plotTitleWidget->plotTitle());
	plot.setStyle(plotStyleWidget->plotStyle());
	if (!plot.isFunction()) {
		plot.setDataFileModifier(plotModifierWidget->plotModifier());
	}
}

void PlotOptionDialog::changePage(QListWidgetItem *current,
		QListWidgetItem *previous)
{
	if (!current) {
		current = previous;
	}
	pagesStackedWidget->setCurrentIndex(contentsListWidget->row(current));
}
