/*
 * LeakChecker.cpp
 *
 * Copyright (C) 2001-2002 by Ito Yoshiichi.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

// DISABLE_CHECK_GLOBAL_NEW:
//   ũO[o new `FbNȂB

// DISABLE_CHECK_ZERO_ALLOCATE:
//   TCY0 callocAmallocArealloc `FbNȂB

// dg_DEBUG:
//   mہAɃbZ[W\B

#include "dgconfig.h"
#include "LeakChecker.h"

#ifndef _CRTDBG_MAP_ALLOC

#undef calloc
#undef malloc
#undef realloc
#undef free
#undef new

#include "DebugUtility.h"
#include "DebugMutex.h"
#include "MallocAllocator.h"
#include <map>
#include <list>

#ifdef dg_DEBUG
#  define dg_LC_TRACE(args) printf ## args
#else
#  define dg_LC_TRACE(args) (void)0
#endif

#define dg_LC_UNKNOWN_FILE "<unknown>"
#define dg_LC_UNKNOWN_LINE 0

dg_NAMESPACE_BEGIN(debug)

dg_USING_NAMESPACE_STD

// ---- class LeakCheckerRegistry ----

/**
 * ̓Imۂ̏ێB
 */
class LeakCheckerRegistry
{
  public:
	enum Method { UNKNOWN, CALLOC, MALLOC, REALLOC, FREE,
			NEW_OBJECT, NEW_ARRAY, DELETE_OBJECT, DELETE_ARRAY };

  public:
	/** VOg擾B */
	static LeakCheckerRegistry* instance();

	/** IsB */
	static void finalize();

	/**
	 * WXgL?.
	 * NAIɖu newAdelete ̏𐧌䂷邽߁B
	 */
	static bool isAvailable() { return s_instance != 0; }

	/** G[̌ԂB */
	size_t count() const;

	/** ʂ\B */
	void dump(FILE* fp) const;

	/**
	 * XgɏǉB.
	 * @param p      ̃AhXB
	 * @param method mۂvB
	 * @param n      ̃TCYB
	 * @param file   \[Xt@CB
	 * @param line   \[Xt@CsԍB
	 */
	void add(void* p, Method method, size_t n, const char* file, int line);

	/**
	 * XgB.
	 * @param p      ̃AhXB
	 * @param method vB
	 * @return       mۂvB
	 */
	Method remove(void* p, Method method);

	/** B */
	void clear();

  protected:
	/** ̓Imۂ̏B */
	struct Entry {
		Entry() : method(UNKNOWN), size(0), file(0), line(0) {}
		Entry(Method method, size_t n, const char* file, int line)
			: method(method), size(n), file(file), line(line) {}
		Method method;
		size_t size;
		const char* file;
		int line;
	};

	/** G[B */
	struct Error {
		Error() : address(0), method2(UNKNOWN) {}
		Error(void* address, Method method2, const Entry& entry)
			: address(address), method2(method2), entry(entry) {}
		void*  address;
		Method method2;
		Entry  entry;
	};

	typedef MallocAllocator<pair<const void*, Entry> > PairAllocator;
	typedef map<void*, Entry, less<void*>, PairAllocator> EntryMap;
	typedef list<Error, MallocAllocator<Error> > ErrorList;

  protected:
	LeakCheckerRegistry() {}
	~LeakCheckerRegistry();

	/**
	 * G[ǉB.
	 * @param address vꂽAhXB
	 * @param method2 vB
	 * @param entry   mێɓo^ꂽB
	 */
	void addError(void* address, Method method2, const Entry& entry)
		{ m_errors.push_back(Error(address, method2, entry)); }

	/** ̓Imۂ̏\B */
	static void printEntry(FILE* fp, void* p, const Entry& entry);

	/** G[\B */
	static void printError(FILE* fp, const Error& error);

  private:
	LeakCheckerRegistry(const LeakCheckerRegistry& rhs);
	LeakCheckerRegistry& operator=(const LeakCheckerRegistry& rhs);

  private:
	EntryMap  m_entries;
	ErrorList m_errors;
	mutable DebugMutex m_mutex;
	static LeakCheckerRegistry* s_instance;
};

LeakCheckerRegistry* LeakCheckerRegistry::s_instance = 0;

static char g_leakCheckerRegistry[sizeof(LeakCheckerRegistry)];

LeakCheckerRegistry* LeakCheckerRegistry::instance()
{
	if (s_instance == 0) {
		dg_LC_TRACE(("%s(%u): LeakCheckRegistry creating...\n", \
				DebugUtility::trimFileName(__FILE__), __LINE__));
//		static LeakCheckerRegistry registry;
//		s_instance = &registry;
		s_instance = new(g_leakCheckerRegistry) LeakCheckerRegistry();
	}
	return s_instance;
}

void LeakCheckerRegistry::finalize()
{
	if (s_instance == 0) return;
	s_instance->~LeakCheckerRegistry();
}

LeakCheckerRegistry::~LeakCheckerRegistry()
{
	s_instance = 0;
}

size_t LeakCheckerRegistry::count() const
{
	DebugMutex::AutoLock lock(&m_mutex);
	return m_entries.size() + m_errors.size();
}

static const char* g_methodNames[] = {
	"<unknown>", "calloc", "malloc", "realloc", "free",
	"new", "new[]", "delete", "delete[]", 0
};

void LeakCheckerRegistry::dump(FILE* fp) const
{
	DebugMutex::AutoLock lock(&m_mutex);
	fprintf(fp, "--------\n");
	fprintf(fp, "Memory Leaks:\n");
	EntryMap::const_iterator p = m_entries.begin();
	if (p != m_entries.end()) {
		do {
			printEntry(fp, p->first, p->second);
		} while (++p != m_entries.end());
	} else {
		fprintf(fp, "  (none)\n");
	}
	ErrorList::const_iterator p2 = m_errors.begin();
	if (p2 != m_errors.end()) {
		fprintf(fp, "Errors:\n");
		do {
			printError(fp, *p2);
		} while (++p2 != m_errors.end());
	}
}

void LeakCheckerRegistry::printEntry(FILE* fp, void* p, const Entry& entry)
{
	fprintf(fp, "  [%p] %s(%d): %s(%u)\n",
			p, DebugUtility::trimFileName(entry.file), entry.line,
			g_methodNames[entry.method], entry.size);
}

void LeakCheckerRegistry::printError(FILE* fp, const Error& error)
{
	const Entry& entry = error.entry;
	if (error.method2 != UNKNOWN) {
		fprintf(fp, "  [%p] %s(%d): %s(%u) -> %s\n", error.address,
				DebugUtility::trimFileName(entry.file), entry.line,
				g_methodNames[entry.method], entry.size,
				g_methodNames[error.method2]);
	} else {
		fprintf(fp, "  [%p] %s(%d): %s(%u) !!\n", error.address,
				DebugUtility::trimFileName(entry.file), entry.line,
				g_methodNames[entry.method], entry.size);
	}
}

void LeakCheckerRegistry::add(
		void* p, Method method, size_t n, const char* file, int line)
{
	if (p == 0) return;
	DebugMutex::AutoLock lock(&m_mutex);
	Entry entry(method, n, file, line);
	EntryMap::iterator pos = m_entries.find(p);
	if (pos != m_entries.end()) {
		addError(p, method, pos->second);
		pos->second = entry;
	} else {
		m_entries.insert(make_pair(p, entry));
	}
#ifndef DISABLE_CHECK_ZERO_ALLOCATE
	if (n == 0 && method != NEW_OBJECT && method != NEW_ARRAY) {
		// WARNING!! size == 0
		addError(p, UNKNOWN, Entry(method, n, file, line));
	}
#endif
}

LeakCheckerRegistry::Method
LeakCheckerRegistry::remove(void* p, Method method)
{
	if (p == 0) return UNKNOWN;
	DebugMutex::AutoLock lock(&m_mutex);
	EntryMap::iterator pos = m_entries.find(p);
	if (pos == m_entries.end()) {
#ifndef DISABLE_CHECK_GLOBAL_NEW
		addError(p, method,
				Entry(UNKNOWN, 0, dg_LC_UNKNOWN_FILE, dg_LC_UNKNOWN_LINE));
#endif
		return UNKNOWN;
	}
	const Entry& entry = pos->second;
	switch (entry.method) {
	  case CALLOC:
	  case MALLOC:
	  case REALLOC:
		if (method != REALLOC && method != FREE) {
			addError(p, method, entry);
		}
		break;
	  case NEW_OBJECT:
		if (method != DELETE_OBJECT) {
			addError(p, method, entry);
		}
		break;
	  case NEW_ARRAY:
		if (method != DELETE_ARRAY) {
			addError(p, method, entry);
		}
		break;
	  default:
		printf("ERROR!! method = %d\n", method);
	}
	m_entries.erase(pos);
	return entry.method;
}

void LeakCheckerRegistry::clear()
{
	DebugMutex::AutoLock lock(&m_mutex);
	m_entries.clear();
	m_errors.clear();
}

dg_NAMESPACE_END

// ---- standard C allocators ----

extern "C" {

void* dg_calloc(size_t n, size_t size, const char* file, int line)
{
	// ASSERT((n * size) != 0);
	void* p = ::calloc((n ? n : 1), size);
	LeakCheckerRegistry::instance()
			->add(p, LeakCheckerRegistry::CALLOC, n * size, file, line);
	dg_LC_TRACE(("calloc(%u, %u, %s, %d) = %p\n", \
			n, size, DebugUtility::trimFileName(file), line, p));
	return p;
}

void* dg_malloc(size_t n, const char* file, int line)
{
	// ASSERT(n != 0);
	void* p = ::malloc(n ? n : 1);
	LeakCheckerRegistry::instance()
			->add(p, LeakCheckerRegistry::MALLOC, n, file, line);
	dg_LC_TRACE(("malloc(%u, %s, %d) = %p\n", \
			n, DebugUtility::trimFileName(file), line, p));
	return p;
}

void* dg_realloc(void* p, size_t n, const char* file, int line)
{
	// ASSERT(n != 0);
	LeakCheckerRegistry::instance()->remove(p, LeakCheckerRegistry::REALLOC);
	void* p2 = ::realloc(p, (n ? n : 1));
	// ̍Ċ蓖ĂɎsƂ͌̃cB
	// \bhA(TCY)At@CAsԍ͍XVB
	LeakCheckerRegistry::instance()
			->add((p2 ? p2 : p), LeakCheckerRegistry::REALLOC, n, file, line);
	dg_LC_TRACE(("realloc(%p, %u, %s, %d) = %p\n", \
			p, n, DebugUtility::trimFileName(file), line, p2));
	return p2;
}

void dg_free(void* p, const char* file, int line)
{
	if (p == 0) return;
	LeakCheckerRegistry::instance()->remove(p, LeakCheckerRegistry::FREE);
	dg_LC_TRACE(("free(%p, %s, %d)\n", \
			p, DebugUtility::trimFileName(file), line));
	::free(p);
}

} /* extern "C" */

// ---- global new / delete ----

void* operator new(size_t n) dg_THROW(dg_STD::bad_alloc)
{
	void* p = ::malloc(n ? n : 1);
#ifndef DISABLE_CHECK_GLOBAL_NEW
	if (LeakCheckerRegistry::isAvailable()) {
		LeakCheckerRegistry::instance()->add(p, LeakCheckerRegistry::NEW_OBJECT,
				n, dg_LC_UNKNOWN_FILE, dg_LC_UNKNOWN_LINE);
	}
	dg_LC_TRACE(("operator new(%u) = %p\n", n, p));
#endif
	return p;
}

void* operator new[](size_t n) dg_THROW(dg_STD::bad_alloc)
{
	void* p = ::malloc(n ? n : 1);
#ifndef DISABLE_CHECK_GLOBAL_NEW
	if (LeakCheckerRegistry::isAvailable()) {
		LeakCheckerRegistry::instance()->add(p, LeakCheckerRegistry::NEW_ARRAY,
				n, dg_LC_UNKNOWN_FILE, dg_LC_UNKNOWN_LINE);
	}
	dg_LC_TRACE(("operator new[](%u) = %p\n", n, p));
#endif
	return p;
}

void* operator new(size_t n, const char* file, int line)
		dg_THROW(dg_STD::bad_alloc)
{
	void* p = ::malloc(n ? n : 1);
	LeakCheckerRegistry::instance()
			->add(p, LeakCheckerRegistry::NEW_OBJECT, n, file, line);
	dg_LC_TRACE(("operator new(%u, %s, %d) = %p\n", \
			n, DebugUtility::trimFileName(file), line, p));
	return p;
}

void* operator new[](size_t n, const char* file, int line)
		dg_THROW(dg_STD::bad_alloc)
{
	void* p = ::malloc(n ? n : 1);
	LeakCheckerRegistry::instance()
			->add(p, LeakCheckerRegistry::NEW_ARRAY, n, file, line);
	dg_LC_TRACE(("operator new[](%u, %s, %d) = %p\n", \
			n, DebugUtility::trimFileName(file), line, p));
	return p;
}

void operator delete(void* p) dg_NOTHROW
{
	if (p == 0) return;
#ifdef DISABLE_CHECK_GLOBAL_NEW
	if (LeakCheckerRegistry::isAvailable()) {
		LeakCheckerRegistry::Method method = LeakCheckerRegistry::instance()
				->remove(p, LeakCheckerRegistry::DELETE_OBJECT);
		if (method != LeakCheckerRegistry::UNKNOWN) {
			dg_LC_TRACE(("operator delete(%p)\n", p));
		}
	}
#else
	if (LeakCheckerRegistry::isAvailable()) {
		LeakCheckerRegistry::instance()
				->remove(p, LeakCheckerRegistry::DELETE_OBJECT);
	}
	dg_LC_TRACE(("operator delete(%p)\n", p));
#endif
	::free(p);
}

void operator delete[](void* p) dg_NOTHROW
{
	if (p == 0) return;
#ifdef DISABLE_CHECK_GLOBAL_NEW
	if (LeakCheckerRegistry::isAvailable()) {
		LeakCheckerRegistry::Method method = LeakCheckerRegistry::instance()
				->remove(p, LeakCheckerRegistry::DELETE_ARRAY);
		if (method != LeakCheckerRegistry::UNKNOWN) {
			dg_LC_TRACE(("operator delete[](%p)\n", p));
		}
	}
#else
	if (LeakCheckerRegistry::isAvailable()) {
		LeakCheckerRegistry::instance()
				->remove(p, LeakCheckerRegistry::DELETE_ARRAY);
	}
	dg_LC_TRACE(("operator delete[](%p)\n", p));
#endif
	::free(p);
}

#ifndef dg_NO_OPERATOR_DELETE_OVERLOADING

void operator delete(void* p, const char* file, int line) dg_NOTHROW
{
	if (p == 0) return;
	LeakCheckerRegistry::instance()
			->remove(p, LeakCheckerRegistry::DELETE_OBJECT);
	dg_LC_TRACE(("operator delete(%p, %s, %d)\n", \
			p, DebugUtility::trimFileName(file), line));
	::free(p);
}

void operator delete[](void* p, const char* file, int line) dg_NOTHROW
{
	if (p == 0) return;
	LeakCheckerRegistry::instance()
			->remove(p, LeakCheckerRegistry::DELETE_ARRAY);
	dg_LC_TRACE(("operator delete[](%p, %s, %d)\n", \
			p, DebugUtility::trimFileName(file), line));
	::free(p);
}

#endif /* !dg_NO_OPERATOR_DELETE_OVERLOADING */

dg_NAMESPACE_BEGIN(debug)

#ifdef _MSC_VER
static LeakChecker::Reporter g_leakCheckerReporter = 0;
#endif

// ---- class LeakChecker ----

LeakChecker::LeakChecker(Reporter reporter)
	: m_reporter(reporter)
{
#ifndef _MSC_VER
	LeakChecker::initialize();
#else
	g_leakCheckerReporter = reporter;
#endif
}

LeakChecker::~LeakChecker()
{
#ifndef _MSC_VER
	LeakChecker::finalize(m_reporter);
#endif
}

void LeakChecker::initialize()
{
	size_t count = LeakCheckerRegistry::instance()->count();
	if (count > 0) {
		printf("WARNING!! %u memories allocated\n", count);
		LeakCheckerRegistry::instance()->clear();
	}
}

void LeakChecker::finalize(Reporter reporter)
{
	if (reporter != 0) (*reporter)();
	size_t count = LeakCheckerRegistry::instance()->count();
	if (count > 0) {
		printf("WARNING!! %u memory leaks/errors\n", count);
	}
	LeakCheckerRegistry::finalize();
	printf("...done!!\n");
}

size_t LeakChecker::count()
{
	return LeakCheckerRegistry::instance()->count();
}

void LeakChecker::dump(FILE* fp)
{
	LeakCheckerRegistry::instance()->dump(fp);
}

void LeakChecker::clear()
{
	LeakCheckerRegistry::instance()->clear();
}

// ̐ݒŁAVXemۂ delete ɁA
// [ŇłB
#ifdef _MSC_VER
#  pragma warning(disable:4073)
#  pragma init_seg(lib)
class LeakCheckerMgr {
  public:
	LeakCheckerMgr()  { LeakChecker::initialize(); }
	~LeakCheckerMgr() { LeakChecker::finalize(g_leakCheckerReporter); }
};
static LeakCheckerMgr g_leakCheckerMgr;
#endif

dg_NAMESPACE_END

#else /* _CRTDBG_MAP_ALLOC */

#include <string.h>

// ---- standard C allocators ----

extern "C" {

void* dg_calloc(size_t n, size_t size, const char* file, int line)
	{ return _calloc_dbg(n, size, _NORMAL_BLOCK, file, line); }
void* dg_malloc(size_t n, const char* file, int line)
	{ return _malloc_dbg(n, _NORMAL_BLOCK, file, line); }
void* dg_realloc(void* p, size_t n, const char* file, int line)
	{ return _realloc_dbg(p, n, _NORMAL_BLOCK, file, line); }
void dg_free(void* p, const char* file, int line)
	{ _free_dbg(p, _NORMAL_BLOCK); }

} /* extern "C" */

dg_NAMESPACE_BEGIN(debug)

// ---- class LeakChecker ----

LeakChecker::LeakChecker(Reporter)
	: m_reporter(0)
{
	LeakChecker::initialize();
}

LeakChecker::~LeakChecker()
{
	LeakChecker::finalize();
}

void LeakChecker::initialize() {}

void LeakChecker::finalize(Reporter)
{
	_CrtSetReportMode(_CRT_WARN, _CRTDBG_MODE_FILE);
	_CrtSetReportFile(_CRT_WARN, _CRTDBG_FILE_STDOUT);
	int flag = _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG);
	_CrtSetDbgFlag(flag | _CRTDBG_LEAK_CHECK_DF);
	printf("...done!!\n");
}

size_t LeakChecker::count()
{
	_CrtMemState state;
	memset(&state, 0, sizeof(state));
	_CrtMemCheckpoint(&state);
	return state.lCounts[_NORMAL_BLOCK];
}

void LeakChecker::dump(FILE*)
{
	_CrtSetReportMode(_CRT_WARN, _CRTDBG_MODE_FILE);
	_CrtSetReportFile(_CRT_WARN, _CRTDBG_FILE_STDOUT);
	_CrtDumpMemoryLeaks();
}

void LeakChecker::clear() {}

dg_NAMESPACE_END

#endif /* _CRTDBG_MAP_ALLOC */
