/*$Id: NicoHttpClient.java 320 2010-05-24 05:17:32Z yuki $*/
package nicobrowser;

import java.net.URISyntaxException;
import java.util.Set;
import java.util.regex.Matcher;
import nicobrowser.entity.NicoContent;
import com.sun.syndication.feed.synd.SyndContentImpl;
import com.sun.syndication.feed.synd.SyndEntryImpl;
import com.sun.syndication.feed.synd.SyndFeed;
import com.sun.syndication.io.FeedException;
import com.sun.syndication.io.SyndFeedInput;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import javax.swing.text.MutableAttributeSet;
import javax.swing.text.html.HTML;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.text.html.parser.ParserDelegator;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import nicobrowser.entity.NicoContent.Status;
import nicobrowser.util.Result;
import nicobrowser.util.Util;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.params.ClientPNames;
import org.apache.http.client.params.CookiePolicy;
import org.apache.http.cookie.Cookie;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 *
 * @author yuki
 */
public class NicoHttpClient {

    private static Log log = LogFactory.getLog(NicoHttpClient.class);
    private final DefaultHttpClient http;
    private static final String LOGIN_PAGE =
            "https://secure.nicovideo.jp/secure/login?site=niconico";
    private static final String LOGOUT_PAGE =
            "https://secure.nicovideo.jp/secure/logout";
    private static final String WATCH_PAGE = "http://www.nicovideo.jp/watch/";
    private static final String MY_LIST_PAGE_HEADER =
            "http://www.nicovideo.jp/mylist/";
    private static final String MOVIE_THUMBNAIL_PAGE_HEADER =
            "http://www.nicovideo.jp/api/getthumbinfo/";
    private static final String GET_FLV_INFO = "http://www.nicovideo.jp/api/getflv/";
    private static final String SEARCH_HEAD = "http://www.nicovideo.jp/search/";
    private static final String SEARCH_TAIL = "?sort=v";
    private static final String ADD_MYLIST_PAGE = "http://www.nicovideo.jp/mylist_add/video/";
    private static final String GET_THREAD_KEY_PAGE = "http://www.nicovideo.jp/api/getthreadkey?thread=";

    public NicoHttpClient() {
        http = new DefaultHttpClient();
        http.getParams().setParameter(
                ClientPNames.COOKIE_POLICY, CookiePolicy.BROWSER_COMPATIBILITY);
    }

    /**
     * ニコニコ動画へログインする.
     * @param mail ログイン識別子(登録メールアドレス).
     * @param password パスワード.
     * @return 認証がOKであればtrue.
     */
    public boolean login(String mail, String password) throws URISyntaxException, HttpException, InterruptedException {
        boolean auth = false;
        HttpPost post = new HttpPost(LOGIN_PAGE);

        try {
            NameValuePair[] nvps = new NameValuePair[]{
                new BasicNameValuePair("mail", mail),
                new BasicNameValuePair("password", password),
                new BasicNameValuePair("next_url", "")
            };
            post.setEntity(new UrlEncodedFormEntity(Arrays.asList(nvps), "UTF-8"));

            //post.getParams().setCookiePolicy(CookiePolicy.BROWSER_COMPATIBILITY);
            HttpResponse response = http.execute(post);
            log.debug("ログインステータスコード: " + response.getStatusLine().getStatusCode());

            // ログイン可否の判定.
            HttpEntity entity = response.getEntity();
            entity.consumeContent();
            List<Cookie> cookies = http.getCookieStore().getCookies();
            if (!cookies.isEmpty()) {
                auth = true;
            }
        } catch (IOException ex) {
            log.error("ログイン時に問題が発生", ex);
        }
        return auth;
    }

    /**
     * ニコニコ動画からログアウトする.
     * @return ログアウトに成功すればtrue.
     */
    public boolean logout() throws URISyntaxException, HttpException, InterruptedException {
        boolean result = false;
        HttpGet method = new HttpGet(LOGOUT_PAGE);
        try {
            HttpResponse response = http.execute(method);
            log.debug("ログアウトステータスコード: " + response.getStatusLine().getStatusCode());

            if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                result = true;
            }
            response.getEntity().consumeContent();
        } catch (IOException ex) {
            log.error("ログアウト時に問題が発生", ex);
        }
        return result;
    }

    /**
     * キーワード検索を行う.
     * @param word 検索キーワード
     * @return 検索結果.
     */
    public List<NicoContent> search(String word) {
        log.debug("検索:" + word);

        InputStream is = null;
        List<NicoContent> conts = new ArrayList<NicoContent>();
        String url = new String(SEARCH_HEAD + word + SEARCH_TAIL);

        try {
            while (url != null) {
                HttpGet get = new HttpGet(url);
                HttpResponse response;
                response = http.execute(get);
                is = new BufferedInputStream(response.getEntity().getContent());
                assert is.markSupported();
                is.mark(1024 * 1024);
                List<Result> results = Util.parseSerchResult(is);
                for (Result r : results) {
                    NicoContent c = loadMyMovie(r.getId());
                    if (c != null) {
                        conts.add(c);
                    }
                }
                is.reset();
                url = Util.getNextPage(is);
                is.close();
            }
        } catch (IOException ex) {
            log.error("検索結果処理時に例外発生", ex);
        }
        return conts;
    }

    /**
     * 「マイリスト登録数ランキング(本日)」の動画一覧を取得する。
     * @return 動画一覧.
     */
    public List<NicoContent> loadMyListDaily() throws URISyntaxException, HttpException, InterruptedException {
        List<NicoContent> list = new ArrayList<NicoContent>();
        String url = new String("http://www.nicovideo.jp/ranking/mylist/daily/all?rss=atom");
        log.debug("全動画サイトのマイリスト登録数ランキング(本日)[全体] : " + url);

        HttpGet get = new HttpGet(url);

        BufferedReader reader = null;
        try {
            HttpResponse response = http.execute(get);
            reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            // BOMを読み捨て
            // reader.skip(1);
            list = getNicoContents(reader);
            deleteRankString(list);
            response.getEntity().consumeContent();
        } catch (FeedException ex) {
            log.error("", ex);
        } catch (IOException ex) {
            log.error("", ex);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException ex) {
                    log.error("", ex);
                }
            }
        }
        return list;
    }

    /**
     * ニコニコ動画のRSSからコンテンツリストを取得する.
     * @param url 取得するrssのurl.
     * @return コンテンツリスト.
     */
    public List<NicoContent> getContentsFromRss(String url) {
        log.debug("アクセスURL: " + url);
        List<NicoContent> list = accessRssUrl(url);
        if (url.contains("ranking")) {
            deleteRankString(list);
        }
        return list;
    }

    /**
     * rankingの場合、本当のタイトルの前に"第XX位："の文字列が
     * 挿入されているため, それを削る.
     * @param list 対象のリスト.
     */
    private void deleteRankString(List<NicoContent> list) {
        for (NicoContent c : list) {
            String title = c.getTitle();
            int offset = title.indexOf("：") + 1;
            c.setTitle(title.substring(offset));
        }
    }

    /**
     * マイリストに登録した動画一覧の取得.
     * 「公開」設定にしていないリストからは取得できない.
     * ログインしていなくても取得可能.
     * @param listNo マイリストNo.
     * @return 動画一覧.
     */
    public List<NicoContent> loadMyList(String listNo) {
        String url = new String(MY_LIST_PAGE_HEADER + listNo + "?rss=atom");
        log.debug("マイリストURL: " + url);
        return accessRssUrl(url);
    }

    /**
     * 動画番号を指定したコンテンツ情報の取得.
     * @param movieNo 動画番号.
     * @return　コンテンツ情報.
     */
    public NicoContent loadMyMovie(String movieNo) {
        NicoContent cont = null;
        InputStream re = null;
        List<SyndEntryImpl> list = null;
        String url = new String(MOVIE_THUMBNAIL_PAGE_HEADER + movieNo);
        log.debug("動画サムネイルURL: " + url);

        HttpGet get;

        try {
            get = new HttpGet(url);
            HttpResponse response = http.execute(get);
            re = response.getEntity().getContent();
            // ドキュメントビルダーファクトリを生成
            DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
            // ドキュメントビルダーを生成
            DocumentBuilder builder = dbfactory.newDocumentBuilder();
            // パースを実行してDocumentオブジェクトを取得
            Document doc = builder.parse(re);
            // ルート要素を取得（タグ名：site）
            Element root = doc.getDocumentElement();

            if ("fail".equals(root.getAttribute("status"))) {
                log.warn("情報取得できません: " + movieNo);
                return null;
            }

            NodeList list2 = root.getElementsByTagName("thumb");
            cont = new NicoContent();
            Element element = (Element) list2.item(0);

            String watch_url = ((Element) element.getElementsByTagName("watch_url").item(0)).getFirstChild().
                    getNodeValue();
            cont.setPageLink(watch_url);

            String title = ((Element) element.getElementsByTagName("title").item(0)).getFirstChild().getNodeValue();
            cont.setTitle(title);

            // TODO 投稿日の設定
//            String first_retrieve = ((Element) element.getElementsByTagName("first_retrieve").item(0)).getFirstChild().getNodeValue();
//            cont.setPublishedDate(DateFormat.getInstance().parse(first_retrieve));
//
//        } catch (ParseException ex) {
//            Logger.getLogger(NicoHttpClient.class.getName()).log(Level.SEVERE, null, ex);
        } catch (SAXException ex) {
            log.error("", ex);
        } catch (IOException ex) {
            log.error("", ex);
        } catch (ParserConfigurationException ex) {
            log.error("", ex);
        } finally {
            try {
                if (re != null) {
                    re.close();
                }
            } catch (IOException ex) {
                log.error("", ex);
            }
        }
        return cont;
    }

    private List<NicoContent> accessRssUrl(String url) {
        List<NicoContent> contList = new ArrayList<NicoContent>();
        HttpGet get = new HttpGet(url);
        BufferedReader reader = null;
        try {
            HttpResponse response = http.execute(get);
            reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            if (log.isTraceEnabled()) {
                reader.mark(1024 * 1024);
                while (true) {
                    String str = reader.readLine();
                    if (str == null) {
                        break;
                    }
                    log.trace(str);
                }
                reader.reset();
            }
            contList = getNicoContents(reader);
        } catch (FeedException ex) {
            log.warn("アクセスできません: " + url);
            log.debug("", ex);
        } catch (IOException ex) {
            log.error("", ex);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException ex) {
                    log.error("", ex);
                }
            }
        }
        return contList;
    }

    private List<NicoContent> getNicoContents(Reader reader) throws FeedException {
        List<SyndEntryImpl> list = null;
        SyndFeedInput input = new SyndFeedInput();
        SyndFeed feed = input.build(reader);

        list = (List<SyndEntryImpl>) feed.getEntries();

        List<NicoContent> contList;
        if (list == null) {
            contList = new ArrayList<NicoContent>();
        } else {
            contList = createContentsList(list);
        }
        return contList;
    }

    private List<NicoContent> createContentsList(List<SyndEntryImpl> list) {
        class CallBack extends HTMLEditorKit.ParserCallback {

            private boolean descFlag;
            private String imageLink = new String();
            private StringBuilder description = new StringBuilder();

            @Override
            public void handleSimpleTag(HTML.Tag t, MutableAttributeSet a, int pos) {
                log.debug("--------<" + t.toString() + ">--------");
                log.debug(a);
                if (HTML.Tag.IMG.equals(t)) {
                    imageLink = a.getAttribute(HTML.Attribute.SRC).toString();
                }
            }

            @Override
            public void handleStartTag(HTML.Tag t, MutableAttributeSet a, int pos) {
                if (HTML.Tag.P.equals(t)) {
                    if ("nico-description".equals(
                            a.getAttribute(HTML.Attribute.CLASS).toString())) {
                        descFlag = true;
                    }
                }
                log.debug("--------<" + t.toString() + ">--------");
                log.debug(a);
            }

            @Override
            public void handleEndTag(HTML.Tag t, int pos) {
                if (HTML.Tag.P.equals(t)) {
                    descFlag = false;
                }
                log.debug("--------</" + t.toString() + ">--------");
            }

            @Override
            public void handleText(char[] data, int pos) {
                if (descFlag) {
                    description.append(data);
                }
                log.debug("--------TEXT--------");
                log.debug(data);
            }

            private void printAttributes(MutableAttributeSet a) {
                Enumeration e = a.getAttributeNames();
                while (e.hasMoreElements()) {
                    Object key = e.nextElement();
                    log.debug("---- " + key.toString() + " : " + a.getAttribute(key));
                }
            }

            public String getImageLink() {
                return imageLink;
            }

            public String getDescription() {
                return description.toString();
            }
        }

        List<NicoContent> contList = new ArrayList<NicoContent>();

        for (SyndEntryImpl entry : list) {
            NicoContent content = new NicoContent();

            String title = entry.getTitle();
            content.setTitle(title);
            content.setPageLink(entry.getLink());

            // サムネイル画像リンクと説明文の取得
            CallBack callBack = new CallBack();
            for (SyndContentImpl sc : (List<SyndContentImpl>) entry.getContents()) {
                try {
                    Reader reader = new StringReader(sc.getValue());
                    new ParserDelegator().parse(reader, callBack, true);
                } catch (IOException ex) {
                    log.error("RSSの読み込み失敗: " + content.getTitle());
                }
            }

// リストへ追加.
            contList.add(content);
        }
        return contList;
    }

    /**
     * FLVファイルのURLを取得する. ログインが必要.
     * また, 実際にFLVファイルの実態をダウンロードするには
     * 一度http://www.nicovideo.jp/watch/ビデオIDに一度アクセスする必要があることに
     * 注意.
     * (参考: http://yusukebe.com/tech/archives/20070803/124356.html)
     * @param videoId ニコニコ動画のビデオID.
     * @return FLVファイル実体があるURL.
     * @throws java.io.IOException ファイル取得失敗. 権限の無いファイルを取得しようとした場合も.
     */
    public VideoInfo getVideoInfo(String videoId) throws IOException {
        final String realVideoId = getRealVideoId(videoId);

        String accessUrl = GET_FLV_INFO + realVideoId;
        if (realVideoId.startsWith("nm")) {
            accessUrl += "?as3=1";
        }
        Map<String, String> map = getParameterMap(accessUrl);

        LinkedHashMap<String, String> keyMap = new LinkedHashMap<String, String>();
        if ("1".equals(map.get("needs_key"))) {
            // 公式動画投稿者コメント取得用パラメータ.
            keyMap = getParameterMap(GET_THREAD_KEY_PAGE + map.get(VideoInfo.KEY_THREAD_ID));
        }
        return new VideoInfo(realVideoId, map, keyMap);
    }

    private LinkedHashMap<String, String> getParameterMap(String accessUrl) throws IOException, IllegalStateException {
        log.debug("アクセス: " + accessUrl);
        HttpGet get = new HttpGet(accessUrl);
        String resultString;
        BufferedReader reader = null;
        try {
            HttpResponse response = http.execute(get);
            reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            String str;
            StringBuilder strBuilder = new StringBuilder();
            while ((str = reader.readLine()) != null) {
                strBuilder.append(str);
            }
            resultString = strBuilder.toString();
            response.getEntity().consumeContent();
            log.debug(resultString);
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        String[] params = resultString.split("&");
        LinkedHashMap<String, String> map = new LinkedHashMap<String, String>();
        for (String param : params) {
            String[] elm = param.split("=");
            map.put(elm[0], elm[1]);
        }
        return map;
    }

    /**
     * watchページへアクセスし, ビデオIDを取得する.
     * soXXXXなど, 実際のビデオIDと異なる場合がある.
     * @param videoId 取得したいビデオのビデオID.
     * @return 実際のビデオID.
     * @throws IOException
     */
    private String getRealVideoId(String videoId) throws IOException {
        String realId = accessWatchPage(videoId);

        // ステータスコード302など、リダイレクトが必要な場合
        if (!videoId.equals(realId)) {
            realId = getRealVideoId(realId);
        }
        return realId;
    }

    /**
     * WATCHページへアクセスする. getflvを行うためには, 必ず事前にWATCHページへアクセスしておく必要があるため.
     * @param videoId ビデオID.
     * @return
     * @throws IOException
     */
    private String accessWatchPage(String videoId) throws IOException {
        String realId = videoId;
        String watchUrl = WATCH_PAGE + videoId;
        log.debug("アクセス: " + watchUrl);
        http.getParams().setBooleanParameter(ClientPNames.HANDLE_REDIRECTS, false);
        try {
            HttpGet get = new HttpGet(watchUrl);
            HttpResponse response = http.execute(get);
            if (response.containsHeader("Location")) {
                realId = response.getFirstHeader("Location").getValue().replace("/watch/", "");
            }
            response.getEntity().consumeContent();
        } finally {
            http.getParams().setBooleanParameter(ClientPNames.HANDLE_REDIRECTS, true);
        }
        return realId;
    }

    /**
     * ニコニコ動画から動画ファイルをダウンロードする.
     * @param vi getVideoInfoメソッドで取得したオブジェクト.
     * @param saveDir ダウンロードしたファイルを保存するディレクトリ.
     * @param np 保存するファイル名の命名規則. 拡張子は別途付与されるため不要.
     * @param nowStatus ダウンロードしようとしている動画ファイルの, 現在のステータス.
     * @param needLowFile エコノミー動画をダウンロードするのであればtrue.
     * @return この処理を行った後の, 対象ファイルのステータス.
     * @throws java.io.IOException ファイル取得失敗. 権限の無いファイルを取得しようとした場合も.
     */
    public GetFlvResult getFlvFile(VideoInfo vi, File saveDir, NamePattern np, Status nowStatus, boolean needLowFile,
            ProgressListener listener) throws IOException, URISyntaxException, HttpException, InterruptedException {

        final URL notifierUrl = vi.getSmileUrl();

        String userName = null;
        if (notifierUrl != null) {
            HttpGet get = new HttpGet(notifierUrl.toString());
            HttpResponse response = http.execute(get);
            userName = Util.getUserName(response.getEntity().getContent());
            response.getEntity().consumeContent();
        }

        final URL url = vi.getVideoUrl();
        if (nowStatus == Status.GET_LOW || !needLowFile) {
            if (url.toString().contains("low")) {
                log.info("エコノミー動画のためスキップ: " + vi.getRealVideoId());
                return new GetFlvResult(null, nowStatus, userName);
            }
        }
        final boolean isNotLow = !url.toString().contains("low");

        final File downloadFile = new File(saveDir, np.createFileName(vi.getRealVideoId(), isNotLow));

        HttpGet get = new HttpGet(url.toURI());
        HttpResponse response = http.execute(get);
        String contentType = response.getEntity().getContentType().getValue();
        log.debug(contentType);
        log.debug(downloadFile.toString());
        if ("text/plain".equals(contentType) || "text/html".equals(contentType)) {
            log.error("取得できませんでした. サーバが混みあっている可能性があります: " + vi.getRealVideoId());
            response.getEntity().consumeContent();
            return new GetFlvResult(null, Status.GET_INFO, userName);
        }
        String ext = Util.getExtention(contentType);
        final long fileSize = response.getEntity().getContentLength();

        final int BUF_SIZE = 1024 * 32;
        BufferedInputStream in = new BufferedInputStream(response.getEntity().getContent());

        File file = new File(downloadFile.toString() + "." + ext);
        log.info("保存します(" + fileSize / 1024 + "KB): " + file.getPath());
        FileOutputStream fos = new FileOutputStream(file);
        BufferedOutputStream out = new BufferedOutputStream(fos);

        long downloadSize = 0;
        int i;
        byte[] buffer = new byte[BUF_SIZE];
        while ((i = in.read(buffer)) != -1) {
            out.write(buffer, 0, i);
            downloadSize += i;
            listener.progress(fileSize, downloadSize);
        }

        response.getEntity().consumeContent();
        out.close();
        in.close();
        if (url.toString().contains("low")) {
            return new GetFlvResult(file, Status.GET_LOW, userName);
        }
        return new GetFlvResult(file, Status.GET_FILE, userName);
    }

    /**
     * ニコニコ動画から動画ファイルをダウンロードする.
     * @param vi getVideoInfoメソッドで取得したオブジェクト.
     * @param fileName ダウンロード後のファイル名. 拡張子は別途付与されるため不要.
     * @param nowStatus ダウンロードしようとしている動画ファイルの, 現在のステータス.
     * @param needLowFile エコノミー動画をダウンロードするのであればtrue.
     * @return この処理を行った後の, 対象ファイルのステータス.
     * @throws java.io.IOException ファイル取得失敗. 権限の無いファイルを取得しようとした場合も.
     */
    public GetFlvResult getFlvFile(VideoInfo vi, String fileName, Status nowStatus, boolean needLowFile,
            ProgressListener listener) throws IOException, URISyntaxException, HttpException, InterruptedException {
        String file = FilenameUtils.getName(fileName);
        String dir = fileName.substring(0, fileName.length() - file.length());
        NamePattern np = new NamePattern(file, "", "", "");
        return getFlvFile(vi, new File(dir), np, nowStatus, needLowFile, listener);
    }

    /**
     * ニコニコ動画から動画ファイルをダウンロードする.
     * @param vi getVideoInfoメソッドで取得したオブジェクト.
     * @param fileName ダウンロード後のファイル名. 拡張子は別途付与されるため不要.
     * @return この処理を行った後の, 対象ファイルのステータス.
     * @throws java.io.IOException ファイル取得失敗. 権限の無いファイルを取得しようとした場合も.
     */
    public GetFlvResult getFlvFile(VideoInfo vi, String fileName, ProgressListener listener) throws IOException,
            URISyntaxException,
            HttpException, InterruptedException {
        return getFlvFile(vi, fileName, Status.GET_INFO, true, listener);
    }

    /**
     * ニコニコ動画から動画ファイルをダウンロードする.
     * ファイル名はビデオID名となる.
     * @param vi getVideoInfoメソッドで取得したオブジェクト.
     * @return この処理を行った後の, 対象ファイルのステータス.
     * @throws java.io.IOException ファイル取得失敗. 権限の無いファイルを取得しようとした場合も.
     */
    public GetFlvResult getFlvFile(VideoInfo vi) throws IOException, URISyntaxException, HttpException,
            InterruptedException {
        return getFlvFile(vi, vi.getRealVideoId(), Status.GET_INFO, true, ProgressListener.EMPTY_LISTENER);
    }

    public File getCommentFile(VideoInfo vi, String fileName) throws Exception {
        return downloadComment(vi, fileName, false);
    }

    public File getTCommentFile(VideoInfo vi, String fileName) throws Exception {
        return downloadComment(vi, fileName, true);
    }

    private File downloadComment(VideoInfo vi, String fileName, boolean isTcomm) throws Exception {
        HttpResponse response = null;
        BufferedOutputStream bos = null;

        try {
            final HttpPost post = new HttpPost(vi.getMessageUrl().toString());
            final String param = createCommendDownloadParameter(vi, isTcomm);
            final StringEntity se = new StringEntity(param);
            post.setEntity(se);
            response = http.execute(post);
            final InputStream is = response.getEntity().getContent();
            final BufferedInputStream bis = new BufferedInputStream(is);

            final String outputFileName = (fileName.endsWith(".xml")) ? fileName : fileName + ".xml";
            bos = new BufferedOutputStream(new FileOutputStream(outputFileName));

            final byte[] buf = new byte[1024 * 1024];
            int read;
            while ((read = bis.read(buf, 0, buf.length)) > 0) {
                bos.write(buf, 0, read);
            }

            return new File(outputFileName);
        } catch (Exception e) {
            throw new Exception("コメントダウンロードに失敗しました。", e);
        } finally {
            if (response != null) {
                response.getEntity().consumeContent();
            }
            if (bos != null) {
                bos.close();
            }
        }
    }

    private String createCommendDownloadParameter(VideoInfo vi, boolean isTcomm) {
        final String tcommStr = (isTcomm) ? "fork=\"1\" " : "";
        StringBuilder builder = new StringBuilder();
        Set<String> keySet = vi.getKeyMap().keySet();
        for (String key : keySet) {
            builder.append(key + "=\"" + vi.getKeyMap().get(key) + "\" ");
        }
        final String officialOption = builder.toString();

        return "<thread " + VideoInfo.KEY_USER_ID + "=\"" + vi.getUserId() + "\" res_from=\"" + vi.getResFrom()
                + "\" version=\"20061206\" thread=\"" + vi.getThreadId() + "\" " + tcommStr + officialOption + "/>";
    }

    /**
     * 動画をマイリストへ登録する. ログインが必要.
     * @param myListId 登録するマイリストのID.
     * @param videoId 登録する動画ID.
     * @throws IOException 登録に失敗した.
     */
    public void addMyList(String myListId, String videoId) throws IOException {
        String itemType = null;
        String itemId = null;
        String token = null;
        HttpGet get = new HttpGet(ADD_MYLIST_PAGE + videoId);
        HttpResponse response = http.execute(get);
        HttpEntity entity = response.getEntity();
        try {
            InputStream is = entity.getContent();
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            String line;

            Pattern pattern = Pattern.compile("input type=\"hidden\" name=\"item_type\" value=\"(.+)\"");
            while ((line = reader.readLine()) != null) {
                Matcher m = pattern.matcher(line);
                if (m.find()) {
                    itemType = m.group(1);
                    break;
                }
            }

            pattern = Pattern.compile("input type=\"hidden\" name=\"item_id\" value=\"(.+)\"");
            while ((line = reader.readLine()) != null) {
                Matcher m = pattern.matcher(line);
                if (m.find()) {
                    itemId = m.group(1);
                    break;
                }
            }

            pattern = Pattern.compile("NicoAPI\\.token = \"(.*)\";");
            while ((line = reader.readLine()) != null) {
                Matcher m = pattern.matcher(line);
                if (m.find()) {
                    token = m.group(1);
                    break;
                }
            }
        } finally {
            entity.consumeContent();
        }

        if (itemType == null || itemId == null || token == null) {
            throw new IOException("マイリスト登録に必要な情報が取得できませんでした。 "
                    + "マイリスト:" + myListId + ", 動画ID:" + videoId + ", item_type:" + itemType + ", item_id:" + itemId
                    + ", token:" + token);
        }

        StringEntity se = new StringEntity(
                "group_id=" + myListId
                + "&item_type=" + itemType
                + "&item_id=" + itemId
                + "&description=" + ""
                + "&token=" + token);

        HttpPost post = new HttpPost("http://www.nicovideo.jp/api/mylist/add");
        post.setHeader("Content-Type", "application/x-www-form-urlencoded");
        post.setEntity(se);
        response = http.execute(post);
        int statusCode = response.getStatusLine().getStatusCode();
        response.getEntity().consumeContent();
        if (statusCode != 200) {
            throw new IOException("マイリスト登録に失敗" + "マイリスト:" + myListId + ", 動画ID:" + videoId);
        }
    }
}
