<?php
/**
 * ストリームへデータを入出力するためのユーティリティです。
 * 
 * これらの関数は、ノンブロッキングI/Oに対しても与えられた(要求される)データを全て出力(入力)します。
 * 
 * 通常、プログラマがこのパッケージを直接使う必要はありません。
 * 
 * @package cssj
 * @subpackage helpers
 */
/**
 * パケットの送信に使うバッファのサイズです。
 * 
 * @access private
 */
define ('CSSJ_BUFFER_SIZE', 1024);

/**
 * 32ビット数値をビッグインディアンで書き出します。
 * 
 * @param $fp resource ストリーム
 * @param $a int 数値
 * @return mixed 書き込んだバイト数、エラーであればfalse
 * @access public
 */
function cssj_utils_write_int(&$fp, $a) {
  $data = pack('N', $a);
  return _cssj_write($fp, $data);
}

/**
 * 8ビット数値を書き出します。
 * 
 * @param $fp resource ストリーム
 * @param $a byte 数値
 * @return mixed 書き込んだバイト数、エラーであればfalse
 * @access public
 */
function cssj_utils_write_byte(&$fp, $b) {
  $data = chr($b);
  return _cssj_write($fp, $data);
}

/**
 * バイト数を16ビットビッグインディアンで書き出した後、バイト列を書き出します。
 * 
 * @param $fp resource ストリーム
 * @param $b string バイト列
 * @return mixed 書き込んだバイト数、エラーであればfalse
 * @access public
 */
function cssj_utils_write_bytes(&$fp, &$b) {
  $data = pack('n', strlen($b));
  $len = _cssj_write($fp, $data);
  if ($len === false) {
    return $len;
  }
  return _cssj_write($fp, $b);
}

/**
 * バイト列を書き出します。
 * 
 * @param $fp resource ストリーム
 * @param $data string バイト列
 * @return mixed 書き込んだバイト数、エラーであればfalse
 * @access private
 */
function _cssj_write(&$fp, &$data) {
  for (;;) {
    $len = fwrite($fp, $data);
    if ($len === false || $len >= strlen($data)) {
      return $len;
    }
    $data = substr($data, $len, strlen($data) - $len);
  }
}

/**
 * 32ビットビッグインディアン数値を読み込みます。
 * 
 * @param $fp resource ストリーム
 * @return mixed 数値、エラーであればfalse
 * @access public
 */
function cssj_utils_read_int(&$fp) {
  $b = _cssj_read($fp, 4);
  if ($b === false) {
    return $b;
  }
  $a = unpack('Nint', $b);
  return $a['int'];
}

/**
 * 8ビット数値を読み込みます。
 * 
 * @param $fp resource ストリーム
 * @return mixed 数値、エラーであればfalse
 * @access public
 */
function cssj_utils_read_byte(&$fp) {
  $b = _cssj_read($fp, 1);
  if ($b === false) {
    return $b;
  }
  return ord($b);
}

/**
 * 16ビットビッグインディアン数値を読み込み、そのバイト数だけバイト列を読み込みます。
 * 
 * @param $fp resource ストリーム
 * @return mixed バイト列、エラーであればfalse
 * @access public
 */
function &cssj_utils_read_bytes(&$fp) {
  $b = _cssj_read($fp, 2);
  if ($b === false) {
    return $b;
  }
  $a = unpack('nshort', $b);
  $len = $a['short'];
  $b = _cssj_read($fp, $len);
  if ($b === false) {
    return $b;
  }
  return $b;
}

/**
 * バイト列を読み込みます。
 * 
 * @param $fp resource ストリーム
 * @param $len int 要求されるバイト数
 * @return mixed バイト列、エラーであればfalse
 * @access private
 */
function &_cssj_read(&$fp, $len) {
  $result = '';
  for (;;) {
    if ($len <= 0) {
      return $result;
    }
    $data = fread($fp, $len);
    if ($data === false) {
      return $data;
    }
    $len -= strlen($data);
    $result .= $data;
  }
}

?>