#import "common.h"
#import "CMToken.h"

#define CMPriorityException @"CMPriorityException"


#define ARRAY_LITERAL_LEFT_PRIORITY      30
#define DICTIONARY_LITERAL_LEFT_PRIORITY 30
#define OPEN_LEFT_PRIORITY               34
#define CLOSE_LEFT_PRIORITY              18
#define SINGLE_MESSAGE_LEFT_PRIORITY     20
#define BINARY_MESSAGE_LEFT_PRIORITY     22
#define KEYWORD_MESSAGE_LEFT_PRIORITY    24
#define SUBSTITUTE_LEFT_PRIORITY         26
#define RETURN_LEFT_PRIORITY             28
#define TERMINATE_LEFT_PRIORITY          31

#define ARRAY_LITERAL_RIGHT_PRIORITY      31
#define DICTIONARY_LITERAL_RIGHT_PRIORITY 16
#define OPEN_RIGHT_PRIORITY               16
#define CLOSE_RIGHT_PRIORITY              32
#define SINGLE_MESSAGE_RIGHT_PRIORITY     20
#define BINARY_MESSAGE_RIGHT_PRIORITY     22
#define KEYWORD_MESSAGE_RIGHT_PRIORITY    24
#define SUBSTITUTE_RIGHT_PRIORITY         26
#define RETURN_RIGHT_PRIORITY             28
#define TERMINATE_RIGHT_PRIORITY          0

// left 0 , right 1
#define ARRAY_LITERAL_EQUAL_PRIORITY      0
#define OPEN_EQUAL_PRIORITY               1
#define CLOSE_EQUAL_PRIORITY              0
#define SINGLE_MESSAGE_EQUAL_PRIORITY     0
#define BINARY_MESSAGE_EQUAL_PRIORITY     0
#define KEYWORD_MESSAGE_EQUAL_PRIORITY    1
#define TERMINATE_EQUAL_PRIORITY          1


@implementation CMToken

+ (id)tokenWithString:(NSString *)aString
{
  return [[[self alloc] initWithString:aString] autorelease];
}

+ (NSString *)classReservedWord { return ClassReservedWord; }
+ (NSString *)endReservedWord { return EndReservedWord; }

+ (NSCharacterSet *)identifierCharacterSet
{
  NSMutableCharacterSet *set = [[[NSCharacterSet letterCharacterSet] mutableCopy]
                                 autorelease];
  [set addCharactersInString:@"_"];
  [set formUnionWithCharacterSet:
             [NSCharacterSet decimalDigitCharacterSet]];
  return set;
}

+ (NSCharacterSet *)operatorMessageCharacterSet
{
  return [NSCharacterSet characterSetWithCharactersInString:@"+-*/%^<>=|&"];
}

+ (NSCharacterSet *)terminateCharacterSet
{
  id set = [[[self newLineCharacterSet] mutableCopy] autorelease];
  [set addCharactersInString:@";"];
  return set;
}

+ (NSCharacterSet *)newLineCharacterSet
{
  unichar chars[] = {0x000A, 0x000B, 0x000C, 0x000D, 0x0085};
  NSString *str = [NSString stringWithCharacters:chars
                            length:(sizeof(chars) / sizeof(unichar))];
  return [NSCharacterSet characterSetWithCharactersInString:str];
}

+ (NSCharacterSet *)escapeCharacterSet
{
  return [NSCharacterSet characterSetWithCharactersInString:@"\\\xc2\xa5"];
}

+ (NSCharacterSet *)regularExpressionOptionCharacterSet
{
  return [NSCharacterSet characterSetWithCharactersInString:
                           @"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"];
}

+ (NSArray *)allTokenCharacterSets
{
  return [NSArray arrayWithObjects:
                    [NSCharacterSet whitespaceCharacterSet],
                  [self terminateCharacterSet],
                  [self operatorMessageCharacterSet],
                  [self identifierCharacterSet], nil];
}

+ (NSArray *)allTokenStrings
{
  return [NSArray arrayWithObjects:@"(", @")", @".", @"[", @"]", @"{", @"}",
                  @",", @":", @"|", @"->", @"@", @"#", nil];
}

- (id)initWithString:(NSString *)aString
{
  [self init];
  tokenString = [[NSMutableString alloc] initWithString:aString];
  return self;
}

- (id)init
{
  [super init];
  isDigitLiteral = NO;
  lineNumber = 0;
  return self;
}

- (void)dealloc
{
  [tokenString release];
  [prefixContent release];
  [regexpOptions release];
  [super dealloc];
}

- (int)lineNumber
{
  return lineNumber;
}

- (void)setLineNumber:(int)number
{
  lineNumber = number;
}

- (NSArray *)regularExpressionOptions
{
  return regexpOptions;
}

- (void)setRegularExpressionOptions:(NSArray *)options
{
  ASSIGN(regexpOptions, options);
}

- (NSString *)selectorString
{
  NSRange range = NSMakeRange(1, [tokenString length]-2);
  return [tokenString substringWithRange:range];
}

- (void)setPrefixContent:(NSString *)aString
{
  ASSIGN(prefixContent, aString);
}

- (void)appendTokenString:(NSString *)aString
{
  [tokenString appendString:aString];
}

- (NSString *)prefixContent { return prefixContent; }
- (NSString *)tokenString { return tokenString; }
- (NSString *)description { return tokenString; }

- (NSString *)sourceCode
{
  if (prefixContent)
    return [prefixContent stringByAppendingString:tokenString];
  else
    return tokenString;
}

- (NSString *)unquotedString
{
  NSRange range = NSMakeRange(1, [tokenString length]-2);
  return [tokenString substringWithRange:range];
}

- (BOOL)isClassReservedWord {
  return [tokenString isEqualToString:[CMToken classReservedWord]];
}

- (BOOL)isEndReservedWord
{
  return [tokenString isEqualToString:[CMToken endReservedWord]];
}

- (BOOL)isAndReservedWord
{
  return [tokenString isEqualToString:AndReservedWord];
}

- (BOOL)isOrReservedWord
{
  return [tokenString isEqualToString:OrReservedWord];
}

- (BOOL)isNotReservedWord
{
  return [tokenString isEqualToString:NotReservedWord];
}

- (BOOL)isReturnReservedWord
{
  return [tokenString isEqualToString:ReturnReservedWord];
}

- (BOOL)isImportReservedWord
{
  return [tokenString isEqualToString:ImportReservedWord];
}

- (BOOL)isStringLiteral
{
  return ([[tokenString substringToIndex:1] isEqualToString:@"\""] ||
          [[tokenString substringToIndex:1] isEqualToString:@"'"]);
}

- (BOOL)isDigitLiteral
{
  return isDigitLiteral;
}

- (void)setIsDigitLiteral:(BOOL)flag
{
  isDigitLiteral = flag;
}

- (BOOL)isAssociationLiteral
{
  return [tokenString isEqualToString:@"->"];
}

- (BOOL)isRegularExpressionLiteral
{
  return [tokenString length] > 1 &&
    [tokenString hasPrefix:@"/"] && [tokenString hasSuffix:@"/"];
}

- (BOOL)isSelectorLiteral
{
  return [tokenString length] > 1 &&
    [tokenString hasPrefix:@"<"] && [tokenString hasSuffix:@">"];
}

- (BOOL)isOpenParenthesis
{
  return [tokenString isEqualToString:@"("];
}

- (BOOL)isCloseParenthesis
{
  return [tokenString isEqualToString:@")"];
}

- (BOOL)isOpenBracket
{
  return [tokenString isEqualToString:@"["];
}

- (BOOL)isCloseBracket
{
  return [tokenString isEqualToString:@"]"];
}

- (BOOL)isOpenBrace
{
  return [tokenString isEqualToString:@"{"];
}

- (BOOL)isCloseBrace
{
  return [tokenString isEqualToString:@"}"];
}

- (BOOL)isOpen
{
  return [self isOpenParenthesis] || [self isOpenBrace] || [self isOpenBracket];
}

- (BOOL)isClose
{
  return [self isCloseParenthesis] || [self isCloseBrace] || [self isCloseBracket];
}

- (BOOL)isPair:(CMToken *)aToken
{
  return (([self isOpenParenthesis] && [aToken isCloseParenthesis]) ||
          ([self isOpenBracket] && ([aToken isCloseBracket] || [aToken isTerminate])) ||
          ([self isOpenBrace] && [aToken isCloseBrace]));
}

- (BOOL)isIdentifier
{
  return ([self isAllCharactersOfSet:[[self class] identifierCharacterSet]] &&
          ![[NSCharacterSet decimalDigitCharacterSet]
             characterIsMember:
               [tokenString characterAtIndex:0]]);
}

- (BOOL)isSubstituteOperatorString
{
  return [tokenString isEqualToString:@"="];
}

- (BOOL)isAtStartOrEndOfString
{
  return [tokenString isEqualToString:@"\""];
}

- (BOOL)isWhitespace
{
  return [self isAllCharactersOfSet:
                 [NSCharacterSet whitespaceCharacterSet]];
}

- (BOOL)isNewLine
{
  return [self isAllCharactersOfSet:
                 [NSCharacterSet whitespaceAndNewlineCharacterSet]];
}

- (BOOL)isSemicolon
{
  return [tokenString isEqualToString:@";"];
}

- (BOOL)isColon
{
  return [tokenString isEqualToString:@":"];
}

- (BOOL)isComma
{
  return [tokenString isEqualToString:@","];
}

- (BOOL)isDot
{
  return [tokenString isEqualToString:@"."];
}

- (BOOL)isVerticalBar
{
  return [tokenString isEqualToString:@"|"];
}

- (BOOL)isComment
{
  if ([tokenString length] > 1) {
    NSRange range = NSMakeRange(0, 2);
    return [[tokenString substringWithRange:range] isEqualToString:@"//"];
  } else {
    return NO;
  }
}

- (BOOL)isAtStartOfMethod
{
  return ([tokenString isEqualToString:@"-"] || [tokenString isEqualToString:@"+"]);
}

- (BOOL)isTerminate
{
  return ([self isNewLine] ||
          [self isAllCharactersOfSet:
                  [NSCharacterSet characterSetWithCharactersInString:@";"]]);
}

- (BOOL)isAllCharactersOfSet:(NSCharacterSet *)aSet
{
  unsigned int i;
  for (i = 0; i < [tokenString length]; i++) {
    if (![aSet characterIsMember:[tokenString characterAtIndex:i]])
      return NO;
  }
  return YES;
}

- (BOOL)isMessage
{
  return [self isAllCharactersOfSet:[NSCharacterSet letterCharacterSet]];
}

- (BOOL)isOperatorMessage
{
  return [self isAllCharactersOfSet:[[self class] operatorMessageCharacterSet]] &&
    ![self isAssociationLiteral] && ![self isSubstituteOperatorString];
}

- (BOOL)isPlusOperatorMessage
{
  return [tokenString isEqualToString:@"+"];
}

- (BOOL)isMinusOperatorMessage
{
  return [tokenString isEqualToString:@"-"];
}

- (BOOL)isMutablePrefix
{
  return [tokenString isEqualToString:MutablePrefix];
}

- (BOOL)isImmutablePrefix
{
  return [tokenString isEqualToString:ImmutablePrefix];
}

- (BOOL)isInstanceOperator
{
  return [tokenString isEqualToString:@"-"];
}

- (BOOL)isClassOperator
{
  return [tokenString isEqualToString:@"+"];
}

- (BOOL)isEscape
{
  return [self isAllCharactersOfSet:[CMToken escapeCharacterSet]];
}

- (BOOL)isSingleMessage
{
  return [self isIdentifier] &&
    [tokenString rangeOfString:@":"].length == 0 &&
    ![self isReturnReservedWord];
}

- (BOOL)isBinaryMessage
{
  return [self isOperatorMessage];
}

- (BOOL)isKeywordMessage
{
  return [tokenString rangeOfString:@":"].length != 0;
}

- (unsigned int)numberOfArguments
{
  if ([self isBinaryMessage])
    return 1;
  else
    return [[tokenString componentsSeparatedByString:@":"] count] - 1;
}

- (BOOL)isEqualToToken:(CMToken *)aToken
{
  return [tokenString isEqualToString:[aToken tokenString]] ||
    ([self isSingleMessage] && [aToken isSingleMessage]) ||
    ([self isBinaryMessage] && [aToken isBinaryMessage]) ||
    ([self isKeywordMessage] && [aToken isKeywordMessage]);
}

- (NSComparisonResult)compare:(CMToken *)aToken
{
  if ([self isEqualToToken:aToken]) {
    LOG(@"* compare '%@' and '%@' -> %d",
        [self description], [aToken description], [self equalPriority]);
    if ([self equalPriority] == 0)
      return NSOrderedAscending;
    else
      return NSOrderedDescending;
  } else {
    int left, right;
    left = [self leftPriority];
    right = [aToken rightPriority];
    LOG(@"* compare '%@':%d and '%@':%d",
        [self description], left, [aToken description], right);
    if (left < right)
      return NSOrderedAscending;
    else
      return NSOrderedDescending;
  }
}

- (int)leftPriority
{
  if ([self isSingleMessage])
    return SINGLE_MESSAGE_LEFT_PRIORITY;
  else if ([self isOperatorMessage])
    return BINARY_MESSAGE_LEFT_PRIORITY;
  else if ([self isKeywordMessage])
    return KEYWORD_MESSAGE_LEFT_PRIORITY;
  else if ([self isComma])
    return ARRAY_LITERAL_LEFT_PRIORITY;
  else if ([self isAssociationLiteral])
    return DICTIONARY_LITERAL_LEFT_PRIORITY;
  else if ([self isOpenParenthesis] || [self isOpenBracket] ||
           [self isOpenBrace])
    return OPEN_LEFT_PRIORITY;
  else if ([self isCloseParenthesis] || [self isCloseBracket] ||
           [self isCloseBrace])
    return CLOSE_LEFT_PRIORITY;
  else if ([self isSubstituteOperatorString])
    return SUBSTITUTE_LEFT_PRIORITY;
  else if ([self isReturnReservedWord])
    return RETURN_LEFT_PRIORITY;
  else if ([self isTerminate])
    return TERMINATE_LEFT_PRIORITY;
  else
    [NSException raise:CMPriorityException
                 format:@"priority error - %@", tokenString];
  return 0;
}

- (int)rightPriority
{
  if ([self isSingleMessage])
    return SINGLE_MESSAGE_RIGHT_PRIORITY;
  else if ([self isOperatorMessage])
    return BINARY_MESSAGE_RIGHT_PRIORITY;
  else if ([self isKeywordMessage])
    return KEYWORD_MESSAGE_RIGHT_PRIORITY;
  else if ([self isComma])
    return ARRAY_LITERAL_RIGHT_PRIORITY;
  else if ([self isAssociationLiteral])
    return DICTIONARY_LITERAL_RIGHT_PRIORITY;
  else if ([self isOpenParenthesis] || [self isOpenBracket] ||
           [self isOpenBrace])
    return OPEN_RIGHT_PRIORITY;
  else if ([self isCloseParenthesis] || [self isCloseBracket] ||
           [self isCloseBrace])
    return CLOSE_RIGHT_PRIORITY;
  else if ([self isSubstituteOperatorString])
    return SUBSTITUTE_RIGHT_PRIORITY;
  else if ([self isReturnReservedWord])
    return RETURN_RIGHT_PRIORITY;
  else if ([self isTerminate])
    return TERMINATE_RIGHT_PRIORITY;
  else
    [NSException raise:CMPriorityException
                 format:@"priority error - %@", tokenString];
  return 0;
}

- (int)equalPriority
{
  if ([self isSingleMessage])
    return SINGLE_MESSAGE_EQUAL_PRIORITY;
  else if ([self isOperatorMessage])
    return BINARY_MESSAGE_EQUAL_PRIORITY;
  else if ([self isKeywordMessage])
    return KEYWORD_MESSAGE_EQUAL_PRIORITY;
  else if ([self isComma])
    return ARRAY_LITERAL_EQUAL_PRIORITY;
  else if ([self isOpenParenthesis] || [self isOpenBracket] ||
           [self isOpenBrace])
    return OPEN_EQUAL_PRIORITY;
  else if ([self isCloseParenthesis] || [self isCloseBracket] ||
           [self isCloseBrace])
    return CLOSE_EQUAL_PRIORITY;
  else if ([self isTerminate])
    return TERMINATE_EQUAL_PRIORITY;
  else
    [NSException raise:CMPriorityException
                 format:@"priority error - %@", tokenString];
  return 0;
}

@end
