#import "common.h"
#import "CMSyntaxRule.h"
#import "CMNode.h"
#import "CMToken.h"
#import "CMMessageNode.h"
#import "CMArrayNode.h"
#import "CMDictionaryNode.h"
#import "CMSubstituteNode.h"
#import "CMReturnNode.h"
#import "CMBlockNode.h"

@implementation CMSyntaxRule

+ (id)ruleWithOperandStack:(NSMutableArray *)opStack
               outputStack:(NSMutableArray *)outStack
{
  return [[[self alloc] initWithOperandStack:opStack outputStack:outStack]
           autorelease];
}

- (id)initWithOperandStack:(NSMutableArray *)opStack
               outputStack:(NSMutableArray *)outStack
{
  [self init];
  ops = [[NSMutableArray alloc] initWithCapacity:1];
  outs = [[NSMutableArray alloc] initWithCapacity:1];
  [self compositeWithOperandStack:opStack outputStack:outStack];
  return self;
}

- (id)init
{
  [super init];
  return self;
}

- (void)dealloc
{
  [ops release];
  [outs release];
  [super dealloc];
}

- (NSArray *)operandStack { return ops; }
- (NSArray *)outputStack { return outs; }

- (void)setLeft:(CMNode *)aNode
{
  [outs insertObject:aNode atIndex:0];
}

- (int)priority
{
  return [[ops lastObject] priority];
}

- (BOOL)requiresNodes
{
  id op = [ops lastObject];
  return !([op isOpen] || [op isComma] || [op isTerminate]);
}

- (void)compositeWithOperandStack:(NSMutableArray *)opStack
                      outputStack:(NSMutableArray *)outStack
{
  id op;
  //LOG(@"==> -[CMSyntaxRule compositeWithOperandStack:outputStack:]");
  LOG(@"* arguments op: %@ out: %@", [opStack description], [outStack description]);

  // single, binary, keyword, array, dict, substitute, return
  op = [opStack lastObject];
  if ([op isSingleMessage] || [op isBinaryMessage] || [op isKeywordMessage]) {
    int i;

    // message
    [ops addObject:op];
    [opStack removeLastObject];

    LOG(@"* %@ -> pop %d arguments", [op tokenString], [op numberOfArguments]);
    // arguments
    for (i = 0; i < [op numberOfArguments]; i++) {
      [outs insertObject:[outStack lastObject] atIndex:0];
      [outStack removeLastObject];
    }

  } else if ([op isComma] || [op isAssociationLiteral]) {
    [ops insertObject:[opStack lastObject] atIndex:0];
    [opStack removeLastObject];
    if ([outStack lastObject]) {
      [outs insertObject:[outStack lastObject] atIndex:0];
      [outStack removeLastObject];
    } else {
      [NSException raise:@"CMSyntaxException" format:@"odd item"];
    }

  } else if ([op isBinaryMessage] || [op isReturnReservedWord]) {
    [ops addObject:op];
    [opStack removeLastObject];
    [outs addObject:[outStack lastObject]];
    [outStack removeLastObject];

  } else if ([op isSubstituteOperatorString]) {
    [ops addObject:op];
    [opStack removeLastObject];
    [outs insertObject:[outStack lastObject] atIndex:0];
    [outStack removeLastObject];
    [outs insertObject:[outStack lastObject] atIndex:0];
    [outStack removeLastObject];

  // start expr or array, dictionary
  } else if ([op isOpenParenthesis] || [op isOpenBrace] || [op isOpenBracket]) {
    [ops addObject:op];
    [opStack removeLastObject];

  } else if ([op isCloseParenthesis] || [op isCloseBracket] || [op isCloseBrace]) {
    [ops addObject:op];
    [opStack removeLastObject];

  } else if ([op isTerminate]) {
    [ops addObject:op];
    [opStack removeLastObject];

  }

  //LOG(@"* result op: %@ out: %@", [ops description], [outs description]);
  //LOG(@"<== -[CMSyntaxRule compositeWithOperandStack:outputStack:]");
}

- (void)returnTokensToOperandStack:(NSMutableArray *)opStack
                       outputStack:(NSMutableArray *)outStack
{
  [opStack addObjectsFromArray:ops];
  [outStack addObjectsFromArray:outs];
}

- (void)takeNodesFromOperandStack:(NSMutableArray *)opStack
                      outputStack:(NSMutableArray *)outStack
{
  id op, out;
  op = [ops lastObject];
  out = [outStack lastObject];

  LOG(@"* takeNodesFromOperandStack:outputStack: - preOp:%@, op:%@, out:%@",
      [op description], [opStack description], [outStack description]);
  if ([op isCloseParenthesis] || [op isCloseBracket] || [op isCloseBrace]) {
    while ([outStack count] > 0) {
      out = [outStack lastObject];
      [outs insertObject:out atIndex:0];
      [outStack removeLastObject];
      if (([op isCloseParenthesis] && [out isEqual:[CMExpressionNode class]]) ||
          ([op isCloseParenthesis] && [out isEqual:[NSArray class]]) ||
          ([op isCloseParenthesis] && [out isEqual:[NSMutableArray class]]) ||
          ([op isCloseBracket] && [out isEqual:[CMBlockNode class]]) ||
          ([op isCloseBrace] && [out isEqual:[NSDictionary class]]) ||
          ([op isCloseBrace] && [out isEqual:[NSMutableDictionary class]]))
        break;
    }
    // lastObject$B$,(Bopen$B$8$c$J$+$C$?$i%(%i!<(B
    [opStack removeLastObject];

  // $B%-!<%o!<%I%a%C%;!<%8$N%l%7!<%P$r7hDj$9$k(B
  } else if ([op isKeywordMessage] &&
             [op numberOfArguments] == [outs count]) {
    [outs insertObject:out atIndex:0];
    [outStack removeLastObject];
    LOG(@"* defined keyword message - %@", [outs description]);

  } else if (![op isSubstituteOperatorString] && ![op isReturnReservedWord]) {
    [outs insertObject:out atIndex:0];
    [outStack removeLastObject];
  }
}

- (void)reduceToStack:(NSMutableArray *)aStack
{
  //LOG(@"-[CMSyntaxRule reduceToStack] op: %@ out: %@ stack: %@",
  //      [ops description], [outs description], [aStack description]);
  id op = [ops lastObject];
  if ([op isSingleMessage] || [op isBinaryMessage] || [op isKeywordMessage])
    [self reduceMessageNodeToStack:aStack];
  else if ([op isComma] || [op isAssociationLiteral])
    [self reduceElementNodeToStack:aStack];
  else if ([op isSubstituteOperatorString])
    [self reduceSubstituteNodeToStack:aStack];
  else if ([op isReturnReservedWord])
    [self reduceReturnNodeToStack:aStack];
  else if ([op isCloseParenthesis])
    [self reduceArrayOrExpressionNodeToStack:aStack];
  else if ([op isCloseBrace])
    [self reduceDictionaryNodeToStack:aStack];
  else if ([op isTerminate])
    [self reduceTerminateToStack:aStack];
  //LOG(@"2 -[CMSyntaxRule reduceToStack] op: %@ out: %@ stack: %@",
  //      [ops description], [outs description], [aStack description]);
}

// $B$3$3$G9=J8%A%'%C%/$b9T$&$Y$-(B

- (void)reduceMessageNodeToStack:(NSMutableArray *)aStack
{
  CMMessageNode *node;

  LOG(@"* reduceMessageNodeToStack - op:%@ out:%@",
      [ops description], [outs description]);
  node = [CMMessageNode nodeWithMessage:[[ops lastObject] tokenString]];
  [node addNodesInArray:outs];
  [aStack addObject:node];
  LOG(@"* after reduceMessageNodeToStack - %@", [aStack description]);
}

- (void)reduceElementNodeToStack:(NSMutableArray *)aStack
{
  [aStack addObjectsFromArray:outs];
}

- (void)reduceArrayOrExpressionNodeToStack:(NSMutableArray *)aStack
{
  CMArrayNode *node = nil;
  id out;
  NSMutableArray *nodes = [NSMutableArray arrayWithCapacity:1];

  while ([outs count] > 0) {
    out = [outs lastObject];
    if ([out isEqual:CMMutableArrayNodeMark] ||
        [out isEqual:CMImmutableArrayNodeMark]) {
      [outs removeLastObject];
      node = [CMArrayNode node];
      [node setIsMutable:[out isEqual:CMMutableArrayNodeMark]];
      break;
    } else if ([out isEqual:CMExpressionNodeMark]) {
      [outs removeLastObject];
      if ([nodes count] > 1)
        node = [CMArrayNode node];
      else
        node = [CMExpressionNode node];
      break;
    } else {
      [nodes insertObject:out atIndex:0];
      [outs removeLastObject];
    }
  }
  if (!node)
    [NSException raise:@"CMSyntaxException" format:@"parenthesis parse error"];

  [node addNodesInArray:nodes];
  [aStack addObject:node];
}

- (void)reduceDictionaryNodeToStack:(NSMutableArray *)aStack
{
  CMDictionaryNode *node = nil;
  id out;
  NSMutableArray *nodes = [NSMutableArray arrayWithCapacity:1];

  while ([outs count] > 0) {
    out = [outs lastObject];
    if ([out isEqual:CMMutableDictionaryNodeMark] ||
        [out isEqual:CMImmutableDictionaryNodeMark]) {
      [outs removeLastObject];
      node = [CMDictionaryNode node];
      [node setIsMutable:[out isEqual:CMMutableDictionaryNodeMark]];
      break;
    } else {
      [nodes insertObject:out atIndex:0];
      [outs removeLastObject];
    }
  }
  if (!node)
    [NSException raise:@"CMSyntaxException" format:@"dictionary parse error"];

  [node addNodesInArray:nodes];
  [aStack addObject:node];
}

- (void)reduceSubstituteNodeToStack:(NSMutableArray *)aStack
{
  if ([outs count] != 2)
    [NSException raise:@"CMSyntaxException"
                 format:@"'=' requires 2 nodes - %d node(s)", [outs count]];
  CMSubstituteNode *node;
  node = [CMSubstituteNode node];
  [node addNodesInArray:outs];
  [aStack addObject:node];
}

- (void)reduceReturnNodeToStack:(NSMutableArray *)aStack
{
  if ([outs count] > 1) {
    LOG(@"[reduceReturnNodeToStack:] %@", [outs description]);
    [NSException raise:@"CMSyntaxException"
                 format:@"'return' requires 0 or 1 node - %d node(s)", [outs count]];
  }
  CMReturnNode *node;
  node = [CMReturnNode node];
  if (![[outs lastObject] isEqual:[NSNull class]])
    [node addNodesInArray:outs];
  [aStack addObject:node];
}

- (void)reduceTerminateToStack:(NSMutableArray *)aStack
{
  [aStack addObjectsFromArray:outs];
}

@end
