#import "common.h"
#import "CMRuntime.h"
#import "CMLexer.h"
#import "CMParser.h"
#import "CMNode.h"
#import "CMLoadNodeVisitor.h"
#import "CMEvaluateNodeVisitor.h"
#import "CMNumber.h"
#import "CMSmallInteger.h"
#import "CMException.h"
#import "CMEnvironment.h"

static id defaultRuntime = nil;

@implementation CMRuntime

+ (id)defaultRuntime
{
    if (!defaultRuntime)
        defaultRuntime = [[self alloc] init];
    return defaultRuntime;
}

- (NSMutableArray *)loadPaths { return loadPaths; }
- (NSString *)loadingFileName { return loadingFileName; }
- (void)setLoadingFileName:(NSString *)aName { ASSIGN(loadingFileName, aName); }

- (void)loadWithPath:(NSString *)path
{
  [self loadWithPath:path reload:false];
}

- (void)loadWithPath:(NSString *)path reload:(BOOL)flag
{
  id loadPath, e;
  NSFileManager *fm;

  if (flag && [loadedPaths containsObject:path])
    return;

  fm = [NSFileManager defaultManager];
  if (![path hasSuffix:@".cm"])
    path = [path stringByAppendingPathExtension:[app pathExtension]];
  if ([fm fileExistsAtPath:path]) {
    LOG(@"* load %@", path);
    [loadedPaths addObject:path];
    [self setLoadingFileName:path];
    [self loadWithContentsOfFile:path];
    [self setLoadingFileName:nil];
    return;
  }

  e = [loadPaths objectEnumerator];
  while (loadPath = [e nextObject]) {
    loadPath = [loadPath stringByAppendingPathComponent:path];
    if ([fm fileExistsAtPath:loadPath]) {
      LOG(@"* load %@", loadPath);
      [loadedPaths addObject:path]; // not load path
      [self setLoadingFileName:loadPath];
      [self loadWithContentsOfFile:loadPath];
      [self setLoadingFileName:nil];
      return;
    }
  }

  [CMException raiseNoSuchFileToLoadExceptionWithPath:path];
}

- (id)loadFromStandardInput
{
  id handle, result;
  NSData *data;
  NSString *script;

  handle = [NSFileHandle fileHandleWithStandardInput];
  data = [handle readDataToEndOfFile];
  script = [[NSString alloc] initWithData:data
                             encoding:[CMApplication defaultEncoding]];
  result = [self loadWithString:script];
  [script release];
  return result;
}

- (void)loadAllFilesInCurrectDirectoryPath
{
  [self loadAllFilesInDirectoryAtPath:
          [[NSFileManager defaultManager] currentDirectoryPath]];
}

- (void)loadAllFilesInDirectoryAtPath:(NSString *)path;
{
  NSString *file;
  NSDirectoryEnumerator *e =
    [[NSFileManager defaultManager] enumeratorAtPath:path];
  NSString *ext = [[CMApplication defaultApplication] pathExtension];
  while (file = [e nextObject]) {
    if ([[file pathExtension] isEqualToString:ext]) {
      [self loadWithContentsOfFile:file];
    }
  }
}

- (id)loadWithContentsOfFile:(NSString *)aFileName
{
  LOG(@"* load script: %@", aFileName);
  [self setCurrentFileName:aFileName];
  NSString *contents = [NSString stringWithContentsOfFile:aFileName];
  if (!contents)
    [CMException raiseNoSuchFileToLoadExceptionWithPath:aFileName];
  id value = [self loadWithString:contents];
  [self setCurrentFileName:CMDefaultFileName];
  return value;
}

- (id)loadWithString:(NSString *)aString
{
  NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
  CMLexer *lexer;
  CMNode *loadNode = nil;
  [CMContext flushSender];
  [CMContext setSender:self];

  NS_DURING
    lexer = [CMLexer lexerWithString:aString context:self];
    loadNode = [CMParser parseWithLexer:lexer];
    [loadNode acceptVisitor:[CMLoadNodeVisitor visitorWithContext:self]];
    [loadNode acceptVisitor:[CMEvaluateNodeVisitor visitorWithContext:self]];

  NS_HANDLER
    [self displayException:localException];
    [pool release];
    [localException raise];
  NS_ENDHANDLER

  [CMContext flushSender];
  [pool release];
  return returnValue;
}

- (id)init
{
  [super init];
  app = [CMApplication defaultApplication];
  [app initializeRuntime:self];
  [self initializeLoadPaths];
  loadPaths = [[NSMutableArray alloc] initWithCapacity:1];
  loadedPaths = [[NSMutableArray alloc] initWithCapacity:1];
  loadingFileName = @"-";
  return self;
}

- (void)dealloc
{
  [app release];
  [loadPaths release];
  [loadedPaths release];
  [loadingFileName release];
  [super dealloc];
}

- (void)initializeLoadPaths
{
  NSString *lib = [[app environment] objectForKey:CMEnvironmentLibraryKey];
  if (lib) [loadPaths addObject:lib];
  [loadPaths addObject:[app userLibrarySiteVersionPath]];
  [loadPaths addObject:[app userLibrarySitePath]];
  [loadPaths addObject:[app userLibraryVersionPath]];
  [loadPaths addObject:[app localLibrarySiteVersionPath]];
  [loadPaths addObject:[app localLibrarySitePath]];
  [loadPaths addObject:[app localLibraryVersionPath]];
  [loadPaths addObject:[app currentDirectoryPath]];
}

- (id)evaluate:(NSString *)aString
{
  CMLexer *lexer;
  CMNode *evalNode = nil;

  NS_DURING
    lexer = [CMLexer lexerWithString:aString context:self];
    evalNode = [CMParser parseWithLexer:lexer];
    [evalNode acceptVisitor:[CMLoadNodeVisitor visitorWithContext:self]];
    [evalNode acceptVisitor:[CMEvaluateNodeVisitor visitorWithContext:self]];
  NS_HANDLER
    [self displayException:localException];
    [localException raise];
  NS_ENDHANDLER

  return returnValue;
}

- (BOOL)loadsCurrentFile
{
  return [loadingFileName isEqualToString:currentFileName];
}

@end
