#import "common.h"
#import "CMParser.h"
#import "CMException.h"
#import "CMSelectorNode.h"
#import "CMRegularExpressionNode.h"
#import "CMToken.h"
#import "CMContext.h"
#import "CMVariableNode.h"
#import "CMVariableDeclarationNode.h"
#import "CMKeyPathNode.h"
#import "CMKeyValueCodingNode.h"
#import "CMMessageNode.h"
#import "CMBlockNode.h"
#import "CMBlockArgumentsNode.h"
#import "CMExpressionNode.h"
#import "CMSubstituteNode.h"
#import "CMClassNode.h"
#import "CMMethodNode.h"
#import "CMStringNode.h"
#import "CMArrayNode.h"
#import "CMDictionaryNode.h"
#import "CMDigitNode.h"
#import "CMReturnNode.h"
#import "CMClass.h"
#import "CMMethod.h"
#import "CMLexer.h"
#import "CMSyntaxRule.h"

#define NotLocalVariableDeclarationReason \
  @"can't declare class/instance variable '%@' in method or block"

@implementation CMParser

+ (CMNode *)parseWithLexer:(CMLexer *)aLexer
{
  CMParser *parser = [[[self alloc] initWithLexer:aLexer] autorelease];
  [parser parse];
  return [parser node];
}

- (id)initWithLexer:(CMLexer *)aLexer
{
  [self init];
  lexer = [aLexer retain];
  context = [[lexer context] retain];
  node = [[CMExpressionNode alloc] init];
  return self;
}

- (void)dealloc
{
  [lexer release];
  [context release];
  [node release];
  [super dealloc];
}

- (CMNode *)node { return node; }

- (void)parse
{
  NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
  NSArray *tokens = [lexer tokens];
  NSMutableArray *tokenBuffer = [NSMutableArray arrayWithCapacity:1];
  NSMutableArray *exprTokens = [NSMutableArray arrayWithCapacity:1];
  NSMutableString *sourceCode = nil;
  CMNode *currentNode = node;
  CMNode *preNode, *newNode = nil;
  CMToken *token, *preToken = nil;
  BOOL continueMessage = NO, isInstanceVariable = YES;;
  int i, bracketCount = 0, parenthesisCount = 0, braceCount = 0;
  //LOG([tokens description]);

  [context clearCurrentLineNumber];
  [currentNode setFileName:[context currentFileName]];
  for (i = 0; i < [tokens count]; i++) {
    token = [tokens objectAtIndex:i];

    // start of class definition
    if ([token isClassReservedWord] &&
        (!preToken || [preToken isTerminate])) {
      preNode = currentNode;
      currentNode = [CMClassNode node];
      [preNode addNode:currentNode];
      [(CMClassNode *)currentNode setComment:[token prefixContent]];
      LOG(@"* class comment:\n%@", [(CMClassNode *)currentNode comment]);
                
    // in class definition
    } else if ([currentNode isMemberOfClass:[CMClassNode class]]) {
      [sourceCode appendString:[token sourceCode]];
            
      // define class name
      if (![(CMClassNode *)currentNode isNamed]) {
        LOG(@"last token: %@", [[tokenBuffer lastObject] description]);

        // set class name
        if ([[tokenBuffer lastObject] isClassReservedWord]) {
          [(CMClassNode *)currentNode setName:[token tokenString]];

        // set superclass name
        } else if ([[tokenBuffer lastObject] isColon]) {
          [(CMClassNode *)currentNode setSuperclassName:[token tokenString]];
          [tokenBuffer removeAllObjects];
                        
        // ignore
        } else if ([token isColon]) {
                    
        // omit superclass
        } else if ([token isTerminate]) {
          [(CMClassNode *)currentNode setSuperclassName:@"NSObject"];
          [tokenBuffer removeAllObjects];
                        
        // error
        } else {
          [context decrementCurrentLineNumber];
          [context raiseSyntaxException];
        }
        
      // start declare class/instance variable
      } else if ([token isVerticalBar]) {
        newNode = [CMVariableDeclarationNode node];
        [currentNode addNode:newNode];
        currentNode = newNode;
        [currentNode setLineNumber:[context currentLineNumber]];

      // determine class/instance method
      } else if ([token isAtStartOfMethod]) {
        newNode = [CMMethodNode nodeWithOperatorString:[token tokenString]];
        [(CMMethodNode *)newNode setComment:[token prefixContent]];
        sourceCode = [NSMutableString stringWithString:[token tokenString]];
        [(CMClassNode *)currentNode addNode:newNode];
        preNode = currentNode;
        currentNode = newNode;

      // end class definition
      } else if ([token isEndReservedWord]) {
        preNode = currentNode;
        currentNode = [currentNode parentNode];
        [tokenBuffer removeAllObjects];
      }

    // declare class/instance variable
    } else if ([currentNode isMemberOfClass:[CMVariableDeclarationNode class]]) {
      if ([token isVerticalBar]) {
        preNode = currentNode;
        currentNode = [currentNode parentNode];
      } else if ([token isInstanceOperator]) {
        isInstanceVariable = YES;
      } else if ([token isClassOperator]) {
        isInstanceVariable = NO;
      } else {
        newNode = [CMVariableNode nodeWithName:[token tokenString]];
        [(CMVariableNode *)newNode setIsInstanceVariable:isInstanceVariable];
        [currentNode addNode:newNode];
        isInstanceVariable = YES;
      }

    // define method
    } else if ([[currentNode parentNode] isMemberOfClass:[CMClassNode class]]) {
      id nextToken, nextNextToken = nil;
      if ([tokens count] >= (i+2)) nextToken = [tokens objectAtIndex:(i+1)];
      if ([tokens count] >= (i+3)) nextNextToken = [tokens objectAtIndex:(i+2)];
      [sourceCode appendString:[token sourceCode]];

      // class/instance method
      if ([currentNode isMemberOfClass:[CMMethodNode class]] &&
                 ![(CMMethodNode *)currentNode isNamed]) {
        // ignore
        if ([token isNewLine] && [nextNextToken isColon]) {

        } else if ([token isSemicolon] ||
            ([token isNewLine] && ![nextNextToken isColon])) {
          LOG(@"current node %@", [currentNode description]);
          [newNode setLineNumber:[context currentLineNumber]];
          id exprNode = [CMExpressionNode node];
          [currentNode addNode:exprNode];
          currentNode = exprNode;

        } else {
          // arguments
          if ([preToken isColon]) {
            if ([token isIdentifier]) {
              [(CMMethodNode *)currentNode addArgumentName:[token tokenString]];
            } else {
              [context raiseSyntaxException];
            }
            
          // keyword
          } else {
            [(CMMethodNode *)currentNode appendName:[token tokenString]];
          }
        }
      } else {
        [context raiseSyntaxException];
      }

    // parse expression
    } else if ([currentNode isMemberOfClass:[CMExpressionNode class]]) {
      [sourceCode appendString:[token sourceCode]];
      id nextToken = nil, nextNextToken = nil, nextNextNextToken = nil;
      if ([tokens count] >= (i+2)) nextToken = [tokens objectAtIndex:(i+1)];
      if ([tokens count] >= (i+3)) nextNextToken = [tokens objectAtIndex:(i+2)];
      if ([tokens count] >= (i+4)) nextNextNextToken = [tokens objectAtIndex:(i+3)];

      if ((bracketCount == 0) && (parenthesisCount == 0) && (braceCount == 0) &&
          ([token isSemicolon] || ([token isNewLine] && !continueMessage))) {
        int i = 0;
        newNode = [self parseExpressionWithTokens:exprTokens indexPointer:&i];
        if (newNode) {
          [newNode setLineNumber:[context currentLineNumber]];
          [currentNode addNode:newNode];
        }
        [exprTokens removeAllObjects];
        continueMessage = NO;

      } else if ([token isEndReservedWord]) {
        [(CMMethodNode *)[currentNode parentNode] setSourceCode:sourceCode];
        LOG(@"* method comment:\n%@",
            [(CMMethodNode *)[currentNode parentNode] comment]);
        LOG(@"* method source code:\n%@",
            [(CMMethodNode *)[currentNode parentNode] sourceCode]);
        preNode = currentNode;
        currentNode = [[currentNode parentNode] parentNode]; // ClassNode

      } else {
        if      ([token isOpenBracket])       bracketCount++;
        else if ([token isCloseBracket])      bracketCount--;
        else if ([token isOpenParenthesis])   parenthesisCount++;
        else if ([token isCloseParenthesis])  parenthesisCount--;
        else if ([token isOpenBrace])         braceCount++;
        else if ([token isCloseBrace])        braceCount--;

        if (!([token isNewLine] && continueMessage))
          [exprTokens addObject:token];

        if ([nextToken isNewLine] &&
            ([token isColon] || [nextNextNextToken isColon])) {
          continueMessage = YES;
        } else {
          continueMessage = NO;
        }
      }

    } else {
      [context raiseSyntaxException];
    }

    if ([token isNewLine]) {
      [currentNode setLineNumber:[context currentLineNumber]];
      [context incrementCurrentLineNumberWithLength:[[token tokenString] length]];
      LOG(@"newline(%d): %@", [currentNode lineNumber], [preToken description]);
    }
    
    // preNode$B$C$FI,MW$+!)(B
    [tokenBuffer addObject:token];
    preToken = token;
  }

  if ((bracketCount > 0) || (braceCount > 0) || (parenthesisCount > 0))
    [context raiseSyntaxException];

  [pool release];
}

#define DEFAULT_MUTABLE YES

- (CMNode *)parseExpressionWithTokens:(NSArray *)tokens
                         indexPointer:(int *)index
{
  NSMutableArray *opStack = [[NSMutableArray alloc] initWithCapacity:1];
  NSMutableArray *outStack = [[NSMutableArray alloc] initWithCapacity:1];
  id e, token = nil, preToken = nil, nextToken = nil, nextNextToken = nil,
    lastOut, lastOp, newNode = nil, buffer = nil, eachExpr;
  CMNode *expr = nil;
  BOOL isInKeywordMessage = NO, isNextTokenArgument = NO,
    isInDeclareVariable = NO, isInArrayLiteral = NO,
    isKeyPath = NO, hasReceiver = NO, isNextBlockArgument = NO,
    isInDeclareBlockArguments = NO;
  NSMutableArray *exprs;

  LOG(@"* %d: parse tokens: %@", *index, [tokens description]);

  for (; *index < [tokens count]; (*index)++) {
    preToken = token;
    token = [tokens objectAtIndex:*index];
    lastOp = [opStack lastObject];
    lastOut = [outStack lastObject];

    if ((*index+1) < [tokens count])
      nextToken = [tokens objectAtIndex:(*index+1)];
    else
      nextToken = nil;

    if ((*index+2) < [tokens count])
      nextNextToken = [tokens objectAtIndex:(*index+2)];
    else
      nextNextToken = nil;

    LOG(@"(%d) current = %@, next = %@, nextnext = %@, pre = %@",
        *index, [token description], [nextToken description],
         [nextNextToken description], [preToken description]);

    // end of expr, array, ditionary, block
    if ([token isCloseParenthesis] || [token isCloseBrace] ||
        [token isCloseBracket] || [token isComma] || [token isTerminate] ||
        [token isAssociationLiteral]) {
      LOG(@"* end of expr, array, ditionary, block");
      break;

    // block arguments
    } else if (!hasReceiver && [token isColon]) {
      LOG(@"* parse block arguments");
      exprs = [NSMutableArray arrayWithCapacity:1];
      newNode = [CMBlockArgumentsNode node];
      isNextBlockArgument = NO;

      while (1) {
        token = [tokens objectAtIndex:*index];
        if (!isNextBlockArgument && [token isColon])
          isNextBlockArgument = YES;
        else if (isNextBlockArgument && [token isIdentifier]) {
          [exprs addObject:[CMVariableNode nodeWithName:[token tokenString]]];
          isNextBlockArgument = NO;
        } else if (!isNextBlockArgument && [token isVerticalBar]) {
          (*index)++;
          break;
        }

        (*index)++;
        if (*index < [tokens count]) {
          token = [tokens objectAtIndex:*index];
          if ([token isVerticalBar])
            break;
          else if (![token isColon] && ![token isIdentifier])
            [context raiseSyntaxException];
        } else {
          [context raiseSyntaxException];
        }
      }

      [newNode addNodesInArray:exprs];
      [outStack addObject:newNode];
      isInKeywordMessage = NO;
      isNextTokenArgument = NO;
      hasReceiver = NO;

    // variable declaration
    } else if (!hasReceiver && !isInDeclareBlockArguments &&
               [token isVerticalBar]) {
      // start variable declaration
      if (!isInDeclareVariable) {
        LOG(@"* start variable declaration");
        buffer = [CMVariableDeclarationNode node];
        isInDeclareVariable = YES;
        isInKeywordMessage = NO;
        isNextTokenArgument = NO;

      // end variable declaration
      } else {
        LOG(@"* end variable declaration");
        [outStack addObject:buffer];
        isInDeclareVariable = NO;
        break;
      }
      hasReceiver = NO;

    // in variable declaration
    } else if (isInDeclareVariable) {
      // error
      if ([token isInstanceOperator] || [token isClassOperator]) {
        NSString *reason = [NSString
                             stringWithFormat:NotLocalVariableDeclarationReason,
                             [nextToken tokenString]];
        [[NSException exceptionWithName:CMNameException
                      reason:reason
                      userInfo:[context userInfo]] raise];

      // add name
      } else if ([token isIdentifier]) {
        newNode = [CMVariableNode nodeWithName:[token tokenString]];
        [buffer addNode:newNode];

      } else {
        [context raiseSyntaxException];
      }

    // key path separater
    } else if ([token isDot]) {
      LOG(@"* key path separator: %@", [token tokenString]);
      if (![lastOut isMemberOfClass:[CMKeyValueCodingNode class]]) {
        newNode = [CMKeyValueCodingNode node];
        [newNode addNode:lastOut];
        [outStack removeLastObject];
        [outStack addObject:newNode];
      }
      isKeyPath = YES;
      hasReceiver = NO;

    // array
    } else if ([token isOpenParenthesis] &&
               ([preToken isMutablePrefix] || [preToken isImmutablePrefix])) {
      LOG(@"* parse array");
      exprs = [NSMutableArray arrayWithCapacity:1];
      newNode = [CMArrayNode node];
      [newNode setIsMutable:[preToken isMutablePrefix]];

      (*index)++;
      while (1) {
        eachExpr = [self parseExpressionWithTokens:tokens
                         indexPointer:index];
        LOG(@"* parsed an element in array");
        if (eachExpr)
          [exprs addObject:eachExpr];
        if (*index < [tokens count]) {
          token = [tokens objectAtIndex:*index];
          LOG(@"* %d: array end - %@", *index, [token tokenString]);
          if ([token isCloseParenthesis])
            break;
          else if ([token isComma])
            (*index)++;
          else
            [context raiseSyntaxException];
        } else {
          [context raiseSyntaxException];
        }
      }

      [newNode addNodesInArray:exprs];
      [outStack addObject:newNode];
      isInKeywordMessage = NO;
      isNextTokenArgument = NO;
      hasReceiver = YES;

    // expr
    } else if ([token isOpenParenthesis]) {
      (*index)++;
      newNode = [self parseExpressionWithTokens:tokens
                      indexPointer:index];
      LOG(@"* %d: parsed an expression", *index);
      if (!newNode)
        newNode = [CMExpressionNode node];
      if (*index < [tokens count]) {
        token = [tokens objectAtIndex:*index];
        LOG(@"* %d: expr end - %@", *index, [token tokenString]);
        if (![token isTerminate] && ![token isCloseParenthesis])
          [context raiseSyntaxException];
      }
      [outStack addObject:newNode];
      isInKeywordMessage = NO;
      isNextTokenArgument = NO;
      hasReceiver = YES;

    // dictionary
    } else if ([token isOpenBrace]) {
      LOG(@"* parse dictionary");
      exprs = [NSMutableArray arrayWithCapacity:1];
      newNode = [CMDictionaryNode node];
      if ([preToken isMutablePrefix] || [preToken isImmutablePrefix])
        [newNode setIsMutable:[preToken isMutablePrefix]];

      (*index)++;
      while (1) {
        eachExpr = [self parseExpressionWithTokens:tokens
                         indexPointer:index];
        LOG(@"* parsed an element in dictionary");
        if (eachExpr)
          [exprs addObject:eachExpr];
        if (*index < [tokens count]) {
          token = [tokens objectAtIndex:*index];
          LOG(@"* %d: dictionary end - %@", *index, [token tokenString]);
          if ([token isCloseBrace])
            break;
          else if ([token isComma] || [token isAssociationLiteral])
            (*index)++;
          else
            [context raiseSyntaxException];
        } else {
          [context raiseSyntaxException];
        }
      }

      [newNode addNodesInArray:exprs];
      [outStack addObject:newNode];
      isInKeywordMessage = NO;
      isNextTokenArgument = NO;
      hasReceiver = YES;

    // block
    } else if ([token isOpenBracket]) {
      LOG(@"* block");
      id blockExpr = [CMExpressionNode node];
      exprs = [NSMutableArray arrayWithCapacity:1];
      newNode = [CMBlockNode node];
      [newNode addNode:blockExpr];

      (*index)++;
      while (1) {
        eachExpr = [self parseExpressionWithTokens:tokens
                         indexPointer:index];
        if (eachExpr)
          [exprs addObject:eachExpr];
        LOG(@"* %d: parsed block expr: %@", *index, [eachExpr description]);
        if (*index < [tokens count]) {
          token = [tokens objectAtIndex:*index];
          if ([token isCloseBracket])
            break;
          else if (![token isVerticalBar] && ![token isTerminate])
            [context raiseSyntaxException];
          else
            (*index)++;
        } else {
          [context raiseSyntaxException];
        }
      }
      [blockExpr addNodesInArray:exprs];
      [outStack addObject:newNode];
      isInKeywordMessage = NO;
      isNextTokenArgument = NO;
      hasReceiver = YES;

    // mutable or immutable
    } else if ([token isMutablePrefix] || [token isImmutablePrefix]) {
      LOG(@"* mutable or immutable symbol: %@", [token tokenString]);

    // string literal
    } else if ([token isStringLiteral]) {
      LOG(@"* string literal: \"%@\"", [token tokenString]);
      newNode = [CMStringNode nodeWithString:[token unquotedString]];
      if ([preToken isMutablePrefix] || [preToken isImmutablePrefix])
        [newNode setIsMutable:[preToken isMutablePrefix]];
      else
        [newNode setIsMutable:DEFAULT_MUTABLE];
      [outStack addObject:newNode];
      hasReceiver = YES;
      isNextTokenArgument = NO;

    // digit literal
    } else if ([token isDigitLiteral]) {
      LOG(@"* digit literal: %@", [token tokenString]);
      newNode = [CMDigitNode nodeWithString:[token tokenString]];
      [outStack addObject:newNode];
      hasReceiver = YES;
      isNextTokenArgument = NO;

    // regexp literal
    } else if ([token isRegularExpressionLiteral]) {
      LOG(@"* regexp literal: %@", [token tokenString]);
      newNode = [CMRegularExpressionNode nodeWithString:[token tokenString]
                                         optionArray:[token regularExpressionOptions]];
      [outStack addObject:newNode];
      hasReceiver = YES;
      isNextTokenArgument = NO;

    // selector literal
    } else if ([token isSelectorLiteral]) {
      LOG(@"* selector literal: %@", [token tokenString]);
      newNode = [CMSelectorNode nodeWithString:[token selectorString]];
      [outStack addObject:newNode];
      hasReceiver = YES;
      isNextTokenArgument = NO;

    // return
    } else if (!hasReceiver && [token isReturnReservedWord]) {
      LOG(@"* return");
      newNode = [CMReturnNode node];
      (*index)++;

      eachExpr = [self parseExpressionWithTokens:tokens
                       indexPointer:index];
      if (!eachExpr)
        eachExpr = [CMVariableNode nodeWithName:@"self"];
      if (*index < [tokens count]) {
        token = [tokens objectAtIndex:*index];
        LOG(@"* parse return - %@", [token tokenString]);
        if (![token isTerminate] &&
            ![token isCloseParenthesis] && ![token isCloseBracket])
          (*index)++;
      }
      [newNode addNode:eachExpr];
      [outStack addObject:newNode];
      break;

    // substitute (=)
    } else if ([token isSubstituteOperatorString]) {
      LOG(@"* substitute: %@", [token tokenString]);
      newNode = [CMSubstituteNode node];
      (*index)++;
      eachExpr = [self parseExpressionWithTokens:tokens
                       indexPointer:index];
      if (!eachExpr)
        [context raiseSyntaxException];
      if (*index < [tokens count]) {
        token = [tokens objectAtIndex:*index];
        if (![token isTerminate] &&
            ![token isCloseParenthesis] && ![token isCloseBracket])
          [context raiseSyntaxException];
      }
      [newNode addNode:[outStack lastObject]];
      [outStack removeLastObject];
      [newNode addNode:eachExpr];
      [outStack addObject:newNode];
      break;

    // single or start keyword message
    } else if (hasReceiver && [token isMessage] && !isInKeywordMessage &&
               !isNextTokenArgument) {

      // keyword message
      if (nextToken && [nextToken isColon]) {
        if ([lastOp isKeywordMessage]) {
          lastOp = [CMToken tokenWithString:
                              [[lastOp tokenString] stringByAppendingString:[token tokenString]]];
          [opStack removeLastObject];
          [opStack addObject:lastOp];
          LOG(@"* concat keyword message: %@", [lastOp tokenString]);
        } else {
          LOG(@"* start keyword message: %@", [token tokenString]);
          [opStack addObject:token];
        }
        isInKeywordMessage = YES;

      } else {
        LOG(@"* single message: %@", [token tokenString]);
        [opStack addObject:token];
      }

    // keyword message (:)
    } else if (hasReceiver && [token isColon]) {
      LOG(@"* keyword message: %@", [token tokenString]);
      if (isInKeywordMessage) {
        LOG(@"* concat keyword message for %@", [lastOp tokenString]);
        lastOp = [CMToken tokenWithString:
                            [NSString stringWithFormat:@"%@:", [lastOp tokenString]]];
        [opStack removeLastObject];
        [opStack addObject:lastOp];
      } else {
        // colon only keyword
        [opStack addObject:token];
      }
      isNextTokenArgument = YES;
      isInKeywordMessage = NO;

    // binary message
    } else if (hasReceiver && [token isOperatorMessage]) {
      LOG(@"* binary message: %@", [token tokenString]);
      hasReceiver = NO;
      [opStack addObject:token];

    // binary message omits receiver (-1, +1)
    } else if (!hasReceiver &&
               ([token isPlusOperatorMessage] || [token isMinusOperatorMessage])) {
      // awkward
      if ([preToken isMinusOperatorMessage] && [preToken isMinusOperatorMessage]) {
        [opStack removeLastObject];
        [opStack addObject:[CMToken tokenWithString:@"+"]];
      } else {
        [opStack addObject:token];
        [outStack addObject:[CMDigitNode nodeWithString:@"0"]];
      }

    // key path
    } else if (isKeyPath) {
      newNode = [CMKeyPathNode nodeWithKeyPath:[token tokenString]];
      [lastOut addNode:newNode];
      isKeyPath = NO;
      hasReceiver = YES;

    // variable
    } else if ([token isIdentifier] && 
               ((nextToken && ![nextToken isColon]) || !nextToken)) {
      LOG(@"* variable: %@, %d, %d", [token tokenString],
          isInKeywordMessage, isNextTokenArgument);
      newNode = [CMVariableNode nodeWithName:[token tokenString]];
      hasReceiver = YES;

      if (isInArrayLiteral) {
        [lastOut addNode:newNode];

      } else {
        [outStack addObject:newNode];
        isNextTokenArgument = NO;
      }

    } else {
      [context raiseSyntaxException];
    }
  }

  LOG(@"* %d: construct stacks", *index);
  while ([opStack count] > 0)
    [self compositeExpressionNodeWithOperandStack:opStack outputStack:outStack];

  if ([outStack count] > 1) {
    expr = [CMExpressionNode node];
    e = [outStack objectEnumerator];
    while (lastOut = [e nextObject]) {
      [expr addNode:lastOut];
    }
  } else if ([outStack count] == 1) {
    expr = [outStack lastObject];
  }
  
  [opStack release];
  [outStack release];
  return expr;
}

- (void)compositeExpressionNodeWithOperandStack:(NSMutableArray *)opStack
                                    outputStack:(NSMutableArray *)outStack
{
  id leftOp;
  CMSyntaxRule *rightRule;

  LOG(@"-[CMParser composite...] op: %@ out: %@", [opStack description], [outStack description]);

  while ([opStack count] > 0) {
    rightRule = [CMSyntaxRule ruleWithOperandStack:opStack outputStack:outStack];
    leftOp = [opStack lastObject];

    if (leftOp &&
        [leftOp compare:[[rightRule operandStack] lastObject]] == NSOrderedAscending) {
      LOG(@"* returnTokensToOperandStack");
      [self compositeExpressionNodeWithOperandStack:opStack
            outputStack:outStack];
      [rightRule returnTokensToOperandStack:opStack outputStack:outStack];
      LOG(@"* now op: %@", [opStack description]);
    } else {
      if ([outStack count] > 0 && [rightRule requiresNodes])
        [rightRule takeNodesFromOperandStack:opStack outputStack:outStack];
      [rightRule reduceToStack:outStack];
      break;
    }
  }
}

@end
