#import "common.h"
#import "CMClass.h"
#import "CMMethod.h"
#import "CMObject.h"
#import "CMMethodNode.h"
#import "CMEvaluateNodeVisitor.h"
#import "CMMethodContext.h"
#import "CMException.h"
#import "CMLargeInteger.h"
#import "CMFloat.h"
#import "CMSelector.h"
#import "CMRect.h"
#import "CMRange.h"
#import "CMSize.h"
#import "CMPoint.h"
#import "CMNil.h"
#import "CMObjCTypeConverter.h"
#import "CMTrue.h"
#import "CMFalse.h"
#import "CMOperatorMethods.h"
#import "CMMethodProxy.h"
#import <objc/objc-runtime.h>

#define TooMuchArgumentsReason @"can't define method with too much arguments - %@"


static NSDictionary *opDict = nil;

@implementation CMMethod

+ (NSDictionary *)operatorDictionary
{
  if (!opDict) {
    opDict = [[NSDictionary alloc] initWithObjectsAndKeys:
      CMStringPlus, CMOperatorPlus,
      CMStringMinus, CMOperatorMinus,
      CMStringTimes, CMOperatorTimes,
      CMStringDividedBy, CMOperatorDividedBy,
      CMStringModuloOfDividedBy, CMOperatorModuloOfDividedBy,
      CMStringRemainderOfDividedBy, CMOperatorRemainderOfDividedBy,
      CMStringRaisingToPower, CMOperatorRaisingToPower,
      CMStringEqualTo, CMOperatorEqualTo,
      CMStringNotEqualTo, CMOperatorNotEqualTo,
      CMStringGreaterThan, CMOperatorGreaterThan,
      CMStringGreaterThanOrEqualTo, CMOperatorGreaterThanOrEqualTo,
      CMStringLessThan, CMOperatorLessThan,
      CMStringLessThanOrEqualTo, CMOperatorLessThanOrEqualTo,
      CMStringCompare, CMOperatorCompare,
      CMStringShiftRight, CMOperatorShiftRight,
      CMStringShiftLeft, CMOperatorShiftLeft, nil];
  }
  return opDict;
}

+ (id)methodWithName:(NSString *)aName
            selector:(SEL)aSelector
           forObject:(id)anObject
{
  return [[[self alloc] initWithName:aName
                        selector:aSelector
                        forObject:anObject] autorelease];
}

+ (id)methodWithNode:(CMMethodNode *)aNode
{
  return [[[self alloc] initWithNode:aNode] autorelease];
}

- (id)initWithNode:(CMMethodNode *)aNode
{
  node = [aNode retain];
  metaClass = [[aNode metaClass] retain];
  name = [[NSString alloc] initWithString:[aNode name]];
  SEL sel = [self invokeSelectorForArgumentCount:[node argumentCount]];
  if ([node isClassMethod]) {
    [self initWithName:[node name]
          selector:sel
          forObject:[CMMethodProxy class]];
  } else {
    [self initWithName:[node name]
          selector:sel
          forObject:[CMMethodProxy defaultMethodProxy]];
  }
  return self;
}

- (id)initWithName:(NSString *)aName
{
  return [self initWithName:aName
               selector:@selector(invokeWithObjects:)
               forObject:[CMMethodProxy defaultMethodProxy]];
}

- (id)initWithName:(NSString *)aName
          selector:(SEL)aSelector
         forObject:(id)anObject
{
  [self init];
  name = [[NSString alloc] initWithString:aName];
  selector = sel_registerName([name cString]);
  methodSelector = aSelector;
  method = [anObject methodForSelector:methodSelector];
  metaClass = [[CMClass classForObject:anObject] retain];
  return self;
}

+ (id)performSelector:(SEL)aSelector
            forObject:(id)anObject
          withObjects:(id)anArgument, ...;
{
  // MethodNode$B$r;}$A!"%a%=%C%I8F$S=P$7;~$K<B9T$9$k(B
  NSMutableArray *arguments = [NSMutableArray arrayWithCapacity:1];
  va_list ap;
  id arg;

  if (anArgument) {
    [arguments addObject:anArgument];
    va_start(ap, anArgument);
    while (arg = va_arg(ap, id)) {
      [arguments addObject:arg];
    }
    va_end(ap);
  }
  return [self performSelector:aSelector forObject:anObject withArguments:arguments];
}

+ (id)performSelector:(SEL)aSelector
            forObject:(id)anObject
        withArguments:(NSArray *)arguments
{
  NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
  CMClass *class;
  NSString *methodName;
  CMMethod *metaMethod;
  CMMethodContext *context;
  CMEvaluateNodeVisitor *visitor;
  id value;

  class = [CMClass classForObject:anObject];
  methodName = NSStringFromSelector(aSelector);
  if (IS_CLASS(anObject))
    metaMethod = [class classMethodNamed:methodName];
  else
    metaMethod = [class instanceMethodNamed:methodName];

  context = [CMMethodContext contextWithSender:[CMContext sender]
                             receiver:anObject
                             method:metaMethod
                             arguments:arguments];
  [CMContext setSender:context];
  visitor = [CMEvaluateNodeVisitor visitorWithContext:context];
  [[metaMethod node] acceptVisitor:visitor];
  value = [[context returnValue] retain];
  [CMContext revertSender];
  [pool release];
  return [value autorelease];
}

- (NSString *)name { return name; }
- (SEL)selector { return selector; }
- (SEL)methodSelector { return methodSelector; }
- (IMP)method { return method; }
- (CMMethodNode *)node { return node; }

- (struct objc_method *)objCMethod
{
  struct objc_method *objc_method;
  Class class;
  Method superMethod;
  NSMethodSignature *sig;
  IMP methodIMP;

  objc_method = malloc(sizeof(struct objc_method));
  objc_method->method_name = selector;
  class = [metaClass objCClass];

  // override method
  if ([class instancesRespondToSelector:selector]) {
    superMethod = class_getInstanceMethod(class, selector);
    sig = [class instanceMethodSignatureForSelector:selector];
    objc_method->method_types = superMethod->method_types;
    methodIMP = [CMMethodProxy methodForEncodeType:[sig methodReturnType]];
    if (!methodIMP)
      [CMException raiseReturnTypeExceptionWithType:[sig methodReturnType]
                   object:class
                   selector:selector];
    objc_method->method_imp = methodIMP;

  // new method
  } else {
    // runtime document describes that runtime ignores this, but do not ignore :-(
    // this value is irresponsible, but works.
    objc_method->method_types = "@@:@@@@@@@@@";
    objc_method->method_imp = method;
  }

  return objc_method;
}

- (CMClass *)metaClass { return metaClass; }

- (void)setMetaClass:(CMClass *)aClass
{
  [aClass retain];
  [metaClass release];
  metaClass = aClass;
}

- (CMVariable *)variableNamed:(NSString *)aName
                forMetaObject:(id)anObject
{
  // local, instance, class
  id var, e, stores, store;
  stores = [NSMutableArray arrayWithObjects:self, anObject, [self metaClass], nil];
  e = [stores objectEnumerator];
  while (store = [e nextObject]) {
    if (var = [store variableNamed:aName])
      return var;
  }
  return nil;
}

- (SEL)invokeSelectorForArgumentCount:(int)count
{
  SEL sel = nil;
  switch (count) {
  case 0: sel = @selector(_invoke); break;
  case 1: sel = @selector(_invokeWithObject:); break;
  case 2: sel = @selector(_invokeWithObject::); break;
  case 3: sel = @selector(_invokeWithObject:::); break;
  case 4: sel = @selector(_invokeWithObject::::); break;
  case 5: sel = @selector(_invokeWithObject:::::); break;
  case 6: sel = @selector(_invokeWithObject::::::); break;
  case 7: sel = @selector(_invokeWithObject::::::); break;
  case 8: sel = @selector(_invokeWithObject:::::::); break;
  case 9: sel = @selector(_invokeWithObject::::::::); break;
  default:
    [NSException raise:CMRuntimeException format:TooMuchArgumentsReason, name];
    break;
  }
  return sel;
}

- (void)dealloc
{
  [name release];
  [node release];
  [metaClass release];
  [super dealloc];
}

+ (SEL)selectorForString:(NSString *)aString
{
  SEL sel = [self operatorSelectorForString:aString];
  if (sel)
    return sel;
  else
    return NSSelectorFromString(aString);
}

+ (SEL)operatorSelectorForString:(NSString *)aString
{
  NSString *msg;
  NSDictionary *dict = [self operatorDictionary];

  if (msg = [dict objectForKey:aString])
    return NSSelectorFromString(msg);
  else
    return nil;
}

+ (NSString *)setterNameForKey:(NSString *)key
{
  NSString *first, *other, *setter;
  first = [[key substringToIndex:1] capitalizedString];
  other = [key substringFromIndex:1];
  setter = [NSString stringWithFormat:@"set%@%@:", first, other];
  return setter;
}

+ (NSString *)descriptionWithObject:(id)anObject selector:(SEL)aSelector
{
  NSString *op, *className, *methodName;

  if (NSClassFromString([anObject description]))
    op = @"+";
  else
    op = @"-";

  className = [[anObject class] description];
  methodName = NSStringFromSelector(aSelector);

  return [NSString stringWithFormat:@"%@[%@ %@]", op, className, methodName];
}

- (NSString *)description
{
  NSString *op;
  if ([metaClass classMethodNamed:name])
    op = @"+";
  else
    op = @"-";

  return [NSString stringWithFormat:@"%@[%@ %@]", op, [metaClass name], name];
}

- (NSArray *)argumentNames
{
  return [node argumentNames];
}

- (unsigned)numberOfArguments
{
  return [[name componentsSeparatedByString:@":"] count] - 1;
}

- (NSString *)comment
{
  return [node comment];
}

- (NSString *)sourceCode
{
  return [node sourceCode];
}

@end

