#import "common.h"
#import "CMNode.h"
#import "CMException.h"
#import "CMLoadNodeVisitor.h"
#import "CMVariableNode.h"
#import "CMVariableDeclarationNode.h"
#import "CMMessageNode.h"
#import "CMBlockNode.h"
#import "CMExpressionNode.h"
#import "CMSubstituteNode.h"
#import "CMArrayNode.h"
#import "CMDictionaryNode.h"
#import "CMReturnNode.h"
#import "CMClassNode.h"
#import "CMMethodNode.h"
#import "CMClass.h"
#import "CMMethod.h"
#import "CMContext.h"
#import "CMKeyPathNode.h"
#import "CMKeyValueCodingNode.h"

#define UndefinedSuperclassReason @"undefined superclass '%@'"
#define SuperclassMismatchReason  @"superclass mismatch for class '%@'"

// $B%/%i%9$N%m!<%I$d%a%=%C%IDj5A!"$=$NB>%a%C%;!<%8Aw?.$J$I$N:GE,2=$r9T$&!#(B
@implementation CMLoadNodeVisitor

- (void)visitClassNode:(CMClassNode *)aNode
{
  NSString *reason = nil, *exception = nil;
  Class class, superclass;

  class = NSClassFromString([aNode name]);
  superclass = NSClassFromString([aNode superclassName]);

  // superclass is not found
  if (!superclass) {
    exception = CMNameException;
    reason = [NSString stringWithFormat:UndefinedSuperclassReason,
                                 [aNode superclassName]];

  // mismatch superclass
  } else if (class && superclass && ![class isSubclassOfClass:superclass]) {
    exception = CMTypeException;
    reason = [NSString stringWithFormat:SuperclassMismatchReason,
                                 [aNode superclassName]];
  }

  if (exception && reason) {
    [[NSException exceptionWithName:exception
                  reason:reason
                  userInfo:[aNode userInfo]] raise];
  }

  [CMClass addClassWithName:[aNode name]
           superclassName:[aNode superclassName]];
  LOG(@"* define class: %@ superclass: %@", [aNode name], [aNode superclassName]);
  [aNode nodesAcceptVisitor:self];
}

- (void)visitVariableNode:(CMVariableNode *)aNode
{
}

- (void)visitMethodNode:(CMMethodNode *)aNode
{
  CMMethod *method;
  CMClass *class;

  // define class/instance method
  if ([[aNode parentNode] isMemberOfClass:[CMClassNode class]]) {
    NS_DURING
      class = [aNode metaClass];
      method = [CMMethod methodWithNode:aNode];
      [method setMetaClass:class];

      if ([aNode isClassMethod])
        [class addClassMethod:method];
      else
        [class addInstanceMethod:method];

    NS_HANDLER
      localException = [NSException exceptionWithName:[localException name]
                                    reason:[localException reason]
                                    userInfo:[aNode userInfo]];
      [localException raise];
    NS_ENDHANDLER
  }
  [aNode nodesAcceptVisitor:self];
}

- (void)visitExpressionNode:(CMExpressionNode *)aNode
{
  [aNode nodesAcceptVisitor:self];
}

- (void)visitVariableDeclarationNode:(CMVariableDeclarationNode *)aNode
{
  id varNode, e;

  // error check
  e = [[aNode nodes] objectEnumerator];
  while (varNode = [e nextObject]) {
    if ([varNode isPseudoVariable]) {
      NSString *reason = [NSString stringWithFormat:CMPseudoVariableReason,
                                   [varNode name]];
      [[NSException exceptionWithName:CMTypeException
                    reason:reason
                    userInfo:[aNode userInfo]] raise];      
    }
  }

  // define class/instance variable
  if ([[aNode parentNode] isMemberOfClass:[CMClassNode class]]) {
    e = [[aNode nodes] objectEnumerator];
    while (varNode = [e nextObject]) {
      CMClass *class = [varNode classWrapper];
      if ([varNode isClassVariable]) {
        [class addClassVariableName:[varNode name]];
        LOG(@"* declare class variable: %@", [varNode name]);
      } else {
        [class addInstanceVariableName:[varNode name]];
        LOG(@"* declare instance variable: %@", [varNode name]);
      }
    }
  }
}

- (void)visitBlockArgumentsNode:(CMBlockArgumentsNode *)aNode
{
}

- (void)visitMessageNode:(CMMessageNode *)aNode
{
  // $B%a%C%;!<%8$r%;%l%/%?$K$7$F%;%C%H$9$k(B
  // $B1i;;;R%*%Z%l!<%?$b$3$3$G%;%l%/%?2=(B
  // $B$3$l$O%Q!<%6$G$d$k$Y$-;E;v$G$O$J$$$N$+(B
  [aNode nodesAcceptVisitor:self];
  [aNode setSelector:[CMMethod selectorForString:[aNode message]]];
}

- (void)visitSubstituteNode:(CMSubstituteNode *)aNode
{
  [[[aNode nodes] objectAtIndex:1] acceptVisitor:self];
}

- (void)visitBlockNode:(CMBlockNode *)aNode
{
  [aNode nodesAcceptVisitor:self];
}

- (void)visitArrayNode:(CMArrayNode *)aNode
{
  [aNode nodesAcceptVisitor:self];
}

- (void)visitDictionaryNode:(CMDictionaryNode *)aNode
{
  [aNode nodesAcceptVisitor:self];
}

- (void)visitReturnNode:(CMReturnNode *)aNode
{
  [aNode nodesAcceptVisitor:self];
}

- (void)visitKeyValueCodingNode:(CMKeyValueCodingNode *)aNode
{
  [[aNode expression] acceptVisitor:self];
}

- (void)visitDigitNode:(CMDigitNode *)aNode {}
- (void)visitStringNode:(CMStringNode *)aNode {}
- (void)visitSelectorNode:(CMSelectorNode *)aNode {}
- (void)visitRegularExpressionNode:(CMRegularExpressionNode *)aNode {}
- (void)visitKeyPathNode:(CMKeyPathNode *)aNode {}

@end
