#import "common.h"
#import "CMObjCTypeConverter.h"
#import "CMLargeInteger.h"
#import "CMSmallInteger.h"
#import "CMException.h"

#define LLSIZE 25

@implementation CMLargeInteger

+ (id)integerWithLongLong:(long long)aValue
{
  BOOL flag = aValue > 0;
  if (!flag) aValue = aValue * -1;
  return [[[self alloc] initWithUnsignedLongLong:(unsigned long long)aValue
                        sign:flag] autorelease];
}

+ (id)integerWithUnsignedLongLong:(unsigned long long)aValue sign:(BOOL)flag
{
  return [[[self alloc] initWithUnsignedLongLong:aValue sign:flag] autorelease];
}

+ (id)integerWithDigitLength:(long)aDigitLength sign:(BOOL)flag
{
  return [[[self alloc] initWithDigitLength:aDigitLength sign:flag] autorelease];
}

- (id)initWithUnsignedLongLong:(unsigned long long)aValue sign:(BOOL)flag
{
  [self init];
  sign = flag;
  unsigned long long num = aValue;
  long i = 0;

  digits = malloc(sizeof(BDIGIT)*DIGSPERLONG);
  while (i < DIGSPERLONG) {
    digits[i++] = BIGLO(num);
    num = BIGDN(num);
  }

  i = DIGSPERLONG;
  while (--i && !digits[i]) ;
  digitLength = i + 1;

  return self;
}

- (id)initWithDigitLength:(long)aDigitLength sign:(BOOL)flag
{
  [self init];
  sign = flag;
  digitLength = aDigitLength;
  digits = malloc(sizeof(BDIGIT)*(digitLength));
  return self;
}

- (BDIGIT *)digits                  { return digits; }
- (long)digitLength                 { return digitLength; }
- (void)setDigitLength:(long)aValue { digitLength = aValue; }
- (BOOL)sign                        { return sign; }
- (void)setSign:(BOOL)flag          { sign = flag; }

- (unsigned)hash
{
  long i, key;

  key = 0;
  for (i=0; i<digitLength; i++) {
    key ^= *digits++;
  }
  return (unsigned)key;
}

- (id)copyWithZone:(NSZone *)zone
{
  id clone = [[[self class] allocWithZone:zone] initWithDigitLength:digitLength
                                                sign:sign];
  memcpy(BDIGITS(clone), BDIGITS(self), sizeof(BDIGIT)*(digitLength));
  return clone;
}

const char digitmap[] = "0123456789abcdefghijklmnopqrstuvwxyz";
- (NSString *)stringWithBase:(int)base
{
  volatile id t;
  BDIGIT *ds;
  long i, j, hbase, slen;
  char *s, c;

  i = digitLength;
  if ([self isZero]) {
    return @"0";
  }

  j = SIZEOF_BDIGITS * CHAR_BIT * i;
  switch (base) {
  case 2: break;
  case 3:
    j = j * 647L / 1024;
    break;
  case 4: case 5: case 6: case 7:
    j /= 2;
    break;
  case 8: case 9:
    j /= 3;
    break;
  case 10: case 11: case 12: case 13: case 14: case 15:
    j = j * 241L / 800;
    break;
  case 16: case 17: case 18: case 19: case 20: case 21:
  case 22: case 23: case 24: case 25: case 26: case 27:
  case 28: case 29: case 30: case 31:
    j /= 4;
    break;
  case 32: case 33: case 34: case 35: case 36:
    j /= 5;
    break;
  default:
    // error
    //rb_raise(rb_eArgError, "illegal radix %d", base);
    LOG(@"illegal radix %d", base);
    break;
  }
  j += 2;

  hbase = base * base;
#if SIZEOF_BDIGITS > 2
  hbase *= hbase;
#endif

  t = [[self copy] autorelease];
  ds = BDIGITS(t);
  slen = j;
  s = malloc(sizeof(char) * (j+1));
  
  s[0] = sign ? '+' : '-';
  while (i && j) {
    long k = i;
    BDIGIT_DBL num = 0;

    while (k--) {
	    num = BIGUP(num) + ds[k];
	    ds[k] = (BDIGIT)(num / hbase);
	    num %= hbase;
    }
    if (ds[i-1] == 0) i--;
    k = SIZEOF_BDIGITS;
    while (k--) {
	    c = (char)(num % base);
	    s[--j] = digitmap[(int)c];
	    num /= base;
	    if (i == 0 && num == 0) break;
    }
  }
  while (s[j] == '0') j++;
  slen -= sign ? j : j - 1;
  memmove(sign?s:s+1, s+j, slen);
  s[slen] = '\0';

  return [NSString stringWithCString:s];
}

- (BOOL)isZero
{
  return (digitLength == 0 || digitLength == 1 && digits[0] == 0);
}

- (CMFloat *)float
{
  return [CMFloat floatWithDouble:[self doubleValue]];
}

- (double)doubleValue
{
  double d = 0.0;
  long i = digitLength;

  while (i--) {
    d = digits[i] + BIGRAD * d;
  }
  if (isinf(d)) {
    //rb_warn("Bignum out of Float range");
    d = HUGE_VAL;
  }
  if (!sign) d = -d;
  return d;
}

- (long)longValue
{
  unsigned long num = [self unsignedLongValueWithType:@"long"];
  if ((long)num < 0 && (sign || (long)num != LONG_MIN)) {
    [NSException raise:NSRangeException
                 format:@"CMLargeInteger too big to convert into `long'"];
  }
  if (!sign) return -(long)num;
  return num;
}

- (unsigned long)unsignedLongValue
{
  return [self unsignedLongValueWithType:@"unsigned long"];
}

- (unsigned long)unsignedLongValueWithType:(NSString *)type
{
  long len = digitLength;
  BDIGIT_DBL num;
  BDIGIT *ds;

  if (len > SIZEOF_LONG/SIZEOF_BDIGITS)
    [NSException raise:NSRangeException
                 format:@"bignum too big to convert into `%@'", type];
  ds = digits;
  num = 0;
  while (len--) {
    num = BIGUP(num);
    num += ds[len];
  }
  return num;
}

- (CMInteger *)integer
{
  long len = digitLength;

  while (len-- && !digits[len]) ;
  digitLength = ++len;

  if (len * SIZEOF_BDIGITS <= sizeof(long)) {
    long num = 0;
    while (len--) {
      num = BIGUP(num) + digits[len];
    }
    if (num >= 0) {
      if (sign) {
        if (POSFIXABLE(num)) return PLNG2SINT(num);
      }
      else if (NEGFIXABLE(-(long)num)) return PLNG2SINT(-(long)num);
    }
  }
  return self;
}

- (NSString *)description
{
  return [self stringWithBase:10];
}

- (NSComparisonResult)compareCMSmallInteger:(CMSmallInteger *)anInteger
{
  return [self compareCMLargeInteger:[anInteger largeInteger]];
}

- (NSComparisonResult)compareCMLargeInteger:(CMLargeInteger *)anInteger
{
  long xlen = digitLength;

  if (sign > [anInteger sign]) return NSOrderedDescending;
  if (sign < [anInteger sign]) return NSOrderedAscending;
  if (digitLength < [anInteger digitLength])
    return (sign) ? NSOrderedAscending : NSOrderedDescending;
  if (digitLength > [anInteger digitLength])
    return (sign) ? NSOrderedDescending : NSOrderedAscending;

  while (xlen-- && (digits[xlen] == [anInteger digits][xlen]));
  if (-1 == xlen) return NSOrderedSame;
  return (digits[xlen] > [anInteger digits][xlen]) ?
    (sign ? NSOrderedDescending : NSOrderedAscending) :
    (sign ? NSOrderedAscending : NSOrderedDescending);
}

- (NSComparisonResult)compareCMFloat:(CMFloat *)anFloat
{
  return [[self float] compareCMFloat:anFloat];
}

- (id)objectByAddingCMSmallInteger:(CMSmallInteger *)anInteger
{
 return [[self objectByAddingCMLargeInteger:[anInteger largeInteger]] integer];
}

- (id)objectByAddingCMLargeInteger:(CMLargeInteger *)anInteger
{
  return [self objectByAddingCMLargeInteger:anInteger withSign:YES];
}

- (id)objectByAddingCMLargeInteger:(CMLargeInteger *)anInteger
                          withSign:(BOOL)flag
{
  CMLargeInteger *x, *y, *z;
  BDIGIT_DBL num;
  long i, len;
  flag = (flag == [anInteger sign]);

  x = self;
  y = anInteger;
  if ([x sign] != flag) {
    if (flag) return [y objectBySubtractingCMLargeInteger:x];
    return [x objectBySubtractingCMLargeInteger:y];
  }
  if (digitLength > [y digitLength]) {
    len = digitLength + 1;
    z = x; x = y; y = z;
  } else {
    len = [y digitLength] + 1;
  }
  z = [CMLargeInteger integerWithDigitLength:len sign:flag];
  len = digitLength;
  for (i = 0, num = 0; i < len; i++) {
    num += (BDIGIT_DBL)[x digits][i] + [y digits][i];
    [z digits][i] = BIGLO(num);
    num = BIGDN(num);
  }
  len = [y digitLength];
  while (num && i < len) {
    num += [y digits][i];
    [z digits][i++] = BIGLO(num);
    num = BIGDN(num);
  }
  while (i < len) {
    [z digits][i] = [y digits][i];
    i++;
  }
  [z digits][i] = (BDIGIT)num;
  return z;
}

- (id)objectByAddingCMFloat:(CMFloat *)aFloat
{
  return [CMFloat floatWithDouble:[self doubleValue] + [aFloat doubleValue]];
}

- (id)objectBySubtractingCMSmallInteger:(CMSmallInteger *)anInteger
{
  return [self objectBySubtractingCMLargeInteger:[anInteger largeInteger]];
}

- (id)objectBySubtractingCMLargeInteger:(CMLargeInteger *)anInteger
{
  CMLargeInteger *x = nil, *y = nil, *z = nil;
  BDIGIT *zds;
  BDIGIT_DBL_SIGNED num;
  long i = [x digitLength];

  x = self;
  y = anInteger;

  /* if x is larger than y, swap */
  if ([x digitLength] < [y digitLength]) {
    z = x; x = y; y = z;    /* swap x y */
  }
  else if ([x digitLength] == [y digitLength]) {
    while (i > 0) {
      i--;
      if ([x digits][i] > [y digits][i]) {
        break;
      }
      if ([x digits][i] < [y digits][i]) {
        z = x; x = y; y = z;    /* swap x y */
        break;
      }
    }
  }
  z = [CMLargeInteger integerWithDigitLength:[x digitLength] sign:(z == nil)?1:0];
  zds = [z digits];
  for (i = 0, num = 0; i < [y digitLength]; i++) { 
    num += (BDIGIT_DBL_SIGNED)[x digits][i] - [y digits][i];
    zds[i] = BIGLO(num);
    num = BIGDN(num);
  } 
  while (num && i < [x digitLength]) {
    num += [x digits][i];
    zds[i++] = BIGLO(num);
    num = BIGDN(num);
  }
  while (i < [x digitLength]) {
    zds[i] = [x digits][i];
    i++;
  }

  return [z integer];
}

- (id)objectBySubtractingCMFloat:(CMFloat *)aFloat
{
  return [CMFloat floatWithDouble:[self doubleValue] - [aFloat doubleValue]];
}

- (id)objectByMultiplyingByCMSmallInteger:(CMSmallInteger *)anInteger
{
  return [self objectByMultiplyingByCMLargeInteger:[anInteger largeInteger]];
}

- (id)objectByMultiplyingByCMLargeInteger:(CMLargeInteger *)anInteger
{
  long i, j;
  BDIGIT_DBL n = 0;
  CMLargeInteger *z;
  BDIGIT *zds;

  j = digitLength + [anInteger digitLength] + 1;
  z = [CMLargeInteger integerWithDigitLength:j
                      sign:(sign == [anInteger sign])];
  zds = [z digits];
  while (j--) zds[j] = 0;
  for (i = 0; i < digitLength; i++) {
    BDIGIT_DBL dd = digits[i]; 
    if (dd == 0) continue;
    n = 0;
    for (j = 0; j < [anInteger digitLength]; j++) {
      BDIGIT_DBL ee = n + (BDIGIT_DBL)dd * [anInteger digits][j];
      n = zds[i + j] + ee;
      if (ee) zds[i + j] = BIGLO(n);
      n = BIGDN(n);
    }
    if (n) {
      zds[i + j] = n;
    }
  }
  return [z integer];
}

- (id)objectByMultiplyingByCMFloat:(CMFloat *)aFloat
{
  return [CMFloat floatWithDouble:([self doubleValue] * [aFloat doubleValue])];
}

- (id)objectByDividingByCMSmallInteger:(CMSmallInteger *)anInteger
{
  id div;
  [self divideByLargeInteger:[anInteger largeInteger]
        withQuotient:&div
        modulo:nil];
  return [div integer];
}

- (id)objectByDividingByCMLargeInteger:(CMLargeInteger *)anInteger
{
  id div;
  [self divideByLargeInteger:anInteger withQuotient:&div modulo:nil];
  return [div integer];
}

- (void)divideByLargeInteger:(CMLargeInteger *)anInteger
                withQuotient:(CMLargeInteger **)divp
                      modulo:(CMLargeInteger **)modp
{
  CMLargeInteger *mod;

  [self divideByLargeInteger:anInteger withQuotient:divp remainder:&mod];
  if (sign != [anInteger sign] && ![mod isZero]) {
    if (divp) *divp = [*divp objectByAddingCMLargeInteger:PINT2SINT(1)
                             withSign:NO];
    if (modp) *modp = [mod objectByAddingCMLargeInteger:anInteger
                           withSign:YES];
  } else {
    if (divp) *divp = *divp;
    if (modp) *modp = mod;
  }
}

- (void)divideByLargeInteger:(CMLargeInteger *)anInteger
                withQuotient:(CMLargeInteger **)divp
                   remainder:(CMLargeInteger **)modp
{
  long nx = digitLength, ny = [anInteger digitLength];
  long i, j;
  id yy, z;
  BDIGIT *xds, *yds, *zds, *tds;
  BDIGIT_DBL t2;
  BDIGIT_DBL_SIGNED num;
  BDIGIT dd, q;

  if ([anInteger isZero]) [CMException raiseZeroDivideException];
  yds = [anInteger digits];
  if (nx < ny || (nx == ny && digits[nx - 1] < [anInteger digits][ny - 1])) {
    if (divp) *divp = PINT2LINT(0);
    if (modp) *modp = self;
    return;
  }
  xds = digits;
  if (ny == 1) {
    dd = yds[0];
    z = [[self copy] autorelease];
    zds = [z digits];
    t2 = 0; i = nx;
    while (i--) {
      t2 = BIGUP(t2) + zds[i];
      zds[i] = (BDIGIT)(t2 / dd);
      t2 %= dd;
    }
    [z setSign:sign == [anInteger sign]];
    if (modp) {
      *modp = PINT2LINT((unsigned long)t2);
      [(*modp) setSign:sign];
    }
    if (divp) *divp = z;
    return;
  }
  z = [CMLargeInteger integerWithDigitLength:nx==ny?nx+2:nx+1
                      sign:sign == [anInteger sign]];
  zds = [z digits];
  if (nx==ny) zds[nx+1] = 0;
  while (!yds[ny-1]) ny--;
  dd = 0;
  q = yds[ny-1];
  while ((q & (1<<(BITSPERDIG-1))) == 0) {
    q <<= 1;
    dd++;
  }
  if (dd) {
    yy = [[anInteger copy] autorelease];
    tds = [yy digits];
    j = 0;
    t2 = 0;
    while (j<ny) {
      t2 += (BDIGIT_DBL)yds[j]<<dd;
      tds[j++] = BIGLO(t2);
      t2 = BIGDN(t2);
    }
    yds = tds;
    j = 0;
    t2 = 0;
    while (j<nx) {
      t2 += (BDIGIT_DBL)xds[j]<<dd;
      zds[j++] = BIGLO(t2);
      t2 = BIGDN(t2);
    }
    zds[j] = (BDIGIT)t2;
  }
  else {
    zds[nx] = 0;
    j = nx;
    while (j--) zds[j] = xds[j];
  }
  j = nx==ny?nx+1:nx;
  do {
    if (zds[j] ==  yds[ny-1]) q = BIGRAD-1;
    else q = (BDIGIT)((BIGUP(zds[j]) + zds[j-1])/yds[ny-1]);
    if (q) {
      i = 0; num = 0; t2 = 0;
      do {                        /* multiply and subtract */
        BDIGIT_DBL ee;
        t2 += (BDIGIT_DBL)yds[i] * q;
        ee = num - BIGLO(t2);
        num = (BDIGIT_DBL)zds[j - ny + i] + ee;
        if (ee) zds[j - ny + i] = BIGLO(num);
        num = BIGDN(num);
        t2 = BIGDN(t2);
      } while (++i < ny);
      num += zds[j - ny + i] - t2;/* borrow from high digit; don't update */
      while (num) {               /* "add back" required */
        i = 0; num = 0; q--;
        do {
          BDIGIT_DBL ee = num + yds[i];
          num = (BDIGIT_DBL)zds[j - ny + i] + ee;
          if (ee) zds[j - ny + i] = BIGLO(num);
          num = BIGDN(num);
        } while (++i < ny);
        num--;
      }
    }
    zds[j] = q;
  } while (--j >= ny);
  if (divp) {                 /* move quotient down in z */
    *divp = [[z copy] autorelease];
    zds = [*divp digits];
    j = (nx==ny ? nx+2 : nx+1) - ny;
    for (i = 0;i < j;i++) zds[i] = zds[i+ny];
    [*divp setDigitLength:i];
  }
  if (modp) {                 /* normalize remainder */
    *modp = [[z copy] autorelease];
    zds = [*modp digits];
    while (--ny && !zds[ny]); ++ny;
    if (dd) {
      t2 = 0; i = ny;
      while(i--) {
        t2 = (t2 | zds[i]) >> dd;
        q = zds[i];
        zds[i] = BIGLO(t2);
        t2 = BIGUP(q);
      }
    }
    [*modp setDigitLength:ny];
    [*modp setSign:sign];
  }
}

- (id)objectByDividingByCMFloat:(CMFloat *)aFloat
{
  return [CMFloat floatWithDouble:[self doubleValue] / [aFloat doubleValue]];
}

- (id)moduloOfDividedByCMSmallInteger:(CMSmallInteger *)anInteger
{
  return [self moduloOfDividedByCMLargeInteger:[anInteger largeInteger]];
}

- (id)moduloOfDividedByCMLargeInteger:(CMLargeInteger *)anInteger
{
  id mod;
  [self divideByLargeInteger:anInteger withQuotient:nil modulo:&mod];
  return mod;
}

- (id)moduloOfDividedByCMFloat:(CMFloat *)aFloat
{
  return [[self float] moduloOfDividedByCMFloat:aFloat];
}

- (id)objectByRaisingToPowerCMSmallInteger:(CMSmallInteger *)anInteger
{
  long yy;
  id x;

  x = self;
  if ([anInteger isZero]) return PINT2SINT(1);
  yy = [anInteger longValue];
  if (yy > 0) {
    id z = x;

    for (;;) {
      yy -= 1;
      if (yy == 0) break;
      while (yy % 2 == 0) {
        yy /= 2;
        x = [x objectByMultiplyingBy:x];
      }
      z = [z objectByMultiplyingBy:x];
    }
    return [z integer];
  }
  return [CMFloat floatWithDouble:pow([self doubleValue], (double)yy)];
}

- (id)objectByRaisingToPowerCMLargeInteger:(CMLargeInteger *)anInteger
{
  double d;
  if ([anInteger isZero]) return PINT2SINT(1);
  //rb_warn("in a**b, b may be too big");
  d = [anInteger doubleValue];
  return [CMFloat floatWithDouble:pow([self doubleValue], (double)d)];
}

- (id)objectByRaisingToPowerCMFloat:(CMFloat *)aFloat
{
  if ([aFloat isZero]) return PINT2SINT(1);
  return [CMFloat floatWithDouble:pow([self doubleValue], [aFloat doubleValue])];
}

- (id)objectByShiftingRight:(id)anObject
{
  BDIGIT *xds, *zds;
  int shift = [anObject intValue];
  long s1 = shift/BITSPERDIG;
  long s2 = shift%BITSPERDIG;
  CMLargeInteger *x, *z;
  BDIGIT_DBL num = 0;
  long i, j;

  x = self;
  if (shift < 0) return [self objectByShiftingLeft:PINT2SINT(-shift)];
  if (s1 > digitLength) {
    if (sign)
      return PINT2SINT(0);
    else
      return PINT2SINT(-1);
  }
  if (!sign) {
    x = [[self copy] autorelease];
    [self complementWithCarry:YES];
  }
  xds = [x digits];
  i = [x digitLength];
  j = i - s1;
  z = [CMLargeInteger integerWithDigitLength:j sign:[x sign]];
  if (![x sign]) {
    num = ((BDIGIT_DBL)~0) << BITSPERDIG;
  }
  zds = [z digits];
  while (i--, j--) {
    num = (num | xds[i]) >> s2;
    zds[j] = BIGLO(num);
    num = BIGUP(xds[i]);
  }
  if (![x sign]) {
    [x complementWithCarry:NO];
  }
  return [z integer];

}

- (void)complementWithCarry:(BOOL)carry
{
  long i = digitLength;
  BDIGIT *ds = digits;
  BDIGIT_DBL num;

  while (i--) ds[i] = ~ds[i];
  i = 0; num = 1;
  do {
    num += ds[i];
    ds[i++] = BIGLO(num);
    num = BIGDN(num);
  } while (i < digitLength);
  if (!carry) return;
  if ((ds[digitLength-1] & (1<<(BITSPERDIG-1))) == 0) {
    REALLOC_N(digits, BDIGIT, ++digitLength);
    ds = digits;
    ds[digitLength-1] = ~0;
  }
}

- (id)objectByShiftingLeft:(id)anObject
{
  BDIGIT *xds, *zds;
  int shift = [anObject intValue];
  int s1 = shift/BITSPERDIG;
  int s2 = shift%BITSPERDIG;
  CMLargeInteger *z;
  BDIGIT_DBL num = 0;
  long len, i;

  if (shift < 0) return [self objectByShiftingRight:PINT2SINT(-shift)];
  len = digitLength;
  z = [CMLargeInteger integerWithDigitLength:len+s1+1 sign:sign];
  zds = [z digits];
  for (i=0; i<s1; i++) {
    *zds++ = 0;
  }
  xds = digits;
  for (i=0; i<len; i++) {
    num = num | (BDIGIT_DBL)*xds++<<s2;
    *zds++ = BIGLO(num);
    num = BIGDN(num);
  }
  *zds = BIGLO(num);
  return [z integer];
}

- (const void *)objCValueWithType:(const char *)objCType
{
  if (IS_ENCODE_FLOAT(objCType) || IS_ENCODE_DOUBLE(objCType)) {
    static double val;
    val = [self doubleValue];
    return &val;

  } else if (IS_ENCODE_SHORT(objCType) || IS_ENCODE_INT(objCType)) {
    static int val;
    val = (int)[self longValue];
    return &val;

  } else if (IS_ENCODE_USHORT(objCType) || IS_ENCODE_UINT(objCType)) {
    static unsigned int val;
    val = (unsigned int)[self longValue];
    return &val;

  } else if (IS_ENCODE_LONG(objCType) || IS_ENCODE_LONGLONG(objCType)) {
    static long val;
    val = [self longValue];
    return &val;

  } else if (IS_ENCODE_ULONG(objCType) || IS_ENCODE_ULONGLONG(objCType)) {
    static unsigned long val;
    val = [self unsignedLongValue];
    return &val;

  }
  return [super objCValueWithType:objCType];
}

@end
