#import "CMFloat.h"
#import "CMSmallInteger.h"
#import "CMLargeInteger.h"
#import "CMObjCTypeConverter.h"
#import <ctype.h>
#import <math.h>

@implementation CMFloat

+ (id)floatWithString:(NSString *)aString
{
  return [self floatWithDouble:[aString doubleValue]];
}

+ (id)floatWithDouble:(double)aValue
{
  return [[[self alloc] initWithDouble:aValue] autorelease];
}

- (id)initWithDouble:(double)aValue
{
  [self init];
  value = aValue;
  return self;
}

- (NSString *)description
{
  char buf[32];
  char *fmt = "%.15g";
  double avalue, d1, d2;

  if (isinf(value))
    return (value < 0 ? @"-Infinity" : @"Infinity");
  else if(isnan(value))
    return @"NaN";
    
  avalue = fabs(value);
  if (avalue == 0.0) {
    fmt = "%.1f";
  }
  else if (avalue < 1.0e-3) {
    d1 = avalue;
    while (d1 < 1.0) d1 *= 10.0;
    d1 = modf(d1, &d2);
    if (d1 == 0) fmt = "%.1e";
  }    
  else if (avalue >= 1.0e15) {
    d1 = avalue;
    while (d1 > 10.0) d1 /= 10.0;
    d1 = modf(d1, &d2);
    if (d1 == 0) fmt = "%.1e";
    else fmt = "%.16e";
  }    
  else if ((d1 = modf(value, &d2)) == 0) {
    fmt = "%.1f";
  }
  sprintf(buf, fmt, value);

  return [NSString stringWithCString:buf];
}

- (float)floatValue { return (float)value; }
- (double)doubleValue { return value; }
- (long)longValue { return (long)value; }

- (CMInteger *)integer
{
  return [CMSmallInteger integerWithLong:(long)value];;
}

- (CMFloat *)float
{
  return self;
}

- (unsigned)hash
{
  char *c;
  int i, hash;

  if (value == 0) value = fabs(value);
  c = (char*)&value;
  for (hash=0, i=0; i<sizeof(double);i++) {
    hash += c[i] * 971;
  }
  if (hash < 0) hash = -hash;

  return (unsigned)hash;
}

- (const void *)objCValueWithType:(const char *)objCType
{
  if (IS_ENCODE_FLOAT(objCType)) {
    static float val;
    val = (float)value;
    return &val;

  } else if (IS_ENCODE_DOUBLE(objCType)) {
    static double val;
    val = value;
    return &val;

  } else if (IS_ENCODE_SHORT(objCType) || IS_ENCODE_INT(objCType)) {
    static int val;
    val = (int)value;
    return &val;

  } else if (IS_ENCODE_USHORT(objCType) || IS_ENCODE_UINT(objCType)) {
    static unsigned int val;
    val = (unsigned int)value;
    return &val;

  } else if (IS_ENCODE_LONG(objCType) || IS_ENCODE_LONGLONG(objCType)) {
    static long val;
    val = (long)value;
    return &val;

  } else if (IS_ENCODE_ULONG(objCType) || IS_ENCODE_ULONGLONG(objCType)) {
    static unsigned long val;
    val = (unsigned long)value;
    return &val;

  }
  return [super objCValueWithType:objCType];
}

- (BOOL)isZero
{
  return value == 0;
}

- (BOOL)isFinite
{
  return finite(value) ? YES : NO;
}

- (BOOL)isInfinite
{
  return isinf(value) ? YES : NO;
}

- (BOOL)isNaN
{
  return isnan(value) ? YES : NO;
}

- (NSComparisonResult)compareCMSmallInteger:(CMSmallInteger *)anInteger
{
  long a, b;

  a = (long)value;
  b = [anInteger longValue];
  if (a == b) return NSOrderedSame;
  if (a > b) return NSOrderedDescending;
  return NSOrderedAscending;
}

- (NSComparisonResult)compareCMLargeInteger:(CMLargeInteger *)anInteger
{
  return [self compareCMFloat:[anInteger float]];
}

- (NSComparisonResult)compareCMFloat:(CMFloat *)aFloat
{
  double a, b;

  a = value;
  b = [aFloat doubleValue];
  if (a == b) return NSOrderedSame;
  if (a > b) return NSOrderedDescending;
  return NSOrderedAscending;
}

- (id)objectByAddingCMSmallInteger:(CMSmallInteger *)anInteger
{
  return [CMFloat floatWithDouble:value + [anInteger longValue]];
}

- (id)objectByAddingCMLargeInteger:(CMLargeInteger *)anInteger
{
  return [CMFloat floatWithDouble:value + [anInteger doubleValue]];
}

- (id)objectByAddingCMFloat:(CMFloat *)aFloat
{
  return [CMFloat floatWithDouble:value + [aFloat doubleValue]];
}

- (id)objectBySubtractingCMSmallInteger:(CMSmallInteger *)anInteger
{
  return [CMFloat floatWithDouble:value - [anInteger longValue]];
}

- (id)objectBySubtractingCMLargeInteger:(CMLargeInteger *)anInteger
{
  return [CMFloat floatWithDouble:value - [anInteger doubleValue]];
}

- (id)objectBySubtractingCMFloat:(CMFloat *)aFloat
{
  return [CMFloat floatWithDouble:value - [aFloat doubleValue]];
}

- (id)objectByMultiplyingByCMSmallInteger:(CMSmallInteger *)anInteger
{
  return [CMFloat floatWithDouble:value * [anInteger longValue]];
}

- (id)objectByMultiplyingByCMLargeInteger:(CMLargeInteger *)anInteger
{
  return [CMFloat floatWithDouble:value * [anInteger doubleValue]];
}

- (id)objectByMultiplyingByCMFloat:(CMFloat *)aFloat
{
  return [CMFloat floatWithDouble:value * [aFloat doubleValue]];
}

- (id)objectByDividingByCMSmallInteger:(CMSmallInteger *)anInteger
{
  return [CMFloat floatWithDouble:value / [anInteger longValue]];
}

- (id)objectByDividingByCMLargeInteger:(CMLargeInteger *)anInteger
{
  return [CMFloat floatWithDouble:value / [anInteger doubleValue]];
}

- (id)objectByDividingByCMFloat:(CMFloat *)aFloat
{
  return [CMFloat floatWithDouble:value / [aFloat doubleValue]];
}

- (id)moduloOfDividedByCMSmallInteger:(CMSmallInteger *)anInteger
{
  double mod;
  [self divideByDouble:(double)[anInteger longValue]
        withQuotient:0
        modulo:&mod];
  return [CMFloat floatWithDouble:mod];
}

- (id)moduloOfDividedByCMLargeInteger:(CMLargeInteger *)anInteger
{
  double mod;
  [self divideByDouble:[anInteger doubleValue]
        withQuotient:0
        modulo:&mod];
  return [CMFloat floatWithDouble:mod];
}

- (id)moduloOfDividedByCMFloat:(CMFloat *)aFloat
{
  double mod;
  [self divideByDouble:[aFloat doubleValue]
        withQuotient:0
        modulo:&mod];
  return [CMFloat floatWithDouble:mod];
}

- (void)divideByDouble:(double)aValue
          withQuotient:(double *)divp
                modulo:(double *)modp
{
  double div, mod;

  mod = fmod(value, aValue);
  div = (value - mod) / aValue;
  if (aValue * mod < 0) {
    mod += aValue;
    div -= 1.0;
  }
  if (modp) *modp = mod;
  if (divp) *divp = div;
}

- (id)objectByRaisingToPowerCMSmallInteger:(CMSmallInteger *)anInteger
{
  return [CMFloat floatWithDouble:pow(value, [anInteger longValue])];
}

- (id)objectByRaisingToPowerCMLargeInteger:(CMLargeInteger *)anInteger
{
  return [CMFloat floatWithDouble:pow(value, [anInteger doubleValue])];
}

- (id)objectByRaisingToPowerCMFloat:(CMFloat *)aFloat
{
  return [CMFloat floatWithDouble:pow(value, [aFloat doubleValue])];
}

@end
