#import "common.h"
#import "CMEvaluateNodeVisitor.h"
#import "CMNode.h"
#import "CMClassNode.h"
#import "CMVariableNode.h"
#import "CMMethodNode.h"
#import "CMSelectorNode.h"
#import "CMRegularExpressionNode.h"
#import "CMMessageNode.h"
#import "CMBlockNode.h"
#import "CMBlockArgumentsNode.h"
#import "CMExpressionNode.h"
#import "CMSubstituteNode.h"
#import "CMVariableDeclarationNode.h"
#import "CMStringNode.h"
#import "CMArrayNode.h"
#import "CMDictionaryNode.h"
#import "CMReturnNode.h"
#import "CMDigitNode.h"
#import "CMClass.h"
#import "CMVariable.h"
#import "CMMethod.h"
#import "CMObject.h"
#import "CMBlock.h"
#import "CMInvocation.h"
#import "CMNumber.h"
#import "CMNil.h"
#import "CMSelector.h"
#import "CMException.h"
#import "CMSuperObject.h"
#import "CMTrue.h"
#import "CMFalse.h"
#import "CMContext.h"
#import "CMBlockContext.h"
#import "CMApplication.h"
#import "CMEmbeddedObjects.h"
#import "CMRuntime.h"
#import "CMKeyPathNode.h"
#import "CMKeyValueCodingNode.h"
#import <OgreKit/OgreKit.h>


@implementation CMEvaluateNodeVisitor

- (void)visitClassNode:(CMClassNode *)aNode
{
}

- (void)visitVariableNode:(CMVariableNode *)aNode
{
  id var = nil, value = nil, class = nil, num = nil, em = nil;

  if ([aNode isSelf]) {
    var = [CMVariable selfVariableWithObject:[context methodReceiver]];

  } else if ([aNode isSuper]) {
    var = [CMVariable superVariableWithObject:
                        [CMSuperObject superObjectWithObject:
                                         [context methodReceiver]]];

  } else if ([aNode isThisContext]) {
    var = [CMVariable thisContextVariableWithObject:context];

  } else if (class = NSClassFromString([aNode name])) {
    var = [CMVariable variableWithName:[aNode name] object:class];

  } else if ([aNode isTrue]) {
    var = [CMVariable trueVariable];

  } else if ([aNode isFalse]) {
    var = [CMVariable falseVariable];

  } else if ([aNode isNil]) {
    var = [CMVariable nilVariable];

  } else if ([aNode isFileName]) {
    var = [CMVariable fileNameVariableWithObject:[aNode fileName]];

  } else if ([aNode isLineNumber]) {
    num = PLNG2SINT((long)[aNode lineNumber]);
    var = [CMVariable lineNumberVariableWithObject:num];

  } else if (em = [CMEmbeddedObjects embeddedObjectForName:[aNode name]]) {
    var = [CMVariable variableWithName:[aNode name] object:em];

  } else {
    var = [context variableNamed:[aNode name]];
    if (!var)
      [CMException raiseUndefinedVariableExceptionWithNode:aNode name:[aNode name]];
  }

  value = [var object];
  if (!value)
    value = [CMNil defaultNil];
  LOG(@"* retrieve value: %@ for: %@", [value description], [aNode name]);
  [context setReturnVariable:var];
  [context setReturnValue:value];
}

- (void)visitMethodNode:(CMMethodNode *)aNode
{
  LOG(@"* invoke method %@ with %@", [aNode name], [[context arguments] description]);
  [aNode nodesAcceptVisitor:self];
}

- (void)visitMessageNode:(CMMessageNode *)aNode
{
  BOOL isSelectorException = NO, isSuper = NO, isAlloc;
  Class originalClass = nil;
  NSMethodSignature *sig;
  SEL sel;
  id expr, target, argNode;
  CMInvocation *invocation;
  NSString *selString;
  int i;

  if ([context isSuspended]) {
    if ([[context resumePoint] isEqual:aNode]) {
      [context setReturnValue:[context objectToResume]];
      [context setIsSuspended:NO];
    }
    return;
  }

  NS_DURING
    expr = [aNode expression];
    [expr acceptVisitor:self];

    sel = [aNode selector];
    target = [context returnValue];
    if (!target) {
      return;
    } else if ([target isMemberOfClass:[CMNil class]] &&
               ![CMNil scriptRespondsToSelector:sel]) {
      return;
    } else if ([target isMemberOfClass:[CMSuperObject class]]) {
      target = [context receiver];
      originalClass = [target class];
      target->isa = [target superclass];
      isSuper = YES;
    }

    sig = [target methodSignatureForSelector:sel];
    if (!sig) {
      LOG(@"*** selector is not recognized - [%@ %@]", [target class], NSStringFromSelector(sel));
      isSelectorException = YES;
      [CMException raiseSelectorNotRecognizedExceptionWithNode:aNode
                   object:target selector:sel];
    }

    invocation = [CMInvocation invocationWithMethodSignature:sig];
    [invocation setTarget:target];
    [invocation setSelector:sel];

    // set arguments
    if (![aNode isSingleMessage]) {
      NSArray *argNodes = [aNode argumentNodes];
      for (i = 0; i < [argNodes count]; i++) {
        argNode = [argNodes objectAtIndex:i];
        [argNode acceptVisitor:self];
        LOG(@"* set argument:%@ (%@) at:%d", [context returnValue],
              [argNode class], i+2);
        if ([argNode isMemberOfClass:[CMVariableNode class]]) {
          [invocation setArgumentToVariable:[context returnVariable] atIndex:(i+2)];
        } else
          [invocation setArgumentToObject:[context returnValue] atIndex:(i+2)];
      }
    }

    selString = NSStringFromSelector(sel);
    isAlloc = [selString isEqualToString:NSStringFromSelector(@selector(alloc))];
    if (isAlloc) {
      LOG(@"* invoke +[%@ alloc]", [target description]);
      [context setReturnValue:[target alloc]];
      LOG(@"* +[%@ alloc] returns an instance", [target description]);
    } else {
      LOG(@"* invoke [%@ %@]", [target class], selString);
      [invocation invoke];
      if ([context continuesVisitation])
        [context setReturnValue:[invocation returnValue]];
      LOG(@"* [%@ %@] returns: %@", [target class], selString,
           [[invocation returnValue] description]);
    }
    if (isSuper)
      target->isa = originalClass;

  NS_HANDLER
    LOG(@"%@ (%@)", [localException description],
        [[localException userInfo] description]);
    [context setResumePoint:aNode];
    NSMutableDictionary *info = [aNode userInfoWithMethodNode:[aNode methodNode]];
    id exceptions = [[localException userInfo] objectForKey:CMExceptionsKey];
    if (!exceptions) {
      exceptions = [NSMutableArray arrayWithCapacity:1];
      [info setObject:exceptions forKey:CMExceptionsKey];
      localException = [NSException exceptionWithName:[localException name]
                                    reason:[localException reason]
                                    userInfo:info];
    } else if (!isSelectorException) {
      NSException *ex = [NSException exceptionWithName:[localException name]
                                     reason:[localException reason]
                                     userInfo:info];
      [exceptions addObject:ex];
    }
    [localException raise];
  NS_ENDHANDLER
}

- (void)visitSubstituteNode:(CMSubstituteNode *)aNode
{
  id varNode, exprNode, var, value;

  varNode = [[aNode nodes] objectAtIndex:0];
  exprNode = [[aNode nodes] objectAtIndex:1];

  [exprNode acceptVisitor:self];
  value = [context returnValue];
  if ([context isSuspended]) return;

  // key-value coding
  if ([varNode isMemberOfClass:[CMKeyValueCodingNode class]]) {
    LOG(@"* substitute setValue:forKeyPath:");
    [[varNode expression] acceptVisitor:self];
    [[context returnValue] setValue:value forKeyPath:[varNode keyPath]];
    [context setReturnVariable:nil];
  } else {
    var = [context variableNamed:[varNode name]];
    if (!var)
      [CMException raiseUndefinedVariableExceptionWithNode:aNode name:[varNode name]];
    LOG(@"* substitute '%@' with %@", [var name], [context returnValue]);
    [var setObject:value];
    [context setReturnVariable:var];
  }
  [context setReturnValue:value];
}

- (void)visitExpressionNode:(CMExpressionNode *)aNode
{
  id node, e = [[aNode nodes] objectEnumerator];
  while (node = [e nextObject]) {
    [node acceptVisitor:self];
    if (![context continuesVisitation]) {
      break;
    }
  }
}

- (void)visitReturnNode:(CMReturnNode *)aNode
{
  id expr = [aNode lastNode];
  [expr acceptVisitor:self];
  LOG(@"return value - %@", [[context returnValue] description]);
  [context setContinuesVisitationForMethodContext:NO]; 
}

- (void)visitVariableDeclarationNode:(CMVariableDeclarationNode *)aNode
{
  if ([[aNode parentNode] isMemberOfClass:[CMClassNode class]]) return;
  if ([context isSuspended]) return;

  id e, varNode;
  e = [[aNode nodes] objectEnumerator];
  while (varNode = [e nextObject]) {
    [context addVariableForName:[varNode name]];
    LOG(@"* add variable:%@ to: %@", [varNode name], [context description]);
  }
}

- (void)visitBlockArgumentsNode:(CMBlockArgumentsNode *)aNode
{
  if ([context isSuspended]) return;

  id obj, varNode;
  int i;

  for (i = 0;i < [context numberOfArguments]; i++) {
    obj = [[context arguments] objectAtIndex:i];
    varNode = [[aNode nodes] objectAtIndex:i];
    [context setObject:obj forVariableNamed:[varNode name]];
    LOG(@"* set variable:%@ to: %@ in block", [varNode name], [context description]);
  }
}

- (void)visitStringNode:(CMStringNode *)aNode
{
  NSString *string;
  if ([aNode isMutable])
    string = [NSMutableString stringWithString:[aNode string]];
  else
    string = [NSString stringWithString:[aNode string]];
  [context setReturnValue:string];
  LOG(@"* generate %@: %@", NSStringFromClass([string class]), string);
}

- (void)visitArrayNode:(CMArrayNode *)aNode
{
  id array = [NSMutableArray arrayWithCapacity:[[aNode nodes] count]];
  id node, e = [[aNode nodes] objectEnumerator];
  while (node = [e nextObject]) {
    [node acceptVisitor:self];
    if ([context returnValue])
      [array addObject:[context returnValue]];
  }
  if ([context isSuspended]) return;
  if (![aNode isMutable])
    array = [NSArray arrayWithArray:array];
  [context setReturnValue:array];
  LOG(@"* generate %@: %@", NSStringFromClass([array class]), [array description]);
}

- (void)visitDictionaryNode:(CMDictionaryNode *)aNode
{
  BOOL isKey = YES;
  id dict, e, node;
  NSMutableArray *objects, *keys;

  objects = [NSMutableArray arrayWithCapacity:1];
  keys = [NSMutableArray arrayWithCapacity:1];
  e = [[aNode nodes] objectEnumerator];
  while (node = [e nextObject]) {
    [node acceptVisitor:self];
    if (isKey)
      [keys addObject:[context returnValue]];
    else
      [objects addObject:[context returnValue]];
    isKey = !isKey;
  }
  if ([context isSuspended]) return;
  if ([aNode isMutable])
    dict = [NSMutableDictionary dictionaryWithObjects:objects forKeys:keys];
  else
    dict = [NSDictionary dictionaryWithObjects:objects forKeys:keys];
  [context setReturnValue:dict];
  LOG(@"* generate %@: %@", NSStringFromClass([dict class]), [dict description]);
}

- (void)visitDigitNode:(CMDigitNode *)aNode
{
  CMNumber *digit = [CMNumber numberWithString:[aNode digitString]];
  [context setReturnValue:digit];
  LOG(@"* generate digit: %@", [digit description]);
}

- (void)visitBlockNode:(CMBlockNode *)aNode
{
  LOG(@"* generate block");
  CMBlock *block = [CMBlock blockWithNode:aNode];
  [block setOuterContext:context];
  [context setReturnValue:block];
}

- (void)visitSelectorNode:(CMSelectorNode *)aNode
{
  CMSelector *sel;

  if ([aNode selector] != nil) {
    sel = [CMSelector selectorForSelector:[aNode selector]];
    [context setReturnValue:sel];
    LOG(@"* generate selector <%@>", NSStringFromSelector([aNode selector]));
  }
}

- (void)visitRegularExpressionNode:(CMRegularExpressionNode *)aNode
{
  [aNode parseOptions];
  OGRegularExpression *regexp = [OGRegularExpression
                                  regularExpressionWithString:[aNode string]
                                  options:[aNode options]
                                  syntax:[aNode syntax]
                                  escapeCharacter:[aNode escapeCharacter]];
  [context setReturnValue:regexp];
  LOG(@"* generate regexp %@", [regexp description]);
}

- (void)visitKeyPathNode:(CMKeyPathNode *)aNode {}

- (void)visitKeyValueCodingNode:(CMKeyValueCodingNode *)aNode
{
  [[aNode expression] acceptVisitor:self];
  [context setReturnValue:[[context returnValue] valueForKeyPath:[aNode keyPath]]];
}

@end
