#import "common.h"
#import "CMContext.h"
#import "CMBlockContext.h"
#import "CMMethodContext.h"
#import "CMRuntime.h"
#import "CMClass.h"
#import "CMObject.h"
#import "CMBlock.h"
#import "CMVariable.h"
#import "CMMethod.h"
#import "CMNode.h"
#import "CMEvaluateNodeVisitor.h"

#define DefaultLineNumber 1
#define DefaultFileName   @"-"

@implementation CMContext

+ (id)sender
{
  return [[[NSThread currentThread] threadDictionary] objectForKey:CMSenderKey];
}

+ (void)setSender:(id)aSender
{
  id parentSender;
  if (parentSender = [[[NSThread currentThread] threadDictionary]
                       objectForKey:CMSenderKey]) {
    if (![parentSender isEqual:aSender]) {
      [aSender setSender:parentSender];
      [aSender setResumePoint:[parentSender resumePoint]];
    }
  }
  [[[NSThread currentThread] threadDictionary] setObject:aSender forKey:CMSenderKey];
}

+ (void)revertSender
{
  id parent;
  if (parent = [[self sender] sender])
    [[[NSThread currentThread] threadDictionary] setObject:parent
                                                 forKey:CMSenderKey];
}

+ (id)setInstanceMethodSenderWithReceiver:(id)aReceiver
                                arguments:(NSArray *)arguments
                                 selector:(SEL)aSelector
                                 forClass:(Class)aClass
{
  return [self setSenderWithReceiver:aReceiver
               arguments:arguments
               selector:aSelector
               forClass:aClass
               isInstanceMethod:YES];
}

+ (id)setClassMethodSenderWithReceiver:(id)aReceiver
                             arguments:(NSArray *)arguments
                              selector:(SEL)aSelector
                              forClass:(Class)aClass
{
  return [self setSenderWithReceiver:aReceiver
               arguments:arguments
               selector:aSelector
               forClass:aClass
               isInstanceMethod:NO];
}

+ (id)setSenderWithReceiver:(id)aReceiver
                  arguments:(NSArray *)arguments
                   selector:(SEL)aSelector
                   forClass:(Class)aClass
           isInstanceMethod:(BOOL)flag
{
  CMClass *class;
  NSString *methodName;
  CMMethod *method;
  CMMethodContext *methodContext;

  class = [CMClass classForClass:aClass];
  methodName = NSStringFromSelector(aSelector);
  if (flag) {
    if (!(method = [class instanceMethodNamed:methodName])) {
      method = [CMMethod methodWithName:methodName
                         selector:aSelector
                         forObject:aReceiver];
      [class addInstanceMethod:method];
    }
  } else {
    if (!(method = [class classMethodNamed:methodName])) {
      method = [CMMethod methodWithName:methodName
                         selector:aSelector
                         forObject:aReceiver];
      [class addClassMethod:method];
    }
  }
  methodContext = [CMMethodContext contextWithSender:[self sender]
                                   receiver:aReceiver
                                   method:method
                                   arguments:arguments];
  [self setSender:methodContext];
  return method;
}

+ (void)flushSender
{
  [[[NSThread currentThread] threadDictionary] removeObjectForKey:CMSenderKey];
}

- (void)registerSender
{
  sender = [[CMContext sender] retain];
  [CMContext setSender:self];
}

+ (id)contextWithSender:(id)aSender
               receiver:(id)aReceiver
                 method:(id)aMethod
              arguments:(NSArray *)arguments
{
  return [[[self alloc] initWithSender:aSender
                        receiver:aReceiver
                        method:aMethod
                        arguments:arguments] autorelease];
}

- (id)initWithSender:(id)aSender
            receiver:(id)aReceiver
              method:(id)aMethod
           arguments:(NSArray *)arguments
{
  [self init];
  sender = [aSender retain];
  receiver = [aReceiver retain];
  method = [aMethod retain];
  args = [arguments retain];
  if (aMethod) {
    int i;
    for (i = 0; i < [aMethod numberOfArguments]; i++) {
      [self setObject:[arguments objectAtIndex:i]
            forVariableNamed:[[aMethod argumentNames] objectAtIndex:i]];
      LOG(@"* set method arg: %@ for: %@", 
          [[arguments objectAtIndex:i] description],
          [[aMethod argumentNames] objectAtIndex:i]);
    }
  }

  return self;
}

- (id)init
{
  [super init];
  currentLineNumber = DefaultLineNumber;
  currentFileName = DefaultFileName;
  continuesVisitation = YES;
  passesException = NO;
  sender = nil;
  receiver = nil;
  args = nil;
  returnVariable = nil;
  returnValue = nil;
  return self;
}

- (void)dealloc
{
  [sender release];
  [receiver release];
  [returnVariable release];
  [returnValue release];
  [args release];
  [visitor release];
  [resumePoint release];

  // leak bug
  //[method release];

  [super dealloc];
}

- (id)sender                              { return sender; }
- (void)setSender:(id)anSender            { ASSIGN(sender, anSender); }
- (id)receiver                            { return receiver; }
- (void)setReceiver:(id)anReceiver        { ASSIGN(receiver, anReceiver); }
- (NSArray *)arguments                    { return args; }
- (void)setArguments:(NSArray *)objects   { ASSIGN(args, objects); }
- (CMMethod *)method                      { return method; }
- (id)returnValue                         { return returnValue; }
- (void)setReturnValue:(id)aValue         { ASSIGN(returnValue, aValue); }
- (CMVariable *)returnVariable            { return returnVariable; }
- (void)setReturnVariable:(id)aVariable   { ASSIGN(returnVariable, aVariable); }
- (BOOL)continuesVisitation               { return continuesVisitation; }
- (void)setContinuesVisitation:(BOOL)flag { continuesVisitation = flag; }
- (BOOL)passesException                   { return passesException; }
- (void)setPassesException:(BOOL)flag     { passesException = flag; }
- (BOOL)doesRetry                         { return doesRetry; }
- (void)setDoesRetry:(BOOL)flag           { doesRetry = flag; }
- (CMBlock *)blockToRetry                 { return blockToRetry; }
- (void)setBlockToRetry:(CMBlock *)aBlock { ASSIGN(blockToRetry, aBlock); }
- (BOOL)isSuspended                       { return isSuspended; }
- (void)setIsSuspended:(BOOL)flag         { isSuspended = flag; }
- (id)objectToResume                      { return objectToResume; }
- (void)setObjectToResume:(id)anObject    { ASSIGN(objectToResume, anObject); }

- (void)clearFlags
{
  continuesVisitation = YES;
  passesException = NO;
  doesRetry = NO;
  isSuspended = NO;
  [self setReturnValue:nil];
  [self setReturnVariable:nil];
}

- (void)setContinuesVisitationForMethodContext:(BOOL)flag
{
  id parentSender = self;
  while (parentSender) {
    [parentSender setContinuesVisitation:flag];
    if ([parentSender isMemberOfClass:[CMMethodContext class]]) {
      [parentSender setReturnValue:returnValue];
      break;
    }
    parentSender = [parentSender sender];
  }
}

- (id)methodReceiver
{
  id parentSender = self;
  while (parentSender) {
    if ([parentSender isMemberOfClass:[CMMethodContext class]]) {
      return [parentSender receiver];
    }
    parentSender = [parentSender sender];
  }
  return nil;
}

- (unsigned int)numberOfArguments
{ 
  return [args count];
}

- (id)metaObject
{
  if ([receiver isKindOfClass:[NSObject class]])
    return [CMObject objectForObject:receiver];
  else
    return [CMClass classForClass:receiver];
}

- (NSString *)currentFileName { return currentFileName; }

- (void)setCurrentFileName:(NSString *)aName
{
  [currentFileName release];
  currentFileName = [[NSString alloc] initWithString:aName];
}

- (int)currentLineNumber { return currentLineNumber; }

- (void)setCurrentLineNumber:(int)number
{
  currentLineNumber = number;
}

- (void)incrementCurrentLineNumber { currentLineNumber++; }

- (void)incrementCurrentLineNumberWithLength:(unsigned int)length
{
  preLineNumber = currentLineNumber;
  currentLineNumber = currentLineNumber + (int)length;
}

- (void)decrementCurrentLineNumber { currentLineNumber = preLineNumber; }
- (void)clearCurrentLineNumber { currentLineNumber = DefaultLineNumber; }

- (void)displayException:(NSException *)anException
{
  id str, e = [[self stacktracesWithException:anException] objectEnumerator];
  while (str = [e nextObject])
    printf("%s\n", [str cString]);
}

- (NSArray *)stacktracesWithException:(NSException *)anException
{
  NSMutableArray *stacks = [NSMutableArray arrayWithCapacity:1];
  NSString *file, *line, *methodName;
  id e;
  NSException *ex;

  file = [[anException userInfo] objectForKey:CMFileNameKey];
  line = [[anException userInfo] objectForKey:CMLineNumberKey];
  methodName = [[anException userInfo] objectForKey:CMMethodNameKey];
  [stacks addObject:[CMContext errorStringWithFileName:file
                               line:[line intValue]
                               methodName:methodName
                               exceptionName:[anException name]
                               exceptionReason:[anException reason]]];

  e = [[[anException userInfo] objectForKey:CMExceptionsKey] objectEnumerator];
  while (ex = [e nextObject]) {
    file = [[ex userInfo] objectForKey:CMFileNameKey];
    line = [[ex userInfo] objectForKey:CMLineNumberKey];
    methodName = [[ex userInfo] objectForKey:CMMethodNameKey];
    [stacks addObject:[CMContext errorStringWithFileName:file
                                 line:[line intValue]
                                 methodName:methodName]];
  }

  return stacks;
}

// (file:line) [method] <ex> msg
+ (NSString *)errorStringWithFileName:(NSString *)fileName
                                 line:(int)lineNumber
                           methodName:(NSString *)methodName
                        exceptionName:(NSString *)exceptionName
                      exceptionReason:(NSString *)exceptionReason
{
  if (!methodName) methodName = @"";
  return [NSString stringWithFormat:@"(%@:%d) %@ <%@> %@",
                   fileName, lineNumber, methodName,
                   exceptionName, exceptionReason];
}

// ... (file:line) [method]
+ (NSString *)errorStringWithFileName:(NSString *)fileName
                                 line:(int)lineNumber
                           methodName:(NSString *)methodName
{
  if (!methodName) methodName = @"";
  return [NSString stringWithFormat:@"... (%@:%d) %@",
                   fileName, lineNumber, methodName];
}

- (void)raiseSyntaxException
{
  NSException *ex;
  NSMutableDictionary *info;

  info = [NSMutableDictionary dictionaryWithCapacity:1];
  [info setObject:currentFileName forKey:CMFileNameKey];
  [info setObject:[NSNumber numberWithInt:currentLineNumber]
        forKey:CMLineNumberKey];
  ex = [NSException exceptionWithName:CMSyntaxExceptionName
                    reason:CMSyntaxExceptionReason
                    userInfo:info];
  [ex raise];

}

- (NSMutableDictionary *)userInfo
{
  NSMutableDictionary *info;

  info = [NSMutableDictionary dictionaryWithCapacity:1];
  [info setObject:currentFileName forKey:CMFileNameKey];
  [info setObject:[NSNumber numberWithInt:currentLineNumber]
        forKey:CMLineNumberKey];
  return info;
}

- (NSMutableDictionary *)preLineUserInfo
{
  NSMutableDictionary *info;

  info = [NSMutableDictionary dictionaryWithCapacity:1];
  [info setObject:currentFileName forKey:CMFileNameKey];
  [info setObject:[NSNumber numberWithInt:preLineNumber]
        forKey:CMLineNumberKey];
  return info;
}

- (void)evaluate
{
  visitor = [[CMEvaluateNodeVisitor visitorWithContext:self] retain];
  [node nodesAcceptVisitor:visitor];
}

- (void)stop
{
  isSuspended = YES;
}

- (void)resume
{
  continuesVisitation = YES;  
  [node nodesAcceptVisitor:visitor];
  isSuspended = NO;
}

- (CMNode *)node { return node; }
- (void)setNode:(CMNode *)aNode { ASSIGN(node, aNode); }
- (CMNode *)resumePoint { return resumePoint; }
- (void)setResumePoint:(CMNode *)aNode { ASSIGN(resumePoint, aNode); }

@end
