#import "common.h"
#import "CMApplication.h"
#import "CMEmbeddedObjects.h"
#import "CMContext.h"
#import "CMRuntime.h"
#import "CMRange.h"
#import "CMSize.h"
#import "CMPoint.h"
#import "CMRect.h"
#import "CMInvocation.h"
#import "CMEnvironment.h"
#import "CMException.h"
#import "CMLexer.h"
#import "CMParser.h"
#import "CMNode.h"
#import "CMLoadNodeVisitor.h"
#import "CMEvaluateNodeVisitor.h"
#import <unistd.h> 
#import <sys/types.h>

NSString *CMVersion                    = @"0.1";
NSString *CMReleaseDate                = @"$Date: 2005/06/13 18:50:42 $";
NSString *CMPathExtension              = @"cm";
NSString *CMEnvironmentLibraryKey      = @"MILLLIB";
NSString *CMApplicationVariableName    = @"mill";
NSString *NSApplicationVariableName    = @"app";
NSString *CMFileNameVariableName       = @"__file__";
NSString *CMLineNumberVariableName     = @"__line__";
NSString *CMArgumentsVariableName      = @"argv";
NSString *CMStandardInputVariableName  = @"stdin";
NSString *CMStandardOutputVariableName = @"stdout";
NSString *CMStandardErrorVariableName  = @"stderr";
NSString *CMEnvironmentVariableName    = @"env";
NSString *CMDefaultFileName            = @"-";
NSString *CMLibraryDirectoryName       = @"Library/CocoaMill";
NSString *CMSiteDirectoryName          = @"Site";

static id defaultApplication = nil;
static NSStringEncoding defaultEncoding = NSShiftJISStringEncoding;
static int uid, euid, gid, egid;


int CMApplicationMain(const char *cm_main_name, int argc, const char* argv[])
{
  CMApplication *app;
  NSString *path, *resourcePath, *subpath, *mainName;
  NSArray *subpaths;
  id e;
  NSFileManager *fm;
  NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];

  fm = [NSFileManager defaultManager];
  app = [CMApplication defaultApplication];
  resourcePath = [[NSBundle mainBundle] resourcePath];
  mainName = [NSString stringWithCString:cm_main_name];
  path = [resourcePath stringByAppendingPathComponent:mainName];
  if ([fm fileExistsAtPath:path]) {
    [[CMRuntime defaultRuntime] loadWithContentsOfFile:path];
    LOG(@"* load main script: %@", path);
  }

  subpaths = [[NSFileManager defaultManager] subpathsAtPath:resourcePath];
  e = [subpaths objectEnumerator];
  while (subpath = [e nextObject]) {
    if ([[subpath pathExtension] isEqualToString:[app pathExtension]]) {
      path = [resourcePath stringByAppendingPathComponent:subpath];
      [[CMRuntime defaultRuntime] loadWithContentsOfFile:path];
    }
  }
  [pool release];

  return NSApplicationMain(argc, argv);
}


@implementation CMApplication

+ (void)initialize
{
  uid = (int)getuid();
  euid = (int)geteuid();
  gid = (int)getgid();
  egid = (int)getegid();
}

+ (id)defaultApplication
{
  if (!defaultApplication)
    defaultApplication = [[self alloc] init];
  return defaultApplication;
}

+ (NSStringEncoding)defaultEncoding
{
  return defaultEncoding;
}

+ (void)setDefaultEncoding:(NSStringEncoding)anEncoding
{
  defaultEncoding = anEncoding;
}

- (id)init
{
  [super init];
  arguments = [[NSMutableArray alloc] initWithCapacity:1];
  encoding = [[self class] defaultEncoding];
  environment = [[CMEnvironment alloc] init];
  checkSytaxOnly = NO;
  debug = NO;
  return self;
}

- (void)dealloc
{
  [runtime release];
  [arguments release];
  [environment release];
  [super dealloc];
}

static void forbid_setid( const char *s )
{
  if (euid != uid)
    [NSException raise:CMSecurityException
                 format:@"No %s allowed while running setuid", s];
  if (egid != gid)
    [NSException raise:CMSecurityException
                 format:@"No %s allowed while running setgid", s];
}

- (void)runWithArguments:(const char **)argv count:(int)argc
{
  char **origargv = (char **)argv;
  char *argv0 = (char *)argv[0];
  const char *s, *script;
  int version = 0;
  int verbose = 0;
  int i;
  NSMutableString *e_script = nil;
  NSString *loadScript = nil;

  [arguments removeAllObjects];
  [CMRuntime defaultRuntime];

  if (argc == 0) return;

  for (argc--,argv++; argc > 0; argc--,argv++) {
    if (argv[0][0] != '-' || !argv[0][1]) break;

    s = argv[0]+1;
  reswitch:
    switch (*s) {
	  case 'h':
	    [self displayHelpWithName:origargv[0]];
	    exit(0);

	  case 'I':
	    forbid_setid("-I");
	    if (*++s) {
        [[runtime loadPaths]
          insertObject:[NSString stringWithCString:s] atIndex:0];
      } else if (argv[1]) {
        [[runtime loadPaths]
          insertObject:[NSString stringWithCString:argv[1]] atIndex:0];
        argc--,argv++;
	    }
	    break;

	  case 'e':
	    forbid_setid("-e");
	    if (!*++s) {
        s = argv[1];
        argc--,argv++;
	    }
	    if (!s) {
        fprintf(stderr, "%s: no code specified for -e\n", origargv[0]);
        exit(2);
	    }
	    if (!e_script) {
        e_script = [NSMutableString stringWithCapacity:1];
        if (script == 0) script = "-e";
	    }
      [runtime setLoadingFileName:@"-e"];
      [runtime setCurrentFileName:@"-e"];
	    [e_script appendString:[NSString stringWithCString:s]];
	    [e_script appendString:@"\n"];
	    break;

	  case 'l':
	    forbid_setid("-l");
	    if (*++s) {
        [runtime loadWithPath:[NSString stringWithCString:s]];
	    } else if (argv[1]) {
        [runtime loadWithPath:[NSString stringWithCString:argv[1]]];
        argc--,argv++;
	    }
	    break;

	  case '-':
	    if (!s[1] || (s[1] == '\r' && !s[2])) {
        argc--,argv++;
        goto switch_end;
	    }
	    s++;
      if (strcmp("version", s) == 0) {
        version = 1;
	    } else if (strcmp("verbose", s) == 0) {
        verbose = 1;
        // isVerboseMode = YES;
	    } else if (strcmp("help", s) == 0) {
        [self displayHelpWithName:origargv[0]];
        exit(0);
	    }
	    else {
        fprintf(stderr, "%s: invalid option --%s  (-h will show valid options)\n",
                origargv[0], s);
        exit(2);
	    }
      [runtime setLoadingFileName:@"-"];
      [runtime setCurrentFileName:@"-"];
	    break;

	  default:
	    fprintf(stderr, "%s: invalid option -%c  (-h will show valid options)\n",
              origargv[0], *s);
	    exit(2);

	  case 0:
	    break;
    }
  }

 switch_end:
  if (argv0 == 0) return;

  if (version) {
    [self displayVersion];
    exit(0);
  }
  if (!e_script) {
    if (argc == 0) {	/* no more args */
	    if (verbose) exit(0);
	    script = "-";
    } else {
	    script = argv[0];
	    if (script[0] == '\0') {
        script = "-";
	    }
	    argc--; argv++;
    }
  }

  for (i = 0; i < argc; i++)
    [arguments addObject:[NSString stringWithCString:argv[i]]];

  loadScript = [NSString stringWithCString:script];
  [runtime setLoadingFileName:loadScript];
  [runtime setCurrentFileName:loadScript];
  if (e_script) {
    [runtime loadWithString:e_script];
  } else if (strlen(script) == 1 && script[0] == '-') {
    [runtime loadFromStandardInput];
  } else {
    [runtime loadWithContentsOfFile:loadScript];
  }
}

- (void)displayVersion
{
  printf("CocoaMill %s (%s)\n", [CMVersion cString], [CMReleaseDate cString]);
  printf("Copyright (C) 2005- SUZUKI Tetsuya\n");
}

- (void)displayHelpWithName:(const char *)aName
{
  static char *usage_msg[] = {
    //"-c              check syntax only",
    // "-d              set debugging flags (set $DEBUG to true)",
    "-e 'command'    one line of script. Several -e's allowed. Omit [programfile]",
    "-Idirectory     specify load path directory (may be used more than once)",
    //"-Eencoding      specifies encoding",
    "-llibrary       load the library, before executing your script",
    // "-v              print version number, then turn on verbose mode",
    // "-w              turn warnings on for your script",
    // "-W[level]       set warning level; 0=silence, 1=medium, 2=verbose (default)",
    // "-x[directory]   strip off text before #!ruby line and perhaps cd to directory",
    "--version       print the version",
    NULL
  };
  char **p = usage_msg;

  printf("Usage: %s [options] [file] [arguments]\n", aName);
  while (*p)
    printf("  %s\n", *p++);
}

- (void)initializeRuntime:(CMRuntime *)aRuntime
{
  runtime = [aRuntime retain];
  [self setVariablesForContext:aRuntime];
}

- (NSString *)pathExtension     { return CMPathExtension; }
- (NSString *)version           { return CMVersion; }
- (NSStringEncoding)encoding    { return encoding; }
- (NSArray *)arguments          { return arguments; }
- (CMEnvironment *)environment  { return environment; }
- (BOOL)debug                   { return debug; }
- (void)setDebug:(BOOL)flag     { debug = flag; }
- (BOOL)terminationStatus       { return terminationStatus; }
- (NSString *)fullUserName      { return NSFullUserName(); }
- (NSString *)userName          { return NSUserName(); }
- (NSString *)homeDirectoryPath { return NSHomeDirectory(); }

- (NSString *)homeDirectoryPathForUser:(NSString *)aName
{
  return NSHomeDirectoryForUser(aName);
}

- (NSString *)currentDirectoryPath
{
  return [[NSFileManager defaultManager] currentDirectoryPath];
}

- (NSString *)userLibrarySiteVersionPath
{
  NSArray *paths = [NSArray arrayWithObjects:NSHomeDirectory(),
                           CMLibraryDirectoryName, CMSiteDirectoryName, CMVersion, nil];
  return [NSString pathWithComponents:paths];
}

- (NSString *)userLibrarySitePath
{
  NSArray *paths = [NSArray arrayWithObjects:NSHomeDirectory(),
                           CMLibraryDirectoryName, CMSiteDirectoryName, nil];
  return [NSString pathWithComponents:paths];
}

- (NSString *)userLibraryVersionPath
{
  NSArray *paths = [NSArray arrayWithObjects:NSHomeDirectory(),
                           CMLibraryDirectoryName, CMVersion, nil];
  return [NSString pathWithComponents:paths];
}

- (NSString *)localLibrarySiteVersionPath
{
  NSArray *paths = [NSArray arrayWithObjects:@"/", CMLibraryDirectoryName,
                           CMSiteDirectoryName, CMVersion, nil];
  return [NSString pathWithComponents:paths];
}

- (NSString *)localLibrarySitePath
{
  NSArray *paths = [NSArray arrayWithObjects:@"/", CMLibraryDirectoryName,
                           CMSiteDirectoryName, nil];
  return [NSString pathWithComponents:paths];
}

- (NSString *)localLibraryVersionPath
{
  NSArray *paths = [NSArray arrayWithObjects:@"/", CMLibraryDirectoryName,
                           CMVersion, nil];
  return [NSString pathWithComponents:paths];
}

- (void)exit
{
  exit(0);
}

- (void)exitWithStatus:(int)status
{
  exit(status);
}

- (void)setVariablesForContext:(CMContext *)aContext
{
  // application
  SETVAR(aContext, self, CMApplicationVariableName);
  SETVAR(aContext, NSApp, NSApplicationVariableName);

  // argv, stdin, stdout, stderr
  SETVAR(aContext, arguments, CMArgumentsVariableName);
  SETVAR(aContext, [NSFileHandle fileHandleWithStandardInput],
         CMStandardInputVariableName);
  SETVAR(aContext, [NSFileHandle fileHandleWithStandardOutput],
         CMStandardOutputVariableName);
  SETVAR(aContext, [NSFileHandle fileHandleWithStandardError],
         CMStandardErrorVariableName);

  SETVAR(aContext, [self environment], CMEnvironmentVariableName);

  // struct
  SETVAR(aContext, [CMRange class], @"NSRange");
  SETVAR(aContext, [CMRect class],  @"NSRect");
  SETVAR(aContext, [CMPoint class], @"NSPoint");
  SETVAR(aContext, [CMSize class],  @"NSSize");
}

- (void)forwardInvocation:(NSInvocation *)anInvocation
{
  if ([runtime respondsToSelector:[anInvocation selector]])
    [anInvocation invokeWithTarget:runtime];
  else
    [self doesNotRecognizeSelector:[anInvocation selector]];
}

- (NSMethodSignature *)methodSignatureForSelector:(SEL)aSelector
{
  if ([super respondsToSelector:aSelector])
    return [super methodSignatureForSelector:aSelector];
  return [runtime methodSignatureForSelector:aSelector];
}

@end
