%% Copyright (c) 2011-2014, Loïc Hoguin <essen@ninenines.eu>
%%
%% Permission to use, copy, modify, and/or distribute this software for any
%% purpose with or without fee is hereby granted, provided that the above
%% copyright notice and this permission notice appear in all copies.
%%
%% THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
%% WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
%% MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
%% ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
%% WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
%% ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
%% OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

-module(cloudi_x_cowboy).

-export([start_http/4]).
-export([start_https/4]).
-export([start_spdy/4]).
-export([stop_listener/1]).
-export([set_env/3]).

-type http_headers() :: [{binary(), iodata()}].
-export_type([http_headers/0]).

-type http_status() :: non_neg_integer() | binary().
-export_type([http_status/0]).

-type http_version() :: 'HTTP/1.1' | 'HTTP/1.0'.
-export_type([http_version/0]).

-type onrequest_fun() :: fun((Req) -> Req).
-export_type([onrequest_fun/0]).

-type onresponse_fun() ::
	fun((http_status(), http_headers(), iodata(), Req) -> Req).
-export_type([onresponse_fun/0]).

-spec start_http(cloudi_x_ranch:ref(), non_neg_integer(), cloudi_x_ranch_tcp:opts(),
	cloudi_x_cowboy_protocol:opts()) -> {ok, pid()} | {error, any()}.
start_http(Ref, NbAcceptors, TransOpts, ProtoOpts)
		when is_integer(NbAcceptors), NbAcceptors > 0 ->
	cloudi_x_ranch:start_listener(Ref, NbAcceptors,
		cloudi_x_ranch_tcp, TransOpts, cloudi_x_cowboy_protocol, ProtoOpts).

-spec start_https(cloudi_x_ranch:ref(), non_neg_integer(), cloudi_x_ranch_ssl:opts(),
	cloudi_x_cowboy_protocol:opts()) -> {ok, pid()} | {error, any()}.
start_https(Ref, NbAcceptors, TransOpts, ProtoOpts)
		when is_integer(NbAcceptors), NbAcceptors > 0 ->
	cloudi_x_ranch:start_listener(Ref, NbAcceptors,
		cloudi_x_ranch_ssl, TransOpts, cloudi_x_cowboy_protocol, ProtoOpts).

-spec start_spdy(cloudi_x_ranch:ref(), non_neg_integer(), cloudi_x_ranch_ssl:opts(),
	cloudi_x_cowboy_spdy:opts()) -> {ok, pid()} | {error, any()}.
start_spdy(Ref, NbAcceptors, TransOpts, ProtoOpts)
		when is_integer(NbAcceptors), NbAcceptors > 0 ->
	TransOpts2 = [
		{connection_type, supervisor},
		{next_protocols_advertised,
			[<<"spdy/3">>, <<"http/1.1">>, <<"http/1.0">>]}
	|TransOpts],
	cloudi_x_ranch:start_listener(Ref, NbAcceptors,
		cloudi_x_ranch_ssl, TransOpts2, cloudi_x_cowboy_spdy, ProtoOpts).

-spec stop_listener(cloudi_x_ranch:ref()) -> ok | {error, not_found}.
stop_listener(Ref) ->
	cloudi_x_ranch:stop_listener(Ref).

-spec set_env(cloudi_x_ranch:ref(), atom(), any()) -> ok.
set_env(Ref, Name, Value) ->
	Opts = cloudi_x_ranch:get_protocol_options(Ref),
	{_, Env} = lists:keyfind(env, 1, Opts),
	Opts2 = lists:keyreplace(env, 1, Opts,
		{env, lists:keystore(Name, 1, Env, {Name, Value})}),
	ok = cloudi_x_ranch:set_protocol_options(Ref, Opts2).
