#!/usr/bin/perl -w

# $Id: clamdsh.pl,v 1.7 2007/05/18 18:45:11 okamura Exp $

require 5.8.0;
use strict;
use utf8;
package ClamAV::Shell;
#-------------------------------------------------------------------------------
# モジュール
#-------------------------------------------------------------------------------
# システム
use Getopt::Long	qw(:config no_ignore_case);
use Pod::Usage;
use Term::ReadLine;
use IO::Socket;
use Cwd;
use Time::HiRes;
use POSIX	qw(strftime);
use Sys::Hostname;
use File::Basename;

# プロジェクト

#-------------------------------------------------------------------------------
# グローバル変数宣言
#-------------------------------------------------------------------------------
use vars qw(
	$Version
	%ExitStatus
	%Option
	%Command
	$RunAsCommand
);

#-------------------------------------------------------------------------------
# 定数
#-------------------------------------------------------------------------------
# バージョン
$Version = 'clamdsh.pl v1.0.2';

# 終了ステータス
%ExitStatus = (
	NoError			=> 0,
	UknownOpts		=> 1,
	BadParams		=> 2,
);

# コマンド関数対応表
%Command = (
	'set'		=> \&ClamAV::Shell::Command::set,
	'unset'		=> \&ClamAV::Shell::Command::unset,
	'echo'		=> \&ClamAV::Shell::Command::echo,
	'cd'		=> \&ClamAV::Shell::Command::cd,
	'help'		=> \&ClamAV::Shell::Command::help,
	'PING'		=> \&ClamAV::Shell::Command::PING,
	'VERSION'	=> \&ClamAV::Shell::Command::VERSION,
	'RELOAD'	=> \&ClamAV::Shell::Command::RELOAD,
	'SHUTDOWN'	=> \&ClamAV::Shell::Command::SHUTDOWN,
	'SCAN'		=> \&ClamAV::Shell::Command::SCAN,
	'RAWSCAN'	=> \&ClamAV::Shell::Command::RAWSCAN,
	'CONTSCAN'	=> \&ClamAV::Shell::Command::CONTSCAN,
	'MULTISCAN'	=> \&ClamAV::Shell::Command::MULTISCAN,
	'STREAM'	=> \&ClamAV::Shell::Command::STREAM,
	'SESSION'	=> \&ClamAV::Shell::Command::SESSION,
);

# コマンドとして実行するかどうか
unless (defined($RunAsCommand)) {
	$RunAsCommand = 1;
}

#-------------------------------------------------------------------------------
# サブルーチン
#-------------------------------------------------------------------------------
# メインループ
sub MainLoop {
	my	$commandLine;

	ClamAV::Shell::UI::Initialize();
	while (defined ($commandLine = ClamAV::Shell::UI::ReadLine())) {
		if (Execute($commandLine)) {
			last;
		}
	}
	ClamAV::Shell::UI::Printf("\n");
}

# コマンド実行
# @param $commandLine コマンドライン
# @return 終了かどうか
sub Execute {
	my	($commandLine) = @_;
	my	@args;

	@args = ClamAV::Shell::Command::InterpretString($commandLine);
	if (0 < scalar(@args)) {
		if (exists $Command{$args[0]}) {
			$ClamAV::Shell::command::Variable{'?'}
				= &{$Command{$args[0]}}(@args[1 .. $#args])
			;
		}
		elsif ($args[0] eq 'exit' or $args[0] eq 'quit') {
			return 1;
		}
		else {
			ClamAV::Shell::UI::Printf("Unknown command: %s\n", $args[0]);
			$ClamAV::Shell::command::Variable{'?'} = -1;
		}
		ClamAV::Shell::UI::UpdatePromptVars();
	}

	return 0;
}

#--- コマンド ----
package ClamAV::Shell::Command;

use vars qw(
	%Variable
	%CmdInfo
);

# ユーザ変数
%Variable = (
	'?'	=> 0,
	'PWD'	=> Cwd::getcwd(),
);

# コマンド情報
%CmdInfo = (
	'set'		=> {
		'name'	=> 'set - set variable.',
		'usage'	=> "set NAME VALUE
    NAME  - variable name
       NAME must include only word charctors.
    VALUE - variable value",
		'desc'	=> "Set value of the variable named NAME to VALUE.
You can use the variable in command-line as \$NAME or \${NAME}.

[SPECIAL VARIABLES]
LocalSocket - path of clamd UNIX domain socket.
TCPSocket - port-number of clamd server.
TCPAddr - IP address of clamd server.
StreamMaxLength - max length of data which is send with STREAM command.
prompt - prompt string.
timeout - timeout for communication to clamd. (sec)
bufsiz - read/write buffer size. (byte)
recvTimeout - timeout for receiving clamd response. (milli sec)
    It is used in STREAM command for incremental receiving.
? - exit code of last command.
    It is readonly.
PWD - current working directory.
    It is readonly.",
		'exitCode'	=> "0 - Success.
1 - Invalid variable name.
2 - Invalid value.
3 - Don't set this variable.",
	},
	'unset'		=> {
		'name'	=> 'unset - unset variable',
		'usage'	=> "unset NAME
    NAME  - variable name
       NAME must include only word charctors.",
		'desc'	=> "Unset the variable named NAME.
When you use unseted variable, it is replaced with null strings.",
		'exitCode'	=> "0 - Success.
1 - Invalid variable name.
2 - Don't unset this variable.",
	},
	'echo'		=> {
		'name'	=> 'echo - echo back arguments',
		'usage'	=> "echo [argument ...]",
		'desc'	=> "Echo back given arguments.
You can check interpretation of arguments.",
		'exitCode'	=> "0 - Success.",
	},
	'cd'		=> {
		'name'	=> 'cd - change current directory',
		'usage'	=> "cd DIR
    DIR - path of directory.",
		'desc'	=> "Change current working directory to DIR.
You can use \$PWD or \${PWD} as current directory.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Failed.",
	},
	'help'		=> {
		'name'	=> 'help - display helps.',
		'usage'	=> "help [TOPIC]",
		'desc'	=> "Display topic list.
If TOPIC is give, display the manual of TOPIC.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Uknown topic.",
	},
	'PING'		=> {
		'name'	=> 'PING - PING clamd-command.',
		'usage'	=> "PING",
		'desc'	=> "Send PING clamd-command. And display clamd response.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send clamd-command.
4 - Can't receive clamd response.",
	},
	'VERSION'	=> {
		'name'	=> 'VERSION - VERSION clamd-command.',
		'usage'	=> "VERSION",
		'desc'	=> "Send VERSION clamd-command. And display clamd response.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send clamd-command.
4 - Can't receive clamd response.",
	},
	'RELOAD'	=> {
		'name'	=> 'RELOAD - RELOAD clamd-command.',
		'usage'	=> "RELOAD",
		'desc'	=> "Send RELOAD clamd-command. And display clamd response.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send clamd-command.
4 - Can't receive clamd response.",
	},
	'SHUTDOWN'	=> {
		'name'	=> 'SHUTDOWN - do SHUTDOWN clamd-command.',
		'usage'	=> "SHUTDOWN",
		'desc'	=> "Send SHUTDOWN clamd-command. And display clamd response.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send clamd-command.
4 - Can't receive clamd response.",
	},
	'SCAN'		=> {
		'name'	=> 'SCAN - SCAN clamd-command.',
		'usage'	=> "SCAN PATH
    PATH - path of file or directory.",
		'desc'	=> "send SCAN clamd-command with PATH.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send clamd-command.
4 - Can't receive clamd response.",
	},
	'RAWSCAN'	=> {
		'name'	=> 'RAWSCAN - RAWSCAN clamd-command.',
		'usage'	=> "RAWSCAN PATH
    PATH - path of file or directory.",
		'desc'	=> "send RAWSCAN clamd-command with PATH.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send clamd-command.
4 - Can't receive clamd response.",
	},
	'CONTSCAN'	=> {
		'name'	=> 'CONTSCAN - CONTSCAN clamd-command.',
		'usage'	=> "CONTSCAN PATH
    PATH - path of file or directory.",
		'desc'	=> "send CONTSCAN clamd-command with PATH.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send clamd-command.
4 - Can't receive clamd response.",
	},
	'MULTISCAN'	=> {
		'name'	=> 'MULTISCAN - MULTISCAN clamd-command.',
		'usage'	=> "MULTISCAN PATH
    PATH - path of file or directory.",
		'desc'	=> "send MULTISCAN clamd-command with PATH.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send clamd-command.
4 - Can't receive clamd response.",
	},
	'STREAM'	=> {
		'name'	=> 'STREAM - STREAM clamd-command.',
		'usage'	=> "STREAM PATH
    PATH - path of file.",
		'desc'	=>
"send STREAM clamd-command. And send contents of PATH through a port which is
indicated by clamd. Then display response of clamd.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send clamd-command.
4 - Can't receive clamd response.
5 - Invalid clamd response.
6 - Can't connect to clamd data port.
7 - Can't open the file.
8 - Can't read from the file or send to clamd.",
	},
	'SESSION'	=> {
		'name'	=> 'SESSION - SESSION clamd-command.',
		'usage'	=> "SESSION PATH
    PATH - path of file.",
		'desc'	=> "Send contents of the PATH using SESSION/END clamd-commands.
And display response of clamd.",
		'exitCode'	=> "0 - Success.
1 - Invalid arguments.
2 - Can't connect to clamd.
3 - Can't send SESSION clamd-command.
4 - Can't open the file.
5 - Can't send clamd-command.
6 - Can't send END clamd-command.",
	},
	'exit'	=> {
		'name'	=> 'exit - Terminate clamd shell.',
		'usage'	=> "exit",
		'desc'	=> "Terminate clamd shell.
You can terminate clamd shell by typing CTRL-D.",
		'exitCode'	=> "(non)",
	},
	'quit'	=> {
		'name'	=> 'quit - Terminate clamd shell.',
		'usage'	=> "quit",
		'desc'	=> "Terminate clamd shell.
You can terminate clamd shell by typing CTRL-D.",
		'exitCode'	=> "(non)",
	},
);

# 文字列から引数リストへの変換
# @param $str 解釈する文字列
# @return 引数のリスト
sub InterpretString {
	my	($str) = @_;
	my	@str;
	my	@interpreted = ();
	my	$status = 0;
	my	@args = ();

	unless (defined $str) {
		return ();
	}
	$str =~ s/^\s+//;
	$str =~ s/\s+$//;

	@str = split '', $str;
	for (my $i = 0; $i < scalar(@str); $i++) {
		# ニュートラル
		if ($status == 0) {
			if ($str[$i] eq '"') {
				$status = 1;	# ダブルクォート内へ
			}
			elsif ($str[$i] eq "'") {
				$status = 2;	# シングルクォート内へ
			}
			elsif ($str[$i] eq '\\') {
				$status = 3;	# エスケープへ
			}
			elsif ($str[$i] eq '$') {
				$status = 4;	# 変数名内へ
			}
			elsif ($str[$i] =~ m/\s/) {
				if (0 < scalar(@interpreted)) {
					push @args, join('', @interpreted);
					@interpreted = ();
				}
				$status = 5;	# 引数セパレータ内へ
			}
			elsif ($str[$i] eq '#') {
				last;
			}
			else {
				push @interpreted, $str[$i];
			}
		}
		# ダブルクォート内
		elsif ($status == 1) {
			if ($str[$i] eq '\\') {
				$status = 6;	# ダブルクォート内エスケープへ
			}
			elsif ($str[$i] eq '$') {
				$status = 7;	# ダブルクォート内変数名内へ
			}
			elsif ($str[$i] eq '"') {
				$status = 0;	# ニュートラルへ
			}
			else {
				push @interpreted, $str[$i];
			}
		}
		# シングルクオート内
		elsif ($status == 2) {
			if ($str[$i] eq '\\') {
				$status = 8;	# シングルクォート内エスケープへ
			}
			elsif ($str[$i] eq "'") {
				$status = 0;	# ニュートラルへ
			}
			else {
				push @interpreted, $str[$i];
			}
		}
		# エスケープ または ダブルクォート内エスケープ
		elsif ($status == 3 or $status == 6) {
			my	$subStr = join '', @str[$i .. $#str];

			if ($subStr =~ m/^(?#
				)([tnrfbae](?#				TAB や LF など
				)|1[0-7][0-7](?#			三桁の八進数の文字コード
				)|[1-7][0-7](?#				二桁の八進数の文字コード
				)|[0-7](?#					一桁の八進数の文字コード
				)|x[\da-fA-F]{1,2}(?#		十六進数の文字コード
				)|x\{[\da-fA-F]{1,4}\}(?#	十六進数のワイド文字コード
				)|N\{\w+\}|c?(?#			文字名
				))/
			) {
				my	$token = $1;

				push @interpreted, eval "\"\\$token\"";
				$i += length($token) - 1;
			}
			else {
				push @interpreted, $str[$i];
			}
			# ニュートラル または ダブルクォート内へ
			$status = $status == 3 ? 0 : 1;
		}
		# 変数名 または ダブルクォート内変数名
		elsif ($status == 4 or $status == 7) {
			my	$subStr = join '', @str[$i .. $#str];
			my	$name = undef;
			my	$value = undef;

			if ($subStr =~ m/^(\w+|\?)/) {
				$name = $1;
				$i += length($name) - 1;
			}
			elsif ($subStr =~ m/^{(\w+|\?)}/) {
				$name = $1;
				$i += length($name) + 2 - 1;
			}

			if (defined($name)) {
				if (exists($Variable{$name})) {
					$value = $Variable{$name};
				}
				elsif (exists($ClamAV::Shell::Option{$name})) {
					$value = $ClamAV::Shell::Option{$name};
				}
			}

			push @interpreted, defined($value) ? $value : '';
			# ニュートラル または ダブルクォート内へ
			$status = $status == 4 ? 0 : 1;
		}
		# 引数セパレータ内
		elsif ($status == 5) {
			if ($str[$i] =~ m/\S/) {
				$i--;
				$status = 0;	# ニュートラルへ
			}
		}
		# シングルクォート内エスケープ
		elsif ($status == 8) {
			unless ($str[$i] eq "'" or $str[$i] eq '\\') {
				push @interpreted, '\\'; 
			}
			push @interpreted, $str[$i];
			$status = 2;	# シングルクォート内へ
		}
	}
	# 不完全な終わり方をしている場合
	unless ($status == 0) {
		# ダブルクォートが開きっぱなしで終わっている場合
		if ($status == 1 or $status == 6 or $status == 7) {
			ClamAV::Shell::UI::Printf("Double quotation is not closed.\n");
		}
		# シングルクォートが開きっぱなしで終わっている場合
		if ($status == 2 or $status == 8) {
			ClamAV::Shell::UI::Printf("Single quotation is not closed.\n");
		}
		# エスケープされぱなしで終わっている場合
		if ($status == 3 or $status == 6 or $status == 8) {
			# 警告せずにエスケープを加える。
			push @interpreted, '\\';
		}
		# 変数が指示されぱなしで終わっている場合
		if ($status == 4 or $status == 7) {
			# 警告せずに $ を加える。
			push @interpreted, '$';
		}
		# 引き数セパレータ内で終わっている場合
		if ($status == 5) {
			# なにもしない
			1;
		}
	}
	push @args, join('', @interpreted);

	# 先頭の空白要素は省略する
	my	$i;

	for ($i = 0; $i < scalar(@args); $i++) {
		if ($args[$i] ne '') {
			last;
		}
	}
	if ($i == scalar(@args)) {
		@args = ();
	}
	else {
		@args = @args[$i .. $#args];
	}

	return @args;
}

# 変数の設定
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な変数名
#	2 - 不正な値
#	3 - 変更禁止変数の指定
sub set {
	my	@args = @_;

	if ($args[0] =~ m/^(\w+)$/ and scalar(@args) == 2) {
		my	($name, $value) = ($args[0], $args[1]);

		if (exists $ClamAV::Shell::Option{$name}) {
			unless (ClamAV::Shell::Option::SetOption($name, $value)) {
				ClamAV::Shell::UI::Printf(
					"Invalid %s value: %s\n", $name, $value
				);
				return 2;
			};
		}
		elsif ($name eq 'PWD') {
			ClamAV::Shell::UI::Printf("%s is readonly.", $name);
			return 3;
		}
		else {
			$Variable{$name} = $value;
		}
	}
	else {
		ClamAV::Shell::UI::Printf("[USAGE]\n%s", $CmdInfo{'set'}->{'usage'});
		return 1;
	}

	return 0;
}

# 変数の解除
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な変数名
#	2 - 変更禁止変数の指定
sub unset {
	my	@args = @_;

	unless ($args[0] =~ m/^\w+$/ and scalar(@args) == 1) {
		ClamAV::Shell::UI::Printf("[USAGE]\n%s", $CmdInfo{'unset'}->{'usage'});
		return 1;
	}
	unless ($args[0] ne 'PWD') {
		ClamAV::Shell::UI::Printf("%s is readonly.", $args[0]);
		return 2
	}

	if (
		grep { $_ eq $args[0] } qw(
			LocalSocket TCPSocket TCPAddr StreamMaxLength
			prompt
			timeout bufsiz recvTimeout
		)
	) {
		$ClamAV::Shell::Option{$args[0]} = undef;
	}
	else {
		delete $Variable{$args[0]};
	}

	return 0;
}

# エコー
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
sub echo {
	my	@args = @_;

	ClamAV::Shell::UI::Printf("%s", join(' ', @args)); 

	return 0;
}

# カレントディレクトリの変更
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 引数不正
#	2 - 変更失敗
sub cd {
	my	@args = @_;

	unless (scalar(@args) == 1) {
		ClamAV::Shell::UI::Printf("[USAGE]\n%s", $CmdInfo{'cd'}->{'usage'});
		return 1;
	}
	unless (chdir($args[0])) {
		ClamAV::Shell::UI::Printf("%s: %s", "$!", $args[0]);
		return 2;
	}

	$Variable{'PWD'} = Cwd::getcwd();

	return 0;
}

# ヘルプ表示
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - 不明なコマンド
sub help {
	my	@args = @_;

	if (scalar(@args) == 0) {
		ClamAV::Shell::UI::Printf("[Topics]\n");
		foreach (sort keys %CmdInfo) {
			ClamAV::Shell::UI::Printf("%s\n", $CmdInfo{$_}->{'name'});
		}
	}
	elsif (scalar(@args) == 1) {
		if (exists $CmdInfo{$args[0]}) {
			ClamAV::Shell::UI::Printf(
				"[NAME]
%s

[USAGE]
%s

[DESCRIPTION]
%s

[EXIT CODE]
%s
",
				$CmdInfo{$args[0]}->{'name'},
				$CmdInfo{$args[0]}->{'usage'},
				$CmdInfo{$args[0]}->{'desc'},
				$CmdInfo{$args[0]}->{'exitCode'}
			);
		}
		else {
			ClamAV::Shell::UI::Printf("HELP: unknown topic: %s", $args[0]);
			return 2;
		}
	}
	else {
		ClamAV::Shell::UI::Printf("[USAGE]\n%s\n", $CmdInfo{'help'}->{'usage'});
		return 1;
	}

	return 0;
}

# コマンド送信応答受信
# @param $clamdCmd clamd コマンド
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub _SendAndRecv {
	my	($clamdCmd, @args) = @_;

	unless (scalar(@args) == 0) {
		ClamAV::Shell::UI::Printf(
			"[USAGE]\n%s", $CmdInfo{$clamdCmd}->{'usage'}
		);
		return 1;
	}

	my	$conn = undef;
	my	$ret = 0;

	$conn = ClamAV::Shell::Clamd::Open();
	if ($conn) {
		if (ClamAV::Shell::Clamd::Send($conn, "$clamdCmd\n")) {
			my	$response = ClamAV::Shell::Clamd::Recv($conn);

			if (defined($response)) {
				ClamAV::Shell::UI::Printf("%s", $response);
			}
			else {
				ClamAV::Shell::UI::Printf("%s: %s.", $clamdCmd, "$!");
				$ret = 4;
			}
		}
		elsif ($!) {
			ClamAV::Shell::UI::Printf("%s: %s.", $clamdCmd, "$!");
			$ret = 3;
		}
		else {
			ClamAV::Shell::UI::Printf("%s: failed.", $clamdCmd);
			$ret = 3;
		}
		ClamAV::Shell::Clamd::Close($conn);
	}
	else {
		$ret = 2;
	}

	return $ret;
}

# PING
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub PING {
	return _SendAndRecv('PING', @_);
}

# VERSION
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub VERSION {
	return _SendAndRecv('VERSION', @_);
}

# RELOAD
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub RELOAD {
	return _SendAndRecv('RELOAD', @_);
}

# SHUTDOWN
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub SHUTDOWN {
	return _SendAndRecv('SHUTDOWN', @_);
}

# スキャンの基底操作
# @param $clamdCmd clamd コマンド
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub _SCAN {
	my	($clamdCmd, @args) = @_;

	unless (scalar(@args) == 1) {
		ClamAV::Shell::UI::Printf(
			"[USAGE]\n%s", $CmdInfo{$clamdCmd}->{'usage'}
		);
		return 1;
	}
	if ($args[0] =~ m/\n/) {
		ClamAV::Shell::UI::Printf("%s: include LF :%s", $clamdCmd, $args[0]);
		return 1;
	}

	my	$conn = undef;
	my	$ret = 0;

	$conn = ClamAV::Shell::Clamd::Open();
	if ($conn) {
		if (ClamAV::Shell::Clamd::Send($conn, "$clamdCmd $args[0]\n")) {
			my	$response;

			$response = ClamAV::Shell::Clamd::Recv($conn);
			if (defined($response)) {
				ClamAV::Shell::UI::Printf("%s", $response);
			}
			else {
				ClamAV::Shell::UI::Printf("%s: %s.", $clamdCmd, "$!");
				$ret = 4;
			}
		}
		elsif ($!) {
			ClamAV::Shell::UI::Printf("%s: %s.", $clamdCmd, "$!");
			$ret = 3;
		}
		else {
			ClamAV::Shell::UI::Printf("%s: failed.", $clamdCmd);
			$ret = 3;
		}
		ClamAV::Shell::Clamd::Close($conn);
	}
	else {
		$ret = 2;
	}

	return $ret;
}

# SCAN
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub SCAN {
	return _SCAN('SCAN', @_);
}

# RAWSCAN
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub RAWSCAN {
	return _SCAN('RAWSCAN', @_);
}

# CONTSCAN
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub CONTSCAN {
	return _SCAN('CONTSCAN', @_);
}

# MULTISCAN
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
sub MULTISCAN {
	return _SCAN('MULTISCAN', @_);
}

# STREAM
# @param @args 引数リスト
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd へのコマンド送信失敗
#	4 - clamd からの応答受信失敗
#	5 - clamd からの応答が不正
#	6 - データポートへの接続失敗
#	7 - ファイルオープン失敗
#	8 - ファイル読み取りまたはデータ送信に失敗
sub STREAM {
	my	(@args) = @_;

	unless (scalar(@args) == 1) {
		ClamAV::Shell::UI::Printf(
			"[USAGE]\n%s", $CmdInfo{'STREAM'}->{'usage'}
		);
		return 1;
	}

	my	$conn = undef;
	my	$ret = 0;

	$conn = ClamAV::Shell::Clamd::Open();
	if ($conn) {
		if (ClamAV::Shell::Clamd::Send($conn, "STREAM\n")) {
			my	$response = ClamAV::Shell::Clamd::RecvLine($conn);

			if (defined($response)) {
				if ($response =~ m/^PORT (\d+)/) {
					my	$port = $1;
					my	$dconn = ClamAV::Shell::Clamd::Open($port);

					if ($dconn) {
						if (open FH, '<', $args[0]) {
							my	$buf;
							my	$len;
							my	$limitLen
								= $ClamAV::Shell::Option{'StreamMaxLength'}
							;
							my	$bufsiz = $ClamAV::Shell::Option{'bufsiz'};

							if ($limitLen < $bufsiz) {
								$bufsiz = $limitLen;
							}

							while (
								0 < $limitLen
									and
								$len = read(FH, $buf, $bufsiz)
							) {
								$limitLen -= $len;
								if ($limitLen < $bufsiz) {
									$bufsiz = $limitLen;
								}

								unless (
									ClamAV::Shell::Clamd::Send(
										$dconn, $buf, $len
									)
								) {
									$ret = 8;
									last;
								}
							}
							if ($ret == 0) {
								close FH;
								ClamAV::Shell::Clamd::Close($dconn);
								$response = ClamAV::Shell::Clamd::Recv($conn);
								ClamAV::Shell::UI::Printf("%s", $response);
							}
							else {
								ClamAV::Shell::UI::Printf(
									"STREAM: %s: %s", "$!", $args[0]
								);
								close FH;
								ClamAV::Shell::Clamd::Close($dconn);
								$response = ClamAV::Shell::Clamd::Recv($conn);
								ClamAV::Shell::UI::Printf("%s", $response);
							}
						}
						else {
							ClamAV::Shell::UI::Printf(
								"STREAM: %s: %s", "$!", $args[0]
							);
							ClamAV::Shell::Clamd::Close($dconn);
							$response = ClamAV::Shell::Clamd::Recv($conn);
							ClamAV::Shell::UI::Printf("%s", $response);
							$ret = 7;
						}
					}
					else {
						ClamAV::Shell::UI::Printf(
							"STREAM: data port connection failed: %s", $port
						);
						$ret = 6;
					}
				}
				else {
					ClamAV::Shell::UI::Printf(
						"STREAM: invalid response: %s", $response
					);
					$ret = 5;
				}
			}
			else {
				ClamAV::Shell::UI::Printf("STREAM: %s.", "$!");
				$ret = 4;
			}
		}
		elsif ($!) {
			ClamAV::Shell::UI::Printf("STREAM: %s.", "$!");
			$ret = 3;
		}
		else {
			ClamAV::Shell::UI::Printf("STREAM: failed.");
			$ret = 3;
		}
		ClamAV::Shell::Clamd::Close($conn);
	}
	else {
		$ret = 2;
	}

	return $ret;
}

# SESSION
# @param $args 引数
# @return 終了ステータス
#	0 - 成功
#	1 - 不正な引数
#	2 - clamd への接続失敗
#	3 - clamd への SESSION コマンド送信失敗
#	4 - ファイルオープン失敗
#	5 - データ送信に失敗
#	6 - clamd への END コマンド送信失敗
sub SESSION {
	my	(@args) = @_;

	unless (scalar(@args) == 1) {
		ClamAV::Shell::UI::Printf(
			"[USAGE]\n%s", $CmdInfo{'SESSION'}->{'usage'}
		);
		return 1;
	}

	my	$conn = undef;
	my	$ret = 0;

	$conn = ClamAV::Shell::Clamd::Open();
	if ($conn) {
		if (ClamAV::Shell::Clamd::Send($conn, "SESSION\n")) {
			if (open FH, '<', $args[0]) {
				my	$lineNum = 1;
				my	$response;
				my	$line;
				
				while ($line = <FH>) {
					unless (ClamAV::Shell::Clamd::Send($conn, $line)) {
						chomp $line;
						if ($!) {
							ClamAV::Shell::UI::Printf(
								"SESSION: send: %s: %s#%d: %s",
								"$!", $args[0], $lineNum, $line
							);
						}
						else {
							ClamAV::Shell::UI::Printf(
								"SESSION: send: faild: %s#%d: %s",
								$args[0], $lineNum, $line
							);
						}
						$ret = 5;
						last;
					}
					$response = ClamAV::Shell::Clamd::RecvNonBreak($conn);
					ClamAV::Shell::UI::Printf("%s", $response);
				}
				continue {
					$lineNum++;
				}
				close FH;

				if ($ret == 0) {
					unless (ClamAV::Shell::Clamd::Send($conn, "END\n")) {
						ClamAV::Shell::UI::Printf(
							"SESSION: send END: %s: %s",
							"$!", $args[0]
						);
						$ret = 6;
					}
					while ($response = ClamAV::Shell::Clamd::RecvLine($conn)) {
						ClamAV::Shell::UI::Printf("%s", $response);
					}
				}
				else {
					ClamAV::Shell::Clamd::Send($conn, "END\n");
					while ($response = ClamAV::Shell::Clamd::RecvLine($conn)) {
						ClamAV::Shell::UI::Printf("%s", $response);
					}
				}
			}
			else {
				ClamAV::Shell::UI::Printf(
					"SESSION: open: %s: %s", "$!", $args[0]
				);
				$ret = 4;
			}
		}
		elsif ($!) {
			ClamAV::Shell::UI::Printf("SESSION: send SESSION: %s.", "$!");
			$ret = 3;
		}
		else {
			ClamAV::Shell::UI::Printf("SESSION: send SESSION: failed.");
			$ret = 3;
		}
		ClamAV::Shell::Clamd::Close($conn);
	}
	else {
		$ret = 2;
	}

	return $ret;
}

#--- オプション解釈 ----
package ClamAV::Shell::Option;

use vars	qw(
	%OptionType
);

# オプション型対応表
%OptionType = (
	#--- clamd.conf 相当のオプション ----
	'LocalSocket'		=> \&ParseStringValue,
	'TCPSocket'			=> \&ParseNumberValue,
	'TCPAddr'			=> \&ParseStringValue,
	'StreamMaxLength'	=> \&ParseSizeValue,
	#--- ユーザーインターフェース ----
	# プロンプト
	'prompt'	=> \&ParseStringValue,
	#--- その他 ----
	'timeout'	=> \&ParseNumberValue,
	'bufsiz'	=> \&ParseSizeValue,
	'recvTimeout'	=> \&ParseNumberValue,
);

# コマンドライン用オプションのセット
#	失敗すると exit する。
# @param $name オプション名
# @param $value オプション値
sub SetOption4CmdLine {
	my	($name, $value) = @_;

	unless (ClamAV::Shell::Option::SetOption($name, $value)) {
		printf STDERR "Invalid %s value: %s\n", $name, $value;
		exit $ClamAV::Shell::ExitStatus{'BadParams'};
	}
}

# オプションのセット
# @param $name オプション名
# @param $value オプション値
# @return 成功/失敗
sub SetOption {
	my	($name, $value) = @_;
	my	$internalValue = &{$OptionType{$name}}($value);

	unless (defined $internalValue) {
		return 0;
	}
	$ClamAV::Shell::Option{$name} = $internalValue;

	return 1;
}

# サイズ値の解釈
# @param $value サイズ値
# @return バイト数
#	undef - 不正な値
sub ParseSizeValue {
	my	($value) = @_;
	my	$retVal = undef;

	if (defined $value) {
		if ($value =~ m/^(\d+)k$/i) {
			$retVal = $1 * 1024;
		}
		elsif ($value =~ m/^(\d+)m$/i) {
			$retVal = $1 * 1024 * 1024;
		}
		elsif ($value =~ m/^(\d+)$/) {
			$retVal = $1;
		}
	}

	return $retVal;
}

# 数値の解釈
# @param $value サイズ値
# @return バイト数
#	undef - 不正な値
sub ParseNumberValue {
	my	($value) = @_;
	my	$retVal = undef;

	if (defined $value) {
		if ($value =~ m/^([1-9]\d+)$/) {
			$retVal = $1;
		}
	}

	return $retVal;
}

# 文字列値の解釈
# @param $value サイズ値
# @return バイト数
#	undef - 不正な値
sub ParseStringValue {
	my	($value) = @_;

	return "$value";
}

#--- ユーザインターフェース ----
package ClamAV::Shell::UI;

use vars	qw(
	$Term
	$HistoryNumber
	%PromptVar
);

# ターミナルオブジェクト
$Term = undef;

# ヒストリー番号
$HistoryNumber = 0;

# プロンプト変数
%PromptVar = (
	# bell
	qr/\\b/	=> sub { "\07" },
	# "Weekday Month Date"
	qr/\\d/	=> sub { POSIX::strftime('%a %b %d', localtime) },
	# strftime
	qr/\\D\{([^}]*)\}/	=> sub { POSIX::strftime($_[0], localtime) },
	# escape
	qr/\\e/	=> sub { "\033" },
	# short hostname
	qr/\\h/	=> sub {
		my	$name = Sys::Hostname::hostname();
		$name =~ s/\..*//;
		$name
	},
	# hostname
	qr/\\H/	=> sub { Sys::Hostname::hostname() },
	# newline
	qr/\\n/	=> sub { "\n" },
	# carriage return
	qr/\\r/	=> sub { "\r" },
	# 24-hour HH:MM:SS
	qr/\\t/	=> sub { POSIX::strftime('%T', localtime) },
	# 12-hour HH:MM:SS
	qr/\\T/	=> sub { POSIX::strftime('%I:%M:%S', localtime) },
	# 12-hour am/pm
	qr/\\@/	=> sub { POSIX::strftime('%r', localtime) },
	# 24-hour HH:MM
	qr/\\A/	=> sub { POSIX::strftime('%R', localtime) },
	# current user name
	qr/\\u/	=> sub { getpwuid($> ) },
	# the version of clamdsh.pl
	qr/\\v/	=> sub {
		my	$v = $ClamAV::Shell::Version;
		$v =~ s/clamdsh\.pl v(\d+\.\d+)/$1/;
		$v
	},
	# the version + patchelvel of clamdsh.pl
	qr/\\V/	=> sub {
		my	$v = $ClamAV::Shell::Version;
		if ($v =~ m/clamdsh\.pl v(\d+\.\d+\.\d+)/) {
			$v = $1;
		}
		elsif ($v =~ m/clamdsh\.pl v(\d+\.\d+)/) {
			$v = "$1.0";
		}
		elsif ($v =~ m/clamdsh\.pl v(\d+)/) {
			$v = "$1.0.0";
		}
		$v
	},
	# current working directory
	qr/\\w/	=> sub { Cwd::cwd() },
	# basename of current working directory
	qr/\\W/	=> sub { File::Basename::basename(Cwd::cwd()) },
	# history number
	qr/\\!/	=> sub { $HistoryNumber },
	# command number (synonym of history number)
	qr/\\\*/	=> sub { $HistoryNumber },
	# # or $
	qr/\\\$/	=> sub { ${>} ? '$' : '#' },
	# charctor octal code
	qr/\\([0-7]{3})/	=> sub { chr(oct($_[0])) },
	# backslash
	qr/\\\\/	=> sub { '\\' },

	# LocalSocket
	qr/\\\{LocalSocket\}/	=> sub {
		$ClamAV::Shell::Option{'LocalSocket'}
			? $ClamAV::Shell::Option{'LocalSocket'}
			: ''
	},
	# TCPAddr
	qr/\\\{TCPAddr\}/	=> sub {
		$ClamAV::Shell::Option{'TCPAddr'}
			? $ClamAV::Shell::Option{'TCPAddr'}
			: ''
	},
	# TCPSocket
	qr/\\\{TCPSocket\}/	=> sub {
		$ClamAV::Shell::Option{'TCPSocket'}
			? $ClamAV::Shell::Option{'TCPSocket'}
			: ''
	},
	# StreamMaxLength
	qr/\\\{StreamMaxLength\}/	=> sub {
		$ClamAV::Shell::Option{'StreamMaxLength'}
			? $ClamAV::Shell::Option{'StreamMaxLength'}
			: ''
	},
	# timeout
	qr/\\\{timeout\}/	=> sub {
		$ClamAV::Shell::Option{'timeout'}
			? $ClamAV::Shell::Option{'timeout'}
			: ''
	},
	# recvTimeout
	qr/\\\{recvTimeout\}/	=> sub {
		$ClamAV::Shell::Option{'recvTimeout'}
			? $ClamAV::Shell::Option{'recvTimeout'}
			: ''
	},
	# bufsiz
	qr/\\\{recvTimeout\}/	=> sub {
		$ClamAV::Shell::Option{'bufsiz'}
			? $ClamAV::Shell::Option{'bufsiz'}
			: ''
	},
	# clamd socket
	qr/\\\{clamd\}/	=> sub {
		if (defined($ClamAV::Shell::Option{'LocalSocket'})) {
			$ClamAV::Shell::Option{'LocalSocket'}
		}
		elsif (
			defined($ClamAV::Shell::Option{'TCPSocket'})
				and
			defined($ClamAV::Shell::Option{'TCPAddr'})
		) {
			sprintf("%s:%s",
				$ClamAV::Shell::Option{'TCPAddr'},
				$ClamAV::Shell::Option{'TCPSocket'}
			)
		}
		else {
			'(no clamd)'
		}
	},
);

# 初期化
sub Initialize {
	unless ($Term) {
		$Term = new Term::ReadLine 'CLAMD SHell';
	}
	UpdatePromptVars();
}

# プロンプト変数更新
sub UpdatePromptVars {
	$HistoryNumber++;
}

# コマンドライン取得
# @param $prompt プロンプト(省略可能)
sub ReadLine {
	my	($prompt) = @_;

	$prompt = $ClamAV::Shell::Option{'prompt'}	unless (defined $prompt);
	# 多重に置き換えられる可能性があるがプロンプトに凝ってもしょうがない。
	while (my ($reg, $func) = each %PromptVar) {
		$prompt =~ s/$reg/&{$func}($1, $2, $3, $4, $5, $6, $7, $8, $9)/eg;
	}

	return $Term->readline($prompt);
}

# 出力
sub Printf {
	printf {$Term->OUT} @_;
}

#--- clamd 操作 ----
package ClamAV::Shell::Clamd;

# 接続
# @param $port 接続ポート(省略可能)
#	$port が指定された場合は STREAM コマンドのデータ送信用の接続と解釈する。
# @return IO::Socket オブジェクト
sub Open {
	my	($port) = @_;
	my	$conn = undef;

	# STREAM コマンドのデータ接続の場合
	if (defined $port) {
		$conn = IO::Socket::INET->new(
			PeerAddr	=> defined($ClamAV::Shell::Option{LocalSocket})
				? '127.0.0.1'
				: $ClamAV::Shell::Option{'TCPAddr'},
			PeerPort	=> $port,
			Proto		=> 'tcp',
			Type		=> IO::Socket::SOCK_STREAM,
			Timeout		=> $ClamAV::Shell::Option{'timeout'},
		);
		unless ($conn and $conn->connected()) {
			ClamAV::Shell::UI::Printf(
				"%s: %s:%s",
				"$!",
				defined($ClamAV::Shell::Option{LocalSocket})
					? $ClamAV::Shell::Option{'TCPAddr'}
					: '127.0.0.1',
				$port
			);
			$conn = undef;
		}
	}
	# UNIX ドメインソケットが指定されている場合
	elsif (defined($ClamAV::Shell::Option{LocalSocket})) {
		$conn = IO::Socket::UNIX->new(
			Type 		=> IO::Socket::SOCK_STREAM,
			PeerPort	=> $port,
			Peer 		=> $ClamAV::Shell::Option{'LocalSocket'},
		);
		unless ($conn and $conn->connected()) {
			ClamAV::Shell::UI::Printf(
				"%s: %s", "$!", $ClamAV::Shell::Option{'LocalSocket'}
			);
			$conn = undef;
		}
	}
	# INET ソケットが指定されている場合
	elsif (
		defined($ClamAV::Shell::Option{'TCPSocket'})
			and
		defined($ClamAV::Shell::Option{'TCPAddr'})
	) {
		$conn = IO::Socket::INET->new(
			PeerAddr	=> $ClamAV::Shell::Option{'TCPAddr'},
			PeerPort	=> $ClamAV::Shell::Option{'TCPSocket'},
			Proto		=> 'tcp',
			Type		=> IO::Socket::SOCK_STREAM,
			Timeout		=> $ClamAV::Shell::Option{'timeout'},
		);
		unless ($conn and $conn->connected()) {
			ClamAV::Shell::UI::Printf(
				"%s: %s:%s", "$!",
				$ClamAV::Shell::Option{'TCPAddr'},
				$ClamAV::Shell::Option{'TCPSocket'}
			);
			$conn = undef;
		}
	}
	# ソケットタイプが特定できない場合
	else {
		ClamAV::Shell::UI::Printf("Socket type is not specified.");
		return undef;
	}

	return $conn;
}

# 切断
sub Close {
	my	($conn) = @_;

	$conn->close();
}

# データ送信
# @param $conn IO::Socket オブジェクト
# @param $data データ
# @param $len データ長(省略可能)
# @return 成功/失敗
sub Send {
	my	($conn, $data, $len) = @_;
	my	$sentLen;
	my	$restLen;

	$! = 0;
	return undef	unless (defined $data);
	$len = length($data)	unless (defined $len);
	for ($restLen = $len; 0 < $restLen; $restLen -= $sentLen) {
		$sentLen = $conn->syswrite(
			substr($data, $len - $restLen, $restLen), $restLen
		);
		return undef	unless (defined $sentLen);
	}

	return 1;
}

# 受信
# @param $conn IO::Socket オブジェクト
# @return 受信内容
sub Recv {
	my	($conn) = @_;
	my	$data = '';

	while (my $line = RecvLine($conn)) {
		$data .= $line;
	}

	return $data;
}

# 一行受信
# @param $conn IO::Socket オブジェクト
# @return 受信行
sub RecvLine {
	my	($conn) = @_;

	$! = 0;
	return $conn->getline();
}

# non-break 受信
# @param $conn IO::Socket オブジェクト
# @return 受信内容
sub RecvNonBreak {
	my	($conn) = @_;
	my	$data = '';

	while (IsEnableRecv($conn)) {
		my	$line = RecvLine($conn);

		unless (defined($line)) {
			last;
		}
		$data .= $line;
	}

	return $data;
}

# 受信可能か
# @param $conn IO::Socket オブジェクト
# @return 受信可能かどうか
sub IsEnableRecv {
	my	($conn) = @_;
	my	$peekCh = undef;
	my	$recvRet = undef;

	eval {
		local	$SIG{ALRM};

		$SIG{ALRM} = sub { die 'peek timeout.' };
		Time::HiRes::alarm($ClamAV::Shell::Option{'recvTimeout'} / 1000);
		$recvRet = $conn->recv($peekCh, 1, 0x2);
		Time::HiRes::alarm(0);
	};
	if ($@) {
		Time::HiRes::alarm(0);
		if ($@ =~ m/^peek timeout\./) {
			return 0;
		}
		ClamAV::Shell::UI::Printf("%s", $@);
		return undef;
	}

	return (defined($recvRet) and (0 < length($peekCh)));
}

#-------------------------------------------------------------------------------
# グローバル変数初期値
#-------------------------------------------------------------------------------
package ClamAV::Shell;

# 与えられたオプション
%Option = (
	#--- clamd.conf 相当のオプション ----
	'LocalSocket'		=> undef,
	'TCPSocket'			=> undef,
	'TCPAddr'			=> undef,
	'StreamMaxLength'	=> 10*1024*1024,
	#--- ユーザーインターフェース ----
	# プロンプト
	'prompt'	=> '\{clamd}\$ ',
	#--- その他 ----
	'timeout'	=> 10,
	'bufsiz'	=> 4096,
	'recvTimeout'	=> 20,
);

#-------------------------------------------------------------------------------
# 引数解釈
#-------------------------------------------------------------------------------
if ($RunAsCommand) {
	GetOptions(
		#--- clamd.conf 相当のオプション ----
		'LocalSocket=s'		=> \&ClamAV::Shell::Option::SetOption4CmdLine,
		'TCPSocket=i'		=> \&ClamAV::Shell::Option::SetOption4CmdLine,
		'TCPAddr=s'			=> \&ClamAV::Shell::Option::SetOption4CmdLine,
		'StreamMaxLength=s'	=> \&ClamAV::Shell::Option::SetOption4CmdLine,
		#--- ユーザーインターフェース ----
		'prompt=s'	=> \&ClamAV::Shell::Option::SetOption4CmdLine,
		#--- その他 ----
		'timeout=i'	=> \&ClamAV::Shell::Option::SetOption4CmdLine,
		'bufsiz=i'	=> \&ClamAV::Shell::Option::SetOption4CmdLine,
		'recvTimeout=i'	=> \&ClamAV::Shell::Option::SetOption4CmdLine,
		#--- 情報表示 ----
		'help|h'	=> sub {
			pod2usage(-verbose => 2, exit => $ExitStatus{NoError})
		},
		'version'	=> sub {
			print <<EOF;
$Version
Copyright (C) 2007 OKAMURA Yuji.
This is free software; see the source for copying conditions. There is NO
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
EOF
			exit $ExitStatus{NoError};
		},
	) or pod2usage(-verbose => 0, exit => $ExitStatus{UknownOpts});
}

#-------------------------------------------------------------------------------
# メイン
#-------------------------------------------------------------------------------
if ($RunAsCommand) {
	MainLoop();
}

#===============================================================================
1;
__END__;

=head1 NAME

clamdsh.pl - simple clamd shell.

=head1 SYNOPSIS

clamdsh.pl [--LocalSocket=I<socket>] [--TCPSocket=I<port>]
[--TCPAddr=I<address>] [--StreamMaxLength=I<stream-size>] [--prompt=I<prompt>]
[--timeout=I<sec>] [--bufsiz=I<buffer-size>] [--recvTimeout=I<msec>]
 

=head1 DESCRIPTION

B<clamdsh.pl> is a simple shell for clamd.

Run B<clamdsh.pl>. And type "help" and LF.

=head1 OPTIONS

=over

=item --LocalSocket=I<socket>

Path to a local (Unix) socket clamd is listening on.

This is same with next internal command line.

=over 4

set LocalSocket I<socket>

=back

DEFAULT: (no)

=item --TCPSocket=I<port>

TCP port number clamd is listening on.

This is same with next internal command line.

=over 4

set TCPSocket I<port>

=back

DEFAULT: (no)

=item --TCPAddr=I<address>

TCP socket address clamd binding to.

This is same with next internal command line.

=over 4

set TCPAddr I<address>

=back

DEFAULT: (no)

=item --StreamMaxLength=I<stream-size>

clamd uses FTP-like protocol to receive stream data. This option allows you to
specify the upper limit for data size that will be transferred to clamd when
scanning a single file with STREAM command.

This is same with next internal command line.

=over 4

set StreamMaxLength I<stream-size>

=back

DEFAULT: 10M

=item --prompt=I<prompt>

I<prompt> is expanded and used as prompt string. See L</PROMPTING> bellow.

This is same with next internal command line.

=over 4

set prompt I<prompt>

=back

DEFAULT: '\{clamd}\$ '

=item --timeout=I<sec>

Timeout value for various operations of INET sockt.

This is same with next internal command line.

=over 4

set timeout I<sec>

=back

DEFAULT: 10

=item --bufsiz=I<buffer-size>

When STREAM clamd-command is issued, B<clamdsh.pl> use read/write buffer. It
reads data from specified file and keep them on the buffer, and it write them
to INET socket for clamd data port. You can specify the size of this buffer with
this option.

This is same with next internal command line.

=over 4

set bufsiz I<buffer-size>

=back

DEFAULT: 4096

=item --recvTimeout=I<msec>

When SESSION clamd-command is issued, B<clamdsh.pl> receives the response from
clamd incrementally. At this time, B<clamdsh.pl> waits for the next response to
arrive only at short time. You can specify the short time by this option. 

This is same with next internal command line.

=over 4

set recvTimeout I<msec>

=back

DEFAULT: 20

=back

=head1 COMMAND ISSUE

A command line of B<clamdsh.pl> is a sequence by blank-separated words.
The first word specifies the command to issue.
The remaining words are aruguments to the invoked command.
Each command lines terminate with line feed.

It is just like a OS shell.
But B<clamdsh.pl> doesn't support multi lines command.
Backslash on end of line doesn't escape line feed,
double or single quote can't include line feeds.

You can display the list of commands by `help' command.

=head1 QUOTING

Single quotes and double quotes are used to remove the special meaning of
certain characters to the shell. They work like quotes in Perl and standard
Unix shell.
Double quoted literals are subject to variables and backslush substitution.
Single quoted literals are not subject except for "\'" and "\\". 

=over 4

SCAN /home/myname/Desktop/try and error

SCAN "/home/myname/Desktop/try and error"

=back

First command line has three arguments for SCAN. But second has only one.

=over 4

SCAN $PWD/clam.exe

SCAN "$PWD/clam.exe"

SCAN '$PWD/clam.exe'

=back

First and second command lines are same. But third is not same with others.

=over 4

SCAN ${PWD}/try\x20and\x20error

SCAN "${PWD}/try and error"

=back

Both command lines are same.

For detail, see perl documents such as perlop and perldata.

=head1 PROMPTING

B<clamdsh.pl> dsiplay the prompt when it is ready to read command.
B<clamdsh.pl> allows this prompt string to be customized by the variable
C<$prompt>.

In the value of C<$prompt>, you can use backslash-escaped special
charctors of L<bash(1)/PROMPTING> except "\l", "\s", "\[" and "\]".
Additionaly you can use the followings.

=over

=item \{LocalSocket}

Replace with current value of C<LocalSocket>.
If it is not set, null string.

=item \{TCPSocket}

Replace with current value of C<TCPSocket>.
If it is not set, null string.

=item \{TCPAddr}

Replace with current value of C<TCPAddr>.
If it is not set, null string.

=item \{StreamMaxLength}

Replace with current value of C<StreamMaxLength>.
If it is not set, null string.

=item \{timeout}

Replace with current value of C<timeout>.
If it is not set, null string.

=item \{recvTimeout}

Replace with current value of C<recvTimeout>.
If it is not set, null string.

=item \{bufsiz}

Replace with current value of C<bufsiz>.
If it is not set, null string.

=item \{clamd}

Replace with current socket for clamd. If it is not defined, "(no clamd)".
B<clamdsh.pl> primarily uses the value of C<LocalSocket> as UNIX domain socket
when C<LocalSocket> is set.
When C<LocalSocket> is not set and both of C<TCPAddr> and C<TCPSocket> are
set, B<clamdsh.pl> uses "$TCPAddr:$TCPSocket" as INET socket.

=back

NOTE: When you use terminal control escape sequences, it is interfered by
Term::ReadLine perl module control.

=head1 KNOWN PROBLEM

B<clamdsh.pl> dies on the following condition. 

=over 4

The value of C<StreamMaxLength> in B<clamdsh.pl> is greater than the value of
C<StreamMaxLength> in clamd.

AND STERAM command is used for the file.

AND size of the file is greater than the value of C<StreamMaxLength> in clamd.

=back

This problem can't be evaded. Use STREAM command carefully.

=head1 EXIT STATUS

0 - No error.

1 - Unknown option is given.

2 - Bad option value is given.

=head1 SEE ALSO

L<clamd.conf(5)>, L<clamd(8)>, perlop, perldata, L<bash(1)>

=head1 AUTHOR

OKAMURA Yuji E<lt>https://sourceforge.jp/users/okamura/E<gt>

=cut
