/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.service.importpolicy.impl;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.excel.impl.ExcelPolicyReaderImpl;
import jp.co.ogis_ri.citk.policytool.common.exception.ExcelPolicyImportException;
import jp.co.ogis_ri.citk.policytool.common.resource.MessageInfo;
import jp.co.ogis_ri.citk.policytool.common.util.ApplicationContextUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.PolicyRepository;
import jp.co.ogis_ri.citk.policytool.domain.policy.impl.JpaPolicyRepositoryImpl;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Permit;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Resource;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Subject;
import jp.co.ogis_ri.citk.policytool.domain.realm.RealmRepository;
import jp.co.ogis_ri.citk.policytool.domain.realm.impl.JpaRealmRepositoryImpl;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Realm;

import mockit.Deencapsulation;
import mockit.Expectations;
import mockit.Mocked;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration({"ImportPolicyServiceImplTest-context.xml"})
public class ImportPolicyServiceImplTest {

    @Autowired
    @Qualifier(value = "importPolicyServiceImpl")
    private ImportPolicyServiceImpl service = null;

    @Mocked
    private ExcelPolicyReaderImpl excelPolicyReader;
    
    @Mocked
    private TestPolicyRepository policyRepository;
    
    @Mocked
    private RealmRepository realmRepository;
    
    @Mocked
    private MessageSourceAccessor messageSourceAccessor;

    @SuppressWarnings("unused")
	@Mocked
    private MessageInfo messageInfo;
    
    @Before
    public void setUp() {
        Deencapsulation.setField(ApplicationContextUtil.class, "context", null);
        Deencapsulation.setField(
                ApplicationContextUtil.class,
                "FILE_APPLICATION_CONTEXT",
                "jp/co/ogis_ri/citk/policytool/service/importpolicy/impl/ImportPolicyServiceImplTest-context.xml");
        
    	// 1回 ApplicationContextUtil のメソッドを呼んでおく
		// ApplicationContextUtilのApplicationContext を初めて使用するときの動作
        new Expectations() {
        	{
				new ClassPathXmlApplicationContext((String[])any);
        	}
        };
        
        ApplicationContextUtil.getContext();
    }
    
	@Test
	public void testImportPolicy_OK(final ClassPathXmlApplicationContext classPathXmlApplicationContext) {
		// messageSourceAccessor フィールドの値を設定しておく
        Deencapsulation.setField(service, "messageSourceAccessor", null);

		final List<Policy> policies = new ArrayList<Policy>();
		Policy policy = new Policy("realm", "policy");
		Resource resource = new Resource();
		resource.setResourceURL("http://www.example.com/");
		resource.setGetPermitValue("ALLOW");
		resource.setPostPermitValue("DENY");
		resource.setPolicy(policy);
		policy.addResource(resource);
		Subject subject = new Subject("subject", "id=subject,ou=group,o=realm,ou=services,dc=opensso,dc=java,dc=net");
		subject.setPolicy(policy);
		policy.addSubject(subject);
		policies.add(policy);
		
		final String message = new String("log messages");
		
		Realm testRealm = new Realm("realm");
		testRealm.addReferral("http://www.example.com/");
		testRealm.addGroup("subject", "id=subject,ou=group,o=realm,ou=services,dc=opensso,dc=java,dc=net");
		final List<Realm> testRealmList = new ArrayList<Realm>();
		testRealmList.add(testRealm);
		
		new Expectations() {
			{
				excelPolicyReader.read("filename");
				result = policies;
				
				// messageSourceAccessor を初めて使用するときの動作↓ここから
				classPathXmlApplicationContext.containsBean(anyString);
				result = true;
				classPathXmlApplicationContext.getBean("messageSourceAccessor");
				result = messageSourceAccessor;
				// messageSourceAccessor を初めて使用するときの動作↓ここまで
				
				messageSourceAccessor.getMessage("view.importService.singleValidationLoopStart", (Object[])any);
				result = message;
				messageSourceAccessor.getMessage("view.importService.singleValidationLoopEnd", (Object[])any);
				result = message;
				
				messageSourceAccessor.getMessage("view.importService.interrelateValidationLoopStart", (Object[])any);
				result = message;
				classPathXmlApplicationContext.getBean(JpaPolicyRepositoryImpl.class);
				result = policyRepository;
				classPathXmlApplicationContext.getBean(JpaRealmRepositoryImpl.class);
				result = realmRepository;
				
				realmRepository.findAll();
				result = testRealmList;
				
				policyRepository.findAll();
				result = new ArrayList<Policy>();
				
				messageSourceAccessor.getMessage("view.importService.interrelateValidationLoopEnd", (Object[])any);
				result = message;
				
				policyRepository.importPolicies(policies);
			}
		};
		
		service.importPolicy("filename");
	}
	
	@Test
	public void testImportPolicy_SingleERROR(final ClassPathXmlApplicationContext classPathXmlApplicationContext) {
		// messageSourceAccessor フィールドの値を設定しておく
        Deencapsulation.setField(service, "messageSourceAccessor", messageSourceAccessor);

		final List<Policy> policies = new ArrayList<Policy>();
		Policy policy = new Policy("realm", "policyあ");
		Resource resource = new Resource();
		resource.setResourceURL("http://www.example.com/");
		resource.setGetPermitValue("ALLOW");
		resource.setPostPermitValue("DENY");
		resource.setPolicy(policy);
		policy.addResource(resource);
		Subject subject = new Subject("subject", "id=subject,ou=group,o=realm,ou=services,dc=opensso,dc=java,dc=net");
		subject.setPolicy(policy);
		policy.addSubject(subject);
		policies.add(policy);
		
		final String message = new String("log messages");

		new Expectations() {
			{
				excelPolicyReader.read("filename");
				result = policies;
				
				messageSourceAccessor.getMessage("view.importService.singleValidationLoopStart", (Object[])any);
				result = message;
				messageSourceAccessor.getMessage("view.importService.singleValidationError", (Object[])any);
				result = message;
				messageSourceAccessor.getMessage("view.importService.singleValidationLoopEnd", (Object[])any);
				result = message;
				
				new MessageInfo("E-0301", (Object[])any);
			}
		};
		
		try {
			service.importPolicy("filename");
			fail();
		} catch (ExcelPolicyImportException e) {
			
		}
	}
	
	@Test
	public void testImportPolicy_InterrelateERROR(final ClassPathXmlApplicationContext classPathXmlApplicationContext) {
		// messageSourceAccessor フィールドの値を設定しておく
        Deencapsulation.setField(service, "messageSourceAccessor", messageSourceAccessor);

		final List<Policy> policies = new ArrayList<Policy>();
		Policy policy = new Policy("realm", "policy");
		Resource resource = new Resource();
		resource.setResourceURL("http://www.google.co.jp/index.html");
		resource.setGetPermitValue("ALLOW");
		resource.setPostPermitValue("DENY");
		resource.setPolicy(policy);
		policy.addResource(resource);
		Subject subject = new Subject("subject", "id=subject,ou=group,o=realm,ou=services,dc=opensso,dc=java,dc=net");
		subject.setPolicy(policy);
		policy.addSubject(subject);
		policies.add(policy);
		
		final String message = new String("log messages");

		new Expectations() {
			{
				excelPolicyReader.read("filename");
				result = policies;
				messageSourceAccessor.getMessage("view.importService.singleValidationLoopStart", (Object[])any);
				result = message;
				messageSourceAccessor.getMessage("view.importService.singleValidationLoopEnd", (Object[])any);
				result = message;
				
				messageSourceAccessor.getMessage("view.importService.interrelateValidationLoopStart", (Object[])any);
				result = message;
				classPathXmlApplicationContext.getBean(JpaPolicyRepositoryImpl.class);
				result = policyRepository;
				classPathXmlApplicationContext.getBean(JpaRealmRepositoryImpl.class);
				result = realmRepository;
				
				realmRepository.findAll();
				result = new ArrayList<Realm>();
				
				policyRepository.findAll();
				result = new ArrayList<Policy>();
				
				messageSourceAccessor.getMessage("view.importService.interrelateValidationError", (Object[])any);
				result = message;
				messageSourceAccessor.getMessage("view.importService.interrelateValidationError", (Object[])any);
				result = message;
				messageSourceAccessor.getMessage("view.importService.interrelateValidationError", (Object[])any);
				result = message;
				
				messageSourceAccessor.getMessage("view.importService.interrelateValidationLoopEnd", (Object[])any);
				result = message;
				
				new MessageInfo("E-0301", (Object[])any);
			}
		};
		
		try {
			service.importPolicy("filename");
			fail();
		} catch (ExcelPolicyImportException e) {
		}
		
		// Policy の Resource の postPermit, getPermit が設定されているか確認
		assertThat(resource.getGetPermit(), is(Permit.ALLOW));
		assertThat(resource.getPostPermit(), is(Permit.DENY));
	}
	
	static class TestPolicyRepository implements PolicyRepository {
		public TestPolicyRepository() {
		}

		@Override
		public Policy findById(Long id) {
			return null;
		}

		@Override
		public List<Policy> findAll() {
			return null;
		}

		@Override
		public void persist(Policy entity) {
		}

		@Override
		public Policy merge(Policy entity) {
			return null;
		}

		@Override
		public void remove(Policy entity) {
		}

		@Override
		public void removeAll() {
		}

		@Override
		public void reflesh(Policy entity) {
		}

		@Override
		public void sync(String realmName) {
		}

		@Override
		public List<Policy> findPoliciesByRealmName(String realmName) {
			return null;
		}

		@Override
		public List<Policy> findPolicies(String realmName, String subjectName,
				String url) {
			return null;
		}

		@Override
		public void importPolicies(List<Policy> policies) {
		}
	}
}
