/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.http;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;

import java.io.IOException;

import mockit.Expectations;
import mockit.Mocked;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.ProtocolVersion;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicStatusLine;
import org.junit.Before;
import org.junit.Test;

public class HttpResponseWrapperTest {
    @Mocked
    private HttpResponse httpResponse;

    private HttpResponseWrapper httpResponseWrapper;
    
    @Before
    public void setUp() throws Exception {
        final HttpEntity entity = new StringEntity("xyz");
        new Expectations() {
            {
                httpResponse.getEntity(); result = entity;
                httpResponse.getEntity(); result = entity;
            }
        };

        httpResponseWrapper = new HttpResponseWrapper(httpResponse);
    }

    @Test
    public void testGetContent() {
        byte[] actual = httpResponseWrapper.getContent();

        assertThat(actual.length, is(3));
        
        assertThat(actual[0], is((byte)'x'));
        assertThat(actual[1], is((byte)'y'));
        assertThat(actual[2], is((byte)'z'));
    }

    @Test
    public void testGetHttpResponse() {
        HttpResponse actual = httpResponseWrapper.getHttpResponse();
        
        assertThat(actual, is(httpResponse));
    }

    @Test
    public void testGetProtocolVersion() throws ClientProtocolException,
            IOException {
        new Expectations() {
            {
                httpResponse.getProtocolVersion();
                result = new ProtocolVersion("A", 123, 456);
            }
        };

        String result = httpResponseWrapper.getProtocolVersion();
        
        assertThat(result, is("A/123.456"));
    }

    @Test
    public void testGetHttpStatusCode() {
        new Expectations() {
            {
                httpResponse.getStatusLine();
                result = new BasicStatusLine(new ProtocolVersion("A", 123, 456), 92, "rrr");
            }
        };

        int result = httpResponseWrapper.getHttpStatusCode();
        
        assertThat(result, is(92));
    }

    @Test
    public void testGetReasonPhrase() {
        new Expectations() {
            {
                httpResponse.getStatusLine();
                result = new BasicStatusLine(new ProtocolVersion("A", 123, 456), 92, "abcxyz");
            }
        };

        String result = httpResponseWrapper.getReasonPhrase();
        
        assertThat(result, is("abcxyz"));
    }

    @Test
    public void testGetHeaders() {
        final Header header = new BasicHeader("name_a", "value_a");
        final Header[] headers = new Header[] {header};
        final String name = "ttt";
        
        new Expectations() {
            {
                httpResponse.getHeaders(name);
                result = headers;
            }
        };

        Header[] result = httpResponseWrapper.getHeaders(name);
        
        assertThat(result.length, is(1));
        assertThat(result[0], is(header));
    }
}
