/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.excel.impl;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.CommonConstants;
import jp.co.ogis_ri.citk.policytool.common.excel.ExcelPolicyReader;
import jp.co.ogis_ri.citk.policytool.common.excel.impl.model.ExcelPolicy;
import jp.co.ogis_ri.citk.policytool.common.exception.ExcelPolicyReadException;
import jp.co.ogis_ri.citk.policytool.common.exception.SystemException;
import jp.co.ogis_ri.citk.policytool.common.resource.MessageInfo;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Resource;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Subject;
import mockit.Delegate;
import mockit.Expectations;
import mockit.Mocked;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.OfficeXmlFileException;
import org.apache.poi.ss.usermodel.Workbook;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration({ "ExcelPolicyReaderImplTest-context.xml" })
public class ExcelPolicyReaderImplTest {
    /**
     * ポリシーエクスポートクラス
     */
    @Autowired
	@Qualifier("excelPolicyReaderImpl") 
    private ExcelPolicyReader excelPolicyWriter = null;
    
	@Mocked
	private WorkbookWrapper workbookWrapper;
	
	@Test
	public void testRead() throws IOException {
		final String fileName = "test.txt";
		final List<Object> values = new ArrayList<Object>();
		values.add("realmName");
		values.add("policyName");
		values.add("resourceUrl");
		values.add("getPermitValue");
		values.add("postPermitValue");
		values.add("id=subject,o=abc");
		
		new Expectations() {
			{
				WorkbookWrapper.createHSSFWorkbook(fileName);
				new WorkbookWrapper((Workbook)any);
				workbookWrapper.selectSheet(CommonConstants.TMPL_SHEET_INDEX, CommonConstants.TMPL_SHEET_ROW_OFFSET, CommonConstants.TMPL_SHEET_COLUMN_OFFSET);
				workbookWrapper.existRow();
				result = true;
				workbookWrapper.readRow(ExcelPolicy.ITEM_SIZE);
				result = values;
				workbookWrapper.existRow();
				result = false;
			}
		};
		
		List<Policy> actual = excelPolicyWriter.read(fileName);
		
		assertThat(actual.size(), is(1));
		Policy actualPolicy = actual.get(0);
		assertThat(actualPolicy.getRealmName(), is("realmName"));
		assertThat(actualPolicy.getPolicyName(), is("policyName"));
		assertThat(actualPolicy.getResources().size(), is(1));
		Resource actualResource = actualPolicy.getResources().get(0);
		assertThat(actualResource.getResourceUrl(), is("resourceUrl"));
		assertThat(actualResource.getGetPermitValue(), is("getPermitValue"));
		assertThat(actualResource.getPostPermitValue(), is("postPermitValue"));
		assertThat(actualPolicy.getSubjects().size(), is(1));
		Subject actualSubject = actualPolicy.getSubjects().get(0);
		assertThat(actualSubject.getSubjectCode(), is("id=subject,o=abc"));
	}
	
	@Test
	public void testRead_IOException(final MessageInfo messageInfo) throws IOException {
		final String fileName = "test.txt";
		final List<Object> values = new ArrayList<Object>();
		values.add("realmName");
		values.add("policyName");
		values.add("resourceUrl");
		values.add("getPermitValue");
		values.add("postPermitValue");
		values.add("id=subject,o=abc");
		
		new Expectations() {
			{
				WorkbookWrapper.createHSSFWorkbook(fileName);
				result = new Delegate() {
					@SuppressWarnings("unused")
					HSSFWorkbook createHSSFWorkbook(String fileName) throws IOException {
						throw new IOException();
					}
				};
			}
		};
		
		try {
			excelPolicyWriter.read(fileName);
			fail();
		} catch (SystemException e) {
			assertTrue(e.getCause() instanceof IOException);
		}
	}
	
	@Test(expected=ExcelPolicyReadException.class)
	public void testRead_FileNameEmpty(final MessageInfo messageInfo) throws IOException {
		final String fileName = "";
		
		new Expectations() {
			{
				new MessageInfo("E-0006");
			}
		};
		
		excelPolicyWriter.read(fileName);
		fail();
	}

	@Test
	public void testRead_OfficeXmlFileException(final MessageInfo messageInfo) throws IOException {
		final String fileName = "test.txt";
		final List<Object> values = new ArrayList<Object>();
		values.add("realmName");
		values.add("policyName");
		values.add("resourceUrl");
		values.add("getPermitValue");
		values.add("postPermitValue");
		values.add("id=subject,o=abc");
		
		new Expectations() {
			{
				WorkbookWrapper.createHSSFWorkbook(fileName);
				result = new Delegate() {
					@SuppressWarnings("unused")
					HSSFWorkbook createHSSFWorkbook(String fileName) throws IOException {
						throw new OfficeXmlFileException("");
					}
				};
				
				new MessageInfo("E-0007", fileName);
			}
		};
		
		try {
			excelPolicyWriter.read(fileName);
			fail();
		} catch (ExcelPolicyReadException e) {
			assertThat(e.getFileName(), is(fileName));
		}
	}
	
	@Test
	public void testRead_FileNotFoundException(final MessageInfo messageInfo) throws IOException {
		final String fileName = "dummy";
		
		new Expectations() {
			{
				WorkbookWrapper.createHSSFWorkbook(fileName);
				result = new Delegate() {
					@SuppressWarnings("unused")
					HSSFWorkbook createHSSFWorkbook(String fileName) throws IOException {
						throw new FileNotFoundException();
					}
				};
				
				new MessageInfo("E-0008", fileName);
			}
		};
		
		try {
			excelPolicyWriter.read(fileName);
			fail();
		} catch (ExcelPolicyReadException e) {
			assertThat(e.getFileName(), is(fileName));
		}
	}

	@Test
	public void testRead_ClassCastException(final MessageInfo messageInfo) throws IOException {
		final String fileName = "test.txt";
		final List<Object> values = new ArrayList<Object>();
		values.add("realmName");
		values.add("policyName");
		values.add("resourceUrl");
		values.add(new Object());
		values.add("postPermitValue");
		values.add("id=subject,o=abc");
		
		new Expectations() {
			{
				WorkbookWrapper.createHSSFWorkbook(fileName);
				new WorkbookWrapper((Workbook)any);
				workbookWrapper.selectSheet(CommonConstants.TMPL_SHEET_INDEX, CommonConstants.TMPL_SHEET_ROW_OFFSET, CommonConstants.TMPL_SHEET_COLUMN_OFFSET);
				workbookWrapper.existRow();
				result = true;
				workbookWrapper.readRow(ExcelPolicy.ITEM_SIZE);
				result = values;
				
				new MessageInfo("E-0009");
			}
		};
		
		try {
			excelPolicyWriter.read(fileName);
		} catch (ExcelPolicyReadException e) {
			assertThat(e.getFileName(), is(fileName));
		}
	}
}
