/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package jp.co.ogis_ri.citk.policytool.domain.policy.model;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.Table;
import javax.validation.Valid;

import jp.co.ogis_ri.citk.policytool.common.model.AbstractModel;
import jp.co.ogis_ri.citk.policytool.common.validation.annotation.PolicyName;
import jp.co.ogis_ri.citk.policytool.common.validation.annotation.RealmName;

/**
 * ポリシーを表すクラス.
 */
@Entity
@Table(name = "t_policy")
public class Policy extends AbstractModel<Long> {

    /**
     * デフォルト・シリアル･バージョンID
     */
    private static final long serialVersionUID = 1L;

    /**
     * JPAで永続化するためのサロゲートキー
     */
    @Id
    @GeneratedValue
    private Long id;

    /**
     * レルム名
     */
    @RealmName
    String realmName = null;

    /**
     * ポリシー名
     */
    @PolicyName
    String policyName = null;

    /**
     * サブジェクト
     */
    @OneToMany(mappedBy = "policy", fetch = FetchType.LAZY, cascade = {CascadeType.ALL}, orphanRemoval = true)
    @OrderBy(value = "subjectName asc")
    @Valid
    List<Subject> subjects = new ArrayList<Subject>();

    /**
     * リソース
     */
    @OneToMany(mappedBy = "policy", fetch = FetchType.LAZY, cascade = {CascadeType.ALL}, orphanRemoval = true)
    @OrderBy(value = "resourceUrl asc")
    @Valid
    List<Resource> resources = new ArrayList<Resource>();

    /**
     * @return サロゲートキー
     */
    @Override
    public Long getId() {
        return this.id;
    }

    /**
     * サブジェクトのリストを取得する.
     * 
     * @return サブジェクト.
     */
    public List<Subject> getSubjects() {
        return subjects;
    }

    /**
     * このポリシーにサブジェクトを追加する.
     * 
     * @param subject 追加するサブジェクト.
     */
    public void addSubject(Subject subject) {
        subject.setPolicy(this);
        subjects.add(subject);
    }

    /**
     * このポリシーに指定された名前, コードのサブジェクトを追加する.
     * 
     * @param subjectName 追加するサブジェクトの名前.
     * @param subjectCode 追加するサブジェクトのコード.
     */
    public void addSubject(String subjectName, String subjectCode) {
        addSubject(new Subject(subjectName, subjectCode));
    }

    /**
     * リソースのリストを取得する.
     * 
     * @return リソースのリスト.
     */
    public List<Resource> getResources() {
        return resources;
    }

    /**
     * このポリシーにリソースを追加する.
     * 
     * @param resource 追加するリソース.
     */
    public void addResource(Resource resource) {
        resource.setPolicy(this);
        resources.add(resource);
    }

    /**
     * このポリシーに指定されたURL, GET権限, POST権限のリソースを追加する.
     * 
     * @param url URL.
     * @param get GET権限.
     * @param post POST権限.
     */
    public void addResource(String url, Permit get, Permit post) {
        addResource(new Resource(url, get, post));
    }

    /**
     * このポリシーに指定されたURL, GET権限文字列, POST権限文字列のリソースを追加する.
     * 
     * @param url URL.
     * @param getValue GET権限文字列.
     * @param postValue POST権限文字列.
     */
    public void addResource(String url, String getValue, String postValue) {
        addResource(new Resource(url, getValue, postValue));
    }

    /**
     * このポリシーのレルム名を取得する.
     * 
     * @return　このポリシーのレルム名.
     */
    public String getRealmName() {
        return realmName;
    }

    /**
     * このポリシーのレルム名を設定する.
     * 
     * @param realmName レルム名.
     */
    public void setRealmName(String realmName) {
        this.realmName = realmName;
    }

    /**
     * このポリシーの名前を取得する.
     * 
     * @return ポリシー名.
     */
    public String getPolicyName() {
        return policyName;
    }

    /**
     * このポリシーの名前を設定する.
     * 
     * @param policyName ポリシー名.
     */
    public void setPolicyName(String policyName) {
        this.policyName = policyName;
    }

    /**
     * コンストラクタ.
     */
    public Policy() {
    }

    /**
     * コンストラクタ.
     * 
     * @param realmName ポリシーのレルム名.
     * @param policyName ポリシー名.
     */
    public Policy(String realmName, String policyName) {
        this.realmName = realmName;
        this.policyName = policyName;
    }

    /**
     * このポリシーのすべてのリソースの権限enum値に, 権限文字列の情報を反映させる.
     */
    public void resetPermit() {
        for (Resource resource : this.resources) {
            resource.resetPermit();
        }
    }
}
