/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;

import org.apache.commons.logging.Log;

/**
 * ストリーム関連のユーティリティクラス.
 */
public class StreamUtil {
    /**
     * ロガー.
     */
    private static final Log logger =
            LogWrapperFactory.getLog(StreamUtil.class);

    /**
     * コンストラクタ.
     */
    private StreamUtil() {
    }

    /**
     * 指定された InputStream を読み込み、 行ごとにリストに格納する.
     * 
     * @param inputStream 読み込む InputStream.
     * @return 変換後のリスト.
     * 
     */
    public static List<String> toTextLine(InputStream inputStream) {
        List<String> context = new ArrayList<String>();

        InputStreamReader inputStreamReader = null;
        BufferedReader bufferedReader = null;
        try {
            inputStreamReader = new InputStreamReader(inputStream);
            bufferedReader = new BufferedReader(inputStreamReader);
            String line;
            while ((line = bufferedReader.readLine()) != null) {
                context.add(line);
            }
        } catch (IOException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        } finally {
            if (inputStreamReader != null) {
                try {
                    inputStreamReader.close();
                } catch (IOException e) {
                    // NOP
                }
            }
            if (bufferedReader != null) {
                try {
                    bufferedReader.close();
                } catch (IOException e) {
                    // NOP
                }
            }
        }
        return context;
    }

    /**
     * 指定された InputStream を読み込み、文字列に変換する.
     * 
     * @param inputStream 読み込む InputStream.
     * @return 変換後の文字列.
     * 
     */
    public static String toString(InputStream inputStream) {
        InputStreamReader inputStreamReader = null;
        BufferedReader bufferedReader = null;
        StringWriter stringWriter = new StringWriter();
        PrintWriter printWriter = new PrintWriter(stringWriter);
        try {
            inputStreamReader = new InputStreamReader(inputStream);
            bufferedReader = new BufferedReader(inputStreamReader);
            int ch;
            while ((ch = inputStreamReader.read()) != -1) {
                printWriter.print((char) ch);
            }
        } catch (IOException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        } finally {
            if (inputStreamReader != null) {
                try {
                    inputStreamReader.close();
                } catch (IOException e) {
                    // NOP
                }
            }
            if (bufferedReader != null) {
                try {
                    bufferedReader.close();
                } catch (IOException e) {
                    // NOP
                }
            }
            if (stringWriter != null) {
                try {
                    stringWriter.close();
                } catch (IOException e) {
                    // NOP
                }
            }
            if (printWriter != null) {
                printWriter.close();
            }
        }
        return stringWriter.toString();
    }

    /**
     * 指定された InputStream を読み込み、バイト配列に変換する.
     * 
     * @param inputStream 読み込む InputStream.
     * @return 変換後のバイト配列.
     * 
     */
    public static byte[] toByteArray(InputStream inputStream) {
        return toString(inputStream).getBytes();
    }
}
