/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.repository;

import java.io.Serializable;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.model.AbstractModel;

/**
 * エンティティに対する汎用的な処理を行うインタフェース（GenericDaoパターン）.
 * 
 * @author michio
 * 
 * @param <E> エンティティ型
 * @param <K> 主キー型
 */
public interface GenericRepository<E extends AbstractModel<K>, K extends Serializable> {

    /**
     * idでエンティティを検索する.
     * 
     * @param id id
     */
    E findById(K id);

    /**
     * エンティティを全件取得する.
     * 
     * @return エンティティのリスト.
     */
    List<E> findAll();

    /**
     * エンティティを永続化する.
     * 
     * @param entity エンティティ
     */
    void persist(E entity);

    /**
     * エンティティをアタッチ状態にする.
     * 
     * @param entity エンティティ
     */
    E merge(E entity);

    /**
     * エンティティを削除状態にする.
     * 
     * @param entity エンティティ
     */
    void remove(E entity);

    /**
     * 全てのエンティティを削除状態にする.
     * 
     */
    void removeAll();

    /**
     * エンティティをリフレッシュにする.
     * 
     * @param entity エンティティ
     */
    void reflesh(E entity);

}
