/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.http;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;
import jp.co.ogis_ri.citk.policytool.common.util.ExceptionUtil;

import org.apache.commons.logging.Log;
import org.apache.http.Header;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.params.ClientPNames;
import org.apache.http.client.params.CookiePolicy;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.cookie.Cookie;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HTTP;

/**
 * HttpClient のラッパークラス.
 * 
 */
public class HttpClientWrapper {
    /**
     * ロガー.
     */
    private final Log logger =
            LogWrapperFactory.getLog(HttpClientWrapper.class);

    /**
     * HttpClient.
     */
    private DefaultHttpClient httpClient = null;

    /**
     * パラメータのリスト.
     */
    private List<NameValuePair> queryParams = new ArrayList<NameValuePair>();

    /**
     * エンコーディング.
     */
    private String encoding = HTTP.UTF_8;

    /**
     * クッキーのポリシー.
     */
    private String cookiePolicy = CookiePolicy.RFC_2109;

    /**
     * エンコーディングを取得する.
     * 
     * @return エンコーディング.
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * エンコーディングを設定する.
     * 
     * @param encoding エンコーディング.
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    /**
     * クッキーのポリシーを取得する.
     * 
     * @return クッキーのポリシー.
     */
    public String getCookiePolicy() {
        return cookiePolicy;
    }

    /**
     * クッキーのポリシーを設定する.
     * 
     * @param cookiePolicy クッキーのポリシー.
     */
    public void setCookiePolicy(String cookiePolicy) {
        this.cookiePolicy = cookiePolicy;
    }

    /**
     * コンストラクタ.
     */
    public HttpClientWrapper() {
        this.httpClient = new DefaultHttpClient();
    }

    /**
     * コンストラクタ.
     * 
     * @param httpClient ラップする HttpClient.
     */
    public HttpClientWrapper(DefaultHttpClient httpClient) {
        this.httpClient = httpClient;
    }

    /**
     * パラメータを追加する.
     * 
     * @param name パラメータの名前.
     * @param value パラメータの値.
     */
    public void addQueryParam(String name, String value) {
        queryParams.add(new BasicNameValuePair(name, value));
    }

    /**
     * パラメータをすべて削除する.
     */
    public void clearQueryParam() {
        queryParams.clear();
    }

    /**
     * クッキーを追加する.
     * 
     * @param cookie 追加するクッキー.
     */
    public void addCookie(Cookie cookie) {
        httpClient.getCookieStore().addCookie(cookie);
    }

    /**
     * クッキーをすべて削除する.
     */
    public void clearCookieStore() {
        httpClient.getCookieStore().clear();
    }

    /**
     * GETリクエストを行う.
     * 
     * @param urlString URL文字列.
     * @return レスポンスの HttpResponseWrapper.
     */
    public HttpResponseWrapper get(String urlString) {
        URI uri = createURI(urlString);
        HttpGet request = new HttpGet(uri);
        return execute(request);
    }

    /**
     * GETリクエストを行う.
     * 
     * @param url URL オブジェクト.
     * @return レスポンスの HttpResponseWrapper.
     */
    public HttpResponseWrapper get(URL url) {
        URI uri = createURI(url);
        HttpGet request = new HttpGet(uri);
        return execute(request);
    }

    /**
     * POSTリクエストを行う.
     * 
     * @param urlString URL文字列.
     * @return レスポンスの HttpResponseWrapper.
     */
    public HttpResponseWrapper post(String urlString) {
        HttpPost request = new HttpPost(urlString);
        try {
            request.setEntity(new UrlEncodedFormEntity(queryParams, encoding));
        } catch (UnsupportedEncodingException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        }
        return execute(request);
    }

    /**
     * POSTリクエストを行う.
     * 
     * @param uri URI オブジェクト.
     * @return レスポンスの HttpResponseWrapper.
     */
    public HttpResponseWrapper post(URI uri) {
        HttpPost request = new HttpPost(uri);
        try {
            request.setEntity(new UrlEncodedFormEntity(queryParams, encoding));
        } catch (UnsupportedEncodingException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        }
        return execute(request);
    }

    /**
     * HttpClient の接続を切断する.
     */
    public void shutdown() {
        httpClient.getConnectionManager().shutdown();
    }

    /**
     * URL文字列から URIオブジェクトを生成する.
     * 
     * @param urlString URL文字列.
     * @return URIオブジェクト.
     */
    private URI createURI(String urlString) {
        URI uri = null;
        try {
            uri = createURI(new URL(urlString));
        } catch (MalformedURLException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        }
        return uri;
    }

    /**
     * URLオブジェクトからURIオブジェクトを生成する.
     * 
     * @param url URLオブジェクト.
     * @return URIオブジェクト.
     */
    private URI createURI(URL url) {
        URI uri = null;
        try {
            uri =
                    URIUtils.createURI(
                            url.getProtocol(),
                            url.getHost(),
                            url.getPort(),
                            url.getPath(),
                            !queryParams.isEmpty() ? URLEncodedUtils.format(
                                    queryParams, encoding)
                                    : (url.getQuery() != null ? url.getQuery()
                                            : ""), null);
        } catch (URISyntaxException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        }
        return uri;
    }

    /**
     * HttpUriRequestのリクエスト処理を行う.
     * 
     * @param request HttpUriRequest.
     * @return レスポンスの HttpResponseWrapper.
     */
    private HttpResponseWrapper execute(HttpUriRequest request) {
        HttpResponseWrapper response = null;
        try {
            logger.debug("Request Line:" + request.getRequestLine().toString());

            // クッキーポリシーの設定
            request.getParams().setParameter(ClientPNames.COOKIE_POLICY,
                    cookiePolicy);
            // リクエスト
            response = new HttpResponseWrapper(httpClient.execute(request));

            logger.debug("HTTP Status Code:"
                    + String.valueOf(response.getHttpStatusCode()));
            if (logger.isDebugEnabled()) {
                for (Header h : response.getHttpResponse().getAllHeaders()) {
                    logger.debug("HTTP HEADER - " + h.getName() + "="
                            + h.getValue());
                }
                List<Cookie> cookies = httpClient.getCookieStore().getCookies();
                if (cookies.isEmpty()) {
                    logger.debug("COOKIES - None");
                } else {
                    for (int i = 0; i < cookies.size(); i++) {
                        logger.debug("COOKIES - " + cookies.get(i).toString());
                    }
                }
            }

        } catch (ClientProtocolException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        } catch (IOException e) {
            logger.debug(e.getMessage());
            throw ExceptionUtil.convertRuntimeException(e);
        } finally {
            clearQueryParam();
        }
        return response;
    }
}
