/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.view.common;

import java.awt.Component;

import javax.swing.JOptionPane;
import javax.validation.ConstraintViolation;

import jp.co.ogis_ri.citk.policytool.common.CommonConstants;
import jp.co.ogis_ri.citk.policytool.common.exception.ApplicationException;
import jp.co.ogis_ri.citk.policytool.common.exception.SystemException;
import jp.co.ogis_ri.citk.policytool.common.util.ApplicationContextUtil;

import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.core.NestedRuntimeException;

/**
 * メッセージダイアログ.
 * 
 */
public class MessageDialog {

    /**
     * コンストラクタ.
     */
    private MessageDialog() {
    }

    /**
     * 確認ダイアログを表示する.
     * 
     * @param parent 親コンポーネント.
     * @param code メッセージコード.
     * @param args プレースフォルダの値.
     * @return YesまたはNo
     */
    public static int showConfirm(Component parent, String code, Object... args) {
        MessageSourceAccessor message = ApplicationContextUtil.getResources();
        return JOptionPane.showConfirmDialog(parent,
                message.getMessage(code, args), getMessageTitle(code),
                JOptionPane.YES_NO_OPTION, getMessageType(code));

    }

    /**
     * メッセージダイアログを表示する.
     * 
     * @param parent 親コンポーネント.
     * @param code メッセージコード.
     * @param args プレースフォルダの値.
     */
    public static void showMessage(Component parent, String code,
            Object... args) {
        MessageSourceAccessor message = ApplicationContextUtil.getResources();
        JOptionPane.showMessageDialog(parent, message.getMessage(code, args),
                getMessageTitle(code), getMessageType(code));

    }

    /**
     * バイオレーションメッセージダイアログを表示する.
     * 
     * @param parent 親コンポーネント.
     * @param violation バイオレーション.
     */
    public static void showViolationMessage(Component parent,
            ConstraintViolation<?> violation) {
        MessageSourceAccessor message = ApplicationContextUtil.getResources();
        JOptionPane.showMessageDialog(parent, violation.getMessage(),
                message.getMessage(CommonConstants.DIALOG_TITLE_CONFIRM),
                JOptionPane.INFORMATION_MESSAGE);
    }

    /**
     * アプリケーション例外ダイアログを表示する.
     * 
     * @param e アプリケーション例外.
     */
    public static void showApplicationException(ApplicationException e) {
        String code = e.getMessageInfo().getCode();
        JOptionPane.showMessageDialog(null, e.getMessage(),
                getMessageTitle(code), getMessageType(code));
    }

    /**
     * システム例外ダイアログを表示する.
     * 
     * @param e システム例外.
     */
    public static void showSystemException(SystemException e) {
        MessageSourceAccessor message = ApplicationContextUtil.getResources();
        JOptionPane.showMessageDialog(null, e.getMessage(),
                message.getMessage(CommonConstants.DIALOG_TITLE_ERROR),
                JOptionPane.ERROR_MESSAGE);
    }

    /**
     * Spring例外ダイアログを表示する.
     * 
     * @param e spring例外.
     */
    public static void showNestedRuntimeException(NestedRuntimeException e) {
        JOptionPane.showMessageDialog(null, e.getRootCause(), "ERROR",
                JOptionPane.ERROR_MESSAGE);
    }

    /**
     * 例外ダイアログを表示する.
     * 
     * @param e 例外.
     */
    public static void showException(Throwable e) {
        JOptionPane.showMessageDialog(null, e, "ERROR",
                JOptionPane.ERROR_MESSAGE);
    }

    /**
     * 指定されたメッセージ コード に対するメッセージボックスのタイトル文字列を取得する.
     * 
     * @param code メッセージコード.
     * @return タイトル文字列.
     */
    private static String getMessageTitle(String code) {
        MessageSourceAccessor message = ApplicationContextUtil.getResources();
        if (code.startsWith("E")) {
            return message.getMessage(CommonConstants.DIALOG_TITLE_ERROR);
        } else if (code.startsWith("W")) {
            return message.getMessage(CommonConstants.DIALOG_TITLE_WARN);
        } else if (code.startsWith("I")) {
            return message.getMessage(CommonConstants.DIALOG_TITLE_INFO);
        } else if (code.startsWith("C")) {
            return message.getMessage(CommonConstants.DIALOG_TITLE_CONFIRM);
        } else {
            return null;
        }
    }

    /**
     * 指定されたメッセージコードに対するメッセージボックスのアイコン種別を取得する.
     * 
     * @param code メッセージコード.
     * @return アイコン種別.
     */
    private static int getMessageType(String code) {
        if (code.startsWith("E")) {
            return JOptionPane.ERROR_MESSAGE;
        } else if (code.startsWith("W")) {
            return JOptionPane.WARNING_MESSAGE;
        } else if (code.startsWith("I")) {
            return JOptionPane.INFORMATION_MESSAGE;
        } else if (code.startsWith("C")) {
            return JOptionPane.QUESTION_MESSAGE;
        } else {
            return Integer.MIN_VALUE;
        }
    }

}
