/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.excel.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import jp.co.ogis_ri.citk.policytool.common.excel.impl.model.ExcelPolicy;
import jp.co.ogis_ri.citk.policytool.common.util.OpenAMNamingUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;

/**
 * ExcelPolicyからPolicyを作成するクラス.
 * 
 */
class PolicyBuilder {
    /**
     * Policyの作成に使用するExcelPolicyのリスト.
     */
    private List<ExcelPolicy> excelPolicies = new ArrayList<ExcelPolicy>();

    /**
     * Policy作成元ExcelPolicyリストにExcelPolicyを追加する.
     * 
     * @param excelPolicy 追加するExcelPolicy.
     */
    public void add(ExcelPolicy excelPolicy) {
        excelPolicies.add(excelPolicy);

    }

    /**
     * 現在のExcelPolicyのリストから, Policyを作成する.
     * 
     * @return 作成されたPolicyのリスト.
     */
    public List<Policy> build() {
        Map<String, Policy> policies = new TreeMap<String, Policy>();
        for (ExcelPolicy excelPolicy : excelPolicies) {

            Policy policy;
            String key = getKey(excelPolicy);
            if (policies.containsKey(key)) {
                policy = policies.get(key);
            } else {
                policy = new Policy();
                policies.put(key, policy);
            }
            policy.setRealmName(excelPolicy.getRealmName());
            policy.setPolicyName(excelPolicy.getPolicyName());

            if (excelPolicy.getResourceUrl() != null
                    && excelPolicy.getGetPermitValue() != null
                    && excelPolicy.getPostPermitValue() != null) {
                policy.addResource(excelPolicy.getResourceUrl(),
                        excelPolicy.getGetPermitValue(),
                        excelPolicy.getPostPermitValue());
            }
            if (excelPolicy.getSubject() != null) {
                Map<String, String> kv =
                        OpenAMNamingUtil.splitValue(excelPolicy.getSubject());
                policy.addSubject(kv.get("id"), excelPolicy.getSubject());
            }
        }

        return new ArrayList<Policy>(policies.values());
    }

    /**
     * Policy作成の内部処理で使用する, Policyのキー文字列を取得する.
     * 
     * @param excelPolicy キー文字列取得元のExcelPolicy.
     * @return Policyのキー文字列.
     */
    private String getKey(ExcelPolicy excelPolicy) {
        return excelPolicy.getRealmName() + "_" + excelPolicy.getPolicyName();
    }
}
