/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.domain.policy.impl;

import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNull.notNullValue;
import static org.hamcrest.core.IsNull.nullValue;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.api.OpenAMAccess;
import jp.co.ogis_ri.citk.policytool.common.exception.ApplicationException;
import jp.co.ogis_ri.citk.policytool.common.resource.MessageInfo;
import jp.co.ogis_ri.citk.policytool.common.util.ApplicationContextUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.PolicyRepository;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Permit;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Resource;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Subject;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Realm;
import jp.co.ogis_ri.citk.policytool.domain.util.DbUnitDataLoader;
import mockit.Deencapsulation;
import mockit.Delegate;
import mockit.Expectations;
import mockit.Mocked;

import org.apache.commons.dbcp.BasicDataSource;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.transaction.TransactionConfiguration;
import org.springframework.transaction.annotation.Transactional;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration({"JpaPolicyRepositoryImplTest-context.xml"})
@TransactionConfiguration(defaultRollback=true)
@Transactional
public class JpaPolicyRepositoryImplTest {

	@Autowired
	BasicDataSource datasource;

    @Autowired
    private PolicyRepository repository;

	@Mocked(capture=1)
	OpenAMAccess amAccess = null;
	
    /**
     * (ここでメソッドの説明を簡単に書く。)
     *
     * @throws java.lang.Exception
     *
     */
	@Before
	public void setUp() throws Exception {
        Deencapsulation.setField(ApplicationContextUtil.class, "context", null);
        Deencapsulation.setField(
                ApplicationContextUtil.class,
                "FILE_APPLICATION_CONTEXT",
                "jp/co/ogis_ri/citk/policytool/domain/policy/impl/JpaPolicyRepositoryImplTest-context.xml");
		
        InputStream policyTestData = Realm.class.getResourceAsStream("/policy.db.xml");
        DbUnitDataLoader policyLoader = new DbUnitDataLoader(policyTestData, datasource.getConnection());
        policyLoader.populateTestData();
		
        InputStream realmTestData = Realm.class.getResourceAsStream("/realm.db.xml");
        DbUnitDataLoader realmLoader = new DbUnitDataLoader(realmTestData, datasource.getConnection());
        realmLoader.populateTestData();
	}

	@Test
	public void testSync() {
		final List<Policy> amAccessPolicyList = new ArrayList<Policy>();
		new Expectations() {
			{
				amAccess.createPolicy((Policy)any);
				result = new Delegate() {
					@SuppressWarnings("unused")
					void createPolicy(Policy policy) {
						amAccessPolicyList.add(policy);
					}
				};
				
				amAccess.getPolicies(anyString, anyString);
				result = new Delegate() {
					@SuppressWarnings("unused")
					List<Policy> getPolicies(String realmName, String policyName) {
						List<Policy> policies = new ArrayList<Policy>();
						if (realmName.equals("sp1")) {
							policies.addAll(amAccessPolicyList);
						}
						return policies;
					}
				};
			}
		};
		
		Policy policy = createSyncPolicy();
        amAccess.createPolicy(policy);
        
    	repository.sync("sp1");
		
    	List<Policy> actual = repository.findPoliciesByRealmName("sp1");
    	assertThat(actual.size(), is(1));
    	assertThat(actual.get(0).getPolicyName(), is("policyName123"));
	}
	
	@Test
	public void testSync_ConstraintViolationException(final MessageInfo messageInfo) {
		final List<Policy> amAccessPolicyList = new ArrayList<Policy>();
		new Expectations() {
			{
				amAccess.createPolicy((Policy)any);
				result = new Delegate() {
					@SuppressWarnings("unused")
					void createPolicy(Policy policy) {
						amAccessPolicyList.add(policy);
					}
				};
				
				amAccess.getPolicies(anyString, anyString);
				result = new Delegate() {
					@SuppressWarnings("unused")
					List<Policy> getPolicies(String realmName, String policyName) {
						List<Policy> policies = new ArrayList<Policy>();
						if (realmName.equals("sp1")) {
							policies.addAll(amAccessPolicyList);
						}
						return policies;
					}
				};
				
				new MessageInfo("I-0007", any);
			}
		};
		
		Policy policy = createSyncPolicy();
		policy.setPolicyName("あ");
        amAccess.createPolicy(policy);

        try {
        	repository.sync("sp1");
        	fail();
        } catch (ApplicationException e) {
        }
	}
	
    /**
     * {@link jp.co.ogis_ri.citk.policytool.common.repository.JpaGenericRepository#findAll()} のためのテスト・メソッド。
     */
    @Test
    public void testFindAll() {
        List<Policy> actual = repository.findAll();
        assertThat(actual.size(), is(3));
    }

    /**
     * {@link jp.co.ogis_ri.citk.policytool.common.repository.JpaGenericRepository#findById(java.io.Serializable)} のためのテスト・メソッド。
     */
    @Test
    public void testFindById_SUCCESS() {
        Policy actual = repository.findById(100L);
        assertThat(actual, is(notNullValue()));
        assertThat(actual.getId(), is(100L));
        assertThat(actual.getSubjects().size(), is(2));
        assertThat(actual.getResources().size(), is(2));
    }

    /**
     * {@link jp.co.ogis_ri.citk.policytool.common.repository.JpaGenericRepository#findById(java.io.Serializable)} のためのテスト・メソッド。
     */
    @Test
    public void testFindById_SUCCESS_NOT_FOUND() {
        Policy actual = repository.findById(999L);
        assertThat(actual, is(nullValue()));
    }

    /**
     * {@link jp.co.ogis_ri.citk.policytool.common.repository.JpaGenericRepository#persist(jp.co.ogis_ri.citk.policytool.common.model.AbstractModel)} のためのテスト・メソッド。
     */
    @Test
    public void testPersist_SUCCESS() {
        Policy policy = new Policy("new_sp", "new_policy");
        policy.addSubject("g1", "o=g1,a=2");
        policy.addResource("http://localhost:8080/index1.jsp", Permit.ALLOW, Permit.ALLOW);
        policy.addResource("http://localhost:8080/index2.jsp", Permit.ALLOW, Permit.DENY);
        repository.persist(policy);
        
        final Policy actual = repository.findById(policy.getId());
        
        assertThat(actual.getId(), is(policy.getId()));
        assertThat(actual.getSubjects().size(), is(policy.getSubjects().size()));
        assertThat(actual.getResources().size(), is(policy.getResources().size()));
    }

    /**
     * {@link jp.co.ogis_ri.citk.policytool.common.repository.JpaGenericRepository#merge(jp.co.ogis_ri.citk.policytool.common.model.AbstractModel)} のためのテスト・メソッド。
     */
    @Test
    public void testMerge_SUCCESS() {
        Policy policy = repository.findById(100L);
        policy.addSubject("g1", "o=g1,a=2");
        policy.addResource("http://localhost:8080/index1.jsp", Permit.ALLOW, Permit.ALLOW);
        policy.addResource("http://localhost:8080/index2.jsp", Permit.ALLOW, Permit.DENY);
        repository.merge(policy);
        
        final Policy actual = repository.findById(100L);
        
        assertThat(actual.getId(), is(policy.getId()));
        assertThat(actual.getSubjects().size(), is(policy.getSubjects().size()));
        assertThat(actual.getResources().size(), is(policy.getResources().size()));
    }

    /**
     * {@link jp.co.ogis_ri.citk.policytool.common.repository.JpaGenericRepository#remove(jp.co.ogis_ri.citk.policytool.common.model.AbstractModel)} のためのテスト・メソッド。
     */
    @Test
    public void testRemove_SUCCESS() {
        Policy policy = repository.findById(100L);
        repository.remove(policy);
        
        Policy actual = repository.findById(100L);
        assertThat(actual, is(nullValue()));
    }
    
    /**
     * {@link jp.co.ogig_ri.citk.policytool.common.repository.JpaGenericRepository#removeAll()} のためのテスト・メソッド。
     */
    @Test
    public void testRemoveAll_SUCCESS() {
    	List<Policy> policies = repository.findAll();
    	assertNotSame(policies.size(), 0);
    	repository.removeAll();
    	
    	List<Policy> actual = repository.findAll();
    	assertThat(actual.size(), is(0));
    }

    /**
     * {@link jp.co.ogis_ri.citk.policytool.domain.policy.impl.JpaPolicyRepositoryImpl#findPoliciesByRealmName(java.lang.String)} のためのテスト・メソッド。
     */
    @Test
    public void testFindPoliciesByRealmName_SUCCESS() {

    	List<Policy> actual = repository.findPoliciesByRealmName("sp1");
    	assertThat(actual.size(), is(2));
    }
    
    @Test
    public void testImportPolicies_SUCCESS() {
    	List<Policy> policies = new ArrayList<Policy>();
    	final Policy policy1 = createImportPolicy1();
    	policies.add(policy1);
    	final Policy policy2 = createImportPolicy2();
    	policies.add(policy2);
    	final Policy policy3 = createImportPolicy3();
    	policies.add(policy3);

    	new Expectations() {
    		{
	    		amAccess.updatePolicy(policy1);
	    		amAccess.createPolicy(policy2);
	    		amAccess.createPolicy(policy2);
    		}
    	};
    	
    	repository.importPolicies(policies);
    }
    
    @Test
    public void testImportPolicies_SUCCESS_PolicyZERO() {
    	List<Policy> policies = new ArrayList<Policy>();

    	repository.importPolicies(policies);
    }
    
    /**
     * Sync のテストで使用する Policy の作成.
     * @return Sync のテストで使用する Policy.
     */
    private Policy createSyncPolicy() {
		Policy policy = new Policy();
		policy.setPolicyName("policyName123");
		policy.setRealmName("sp1");
		Subject subject = new Subject();
		subject.setSubjectName("group1");
		subject.setSubjectCode("id=group1,ou=group,o=sp1,ou=services,dc=opensso,dc=java,dc=net");
		subject.setPolicy(policy);
		policy.getSubjects().add(subject);
		Resource resource = new Resource();
		resource.setResourceURL("http://localhost:8000/");
		resource.setGetPermit(Permit.ALLOW);
		resource.setGetPermitValue("ALLOW");
		resource.setPostPermit(Permit.DENY);
		resource.setPostPermitValue("DENY");
		resource.setPolicy(policy);
		policy.getResources().add(resource);

		return policy;
    }
    
    /**
     * import のテストで使用する Policy の作成.<br>
     * ポリシー名 policy1 レルム名 sp1
     * @return import のテストで使用する Policy.
     */
    private Policy createImportPolicy1() {
    	Policy policy1 = new Policy();
    	policy1.setPolicyName("policy1");
    	policy1.setRealmName("sp1");

		return policy1;
    }
    
    /**
     * import のテストで使用する Policy の作成.<br>
     * ポリシー名 policy2 レルム名 sp1
     * @return import のテストで使用する Policy.
     */
    private Policy createImportPolicy2() {
    	Policy policy = new Policy();
    	policy.setPolicyName("policy2");
    	policy.setRealmName("sp1");

		return policy;
    }
    
    /**
     * import のテストで使用する Policy の作成.<br>
     * ポリシー名 policy1 レルム名 sp2
     * @return import のテストで使用する Policy.
     */
    private Policy createImportPolicy3() {
    	Policy policy1 = new Policy();
    	policy1.setPolicyName("policy1");
    	policy1.setRealmName("sp2");

		return policy1;
    }
}
