/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.common.util;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.junit.matchers.JUnitMatchers.containsString;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.exception.SystemException;
import mockit.Delegate;
import mockit.Expectations;

import org.junit.Test;

public class StreamUtilTest {
    private final String EXPECTED = "あいうえおABC123\nなにぬねの\r\n１２３\n";
    
    @Test
    public void testToTextLineSuccess() throws IOException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(outputStream));
        bufferedWriter.write(EXPECTED);
        bufferedWriter.flush();
        bufferedWriter.close();

        byte[] byteArray = outputStream.toByteArray();
        ByteArrayInputStream inputStream = new ByteArrayInputStream(byteArray);
        List<String> result = StreamUtil.toTextLine(inputStream);
        
        assertThat(result.size(), is(3));
        assertThat(result.get(0), containsString("あいうえおABC123"));
        assertThat(result.get(1), containsString("なにぬねの"));
        assertThat(result.get(2), containsString("１２３"));

        result = StreamUtil.toTextLine(inputStream);
        assertThat(result.size(), is(0));
        inputStream.close();
    }

    @Test
    public void testToTextLineIOException(final InputStreamReader inputStreamReader, final BufferedReader bufferedReader, final ExceptionUtil exceptionUtil) throws IOException {
        byte[] byteArray = new byte[0];
        final ByteArrayInputStream inputStream = new ByteArrayInputStream(byteArray);
    
        new Expectations() {
        	{
        		new InputStreamReader((InputStream)any);
        		new BufferedReader((Reader)any);
        		
        		bufferedReader.readLine();
        		result = new Delegate() {
        			@SuppressWarnings("unused")
					void readLine() throws IOException {
        				throw new IOException();
        			}
        		};
        		
        		ExceptionUtil.convertRuntimeException((Exception)any);
        		result = new SystemException();
        		
        		inputStreamReader.close();
        		bufferedReader.close();
        	}
        };
        
        try {
        	StreamUtil.toTextLine(inputStream);
        	fail();
        } catch (SystemException e) {
        	
        }
    }

    @Test
    public void testToStringSuccess() throws IOException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(outputStream));
        bufferedWriter.write(EXPECTED);
        bufferedWriter.flush();
        bufferedWriter.close();

        byte[] byteArray = outputStream.toByteArray();
        ByteArrayInputStream inputStream = new ByteArrayInputStream(byteArray);
        String result = StreamUtil.toString(inputStream);
        
        assertThat(result, is(EXPECTED));
        result = StreamUtil.toString(inputStream);
        assertThat(result.length(), is(0));
        inputStream.close();
    }

    @Test
    public void testToStringIOException(final InputStreamReader inputStreamReader, final BufferedReader bufferedReader, final ExceptionUtil exceptionUtil) throws IOException {
        byte[] byteArray = new byte[0];
        final ByteArrayInputStream inputStream = new ByteArrayInputStream(byteArray);
    
        new Expectations() {
        	{
        		new InputStreamReader((InputStream)any);
        		new BufferedReader((Reader)any);
        		
        		inputStreamReader.read();
        		result = new Delegate() {
        			@SuppressWarnings("unused")
					void read() throws IOException {
        				throw new IOException();
        			}
        		};
        		
        		ExceptionUtil.convertRuntimeException((Exception)any);
        		result = new SystemException();
        		
        		inputStreamReader.close();
        		bufferedReader.close();
        	}
        };
        
        try {
        	StreamUtil.toString(inputStream);
        	fail();
        } catch (SystemException e) {
        	
        }
    }
    
    @Test
    public void testToByteArraySuccess() throws IOException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(outputStream));
        bufferedWriter.write(EXPECTED);
        bufferedWriter.flush();
        bufferedWriter.close();

        byte[] byteArray = outputStream.toByteArray();
        ByteArrayInputStream inputStream = new ByteArrayInputStream(byteArray);
        byte[] result = StreamUtil.toByteArray(inputStream);
        
        assertThat(result, is(EXPECTED.getBytes()));
        result = StreamUtil.toByteArray(inputStream);
        assertThat(result.length, is(0));
        inputStream.close();
    }
}
