/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.view.common;

import java.awt.FontMetrics;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

/**
 * JTable の列幅を自動計算する.<br>
 * ヘッダの値が文字列型である場合に対応する.<br>
 * セルの値が文字列型, 文字列配列型である場合に対応する.
 */
public class JTableColumnWidthCalculator {
    /** コンボボックスの幅として設定する数値において, 文字列の幅に加算する数値. */
    private static final int COMBO_BOX_ADDITIONAL_WIDTH = 28;

    /** 列の幅として設定する数値において, セル内幅に加算する数値. */
    private static final int COLUMN_ADDITIONAL_WIDTH = 3;

    /**
     * 操作対象の JTable.
     */
    private JTable table;

    /**
     * 新しい JTableColumnWidthCalculator を構築する.
     * 
     * @param table JTable.
     */
    public JTableColumnWidthCalculator(JTable table) {
        this.table = table;
    }

    /**
     * すべての列の幅を自動計算し, JTable に設定する.
     */
    public void setAutomatically() {
        // 列ごとに, 文字列の最大長取得
        for (int columnIndex = 0; columnIndex < this.table.getColumnCount(); columnIndex++) {
            setAutomatically(columnIndex);
        }
    }

    /**
     * 指定された列の幅を自動計算し, JTable に設定する.
     * 
     * @param columnIndex 列番号.
     */
    public void setAutomatically(int columnIndex) {
        setAutomatically(columnIndex, new ArrayList<String>());
    }

    /**
     * 指定された列の幅を自動計算し, JTable に設定する.
     * 
     * @param columnIndex 列番号.
     * @param stringList 列に含まれる文字列に加えて, 最大幅計算対象にする文字列のリスト.
     * 
     */
    public void setAutomatically(int columnIndex, List<String> stringList) {
        TableColumnModel tableColumnModel = this.table.getColumnModel();
        TableColumn tableColumn = tableColumnModel.getColumn(columnIndex);

        // ヘッダ文字列の幅を取得. 最初の最大幅とする.
        FontMetrics headerFontMetrics =
                this.table.getFontMetrics(table.getTableHeader().getFont());
        String headerString = (String) tableColumn.getHeaderValue();
        int maxWidth = headerFontMetrics.stringWidth(headerString);

        // カラム文字列の最大長取得. TableCellRenderer の種類によって処理が異なる.
        TableCellRenderer renderer = tableColumn.getCellRenderer();
        if (renderer instanceof ComboBoxTableCellRenderer) {

            // コンボボックスの項目の文字列幅から取得
            ComboBoxTableCellRenderer comboBoxTableCellRenderer =
                    (ComboBoxTableCellRenderer) renderer;
            FontMetrics comboBoxFontMetrics =
                    comboBoxTableCellRenderer.getFontMetrics(comboBoxTableCellRenderer.getFont());
            int maxStringWidth = 0;
            for (int i = 0; i < comboBoxTableCellRenderer.getItemCount(); i++) {
                int stringWidth =
                        comboBoxFontMetrics.stringWidth(comboBoxTableCellRenderer.getItemAt(
                                i)
                                .toString());
                maxStringWidth = Math.max(stringWidth, maxStringWidth);
            }

            // 追加で計算対象にする文字列による計算.
            for (String string : stringList) {
                int width = comboBoxFontMetrics.stringWidth(string);
                maxWidth = Math.max(width, maxWidth);
            }

            // コンボボックスの「▼」ボタンの幅等を考慮し, (文字列長+COMBO_BOX_ADDITIONAL_WIDTH) とする.
            maxWidth = maxStringWidth + COMBO_BOX_ADDITIONAL_WIDTH;
        } else {
            // 入力されたデータ文字列から, カラム文字列の最大長取得
            TableModel tableModel = this.table.getModel();
            FontMetrics fontMetrics =
                    this.table.getFontMetrics(this.table.getFont());
            for (int rowIndex = 0; rowIndex < this.table.getRowCount(); rowIndex++) {
                Object value = tableModel.getValueAt(rowIndex, columnIndex);
                if (value instanceof String[]) {
                    String[] strings = (String[]) value;
                    for (int i = 0; i < strings.length; i++) {
                        int width = fontMetrics.stringWidth(strings[i]);
                        maxWidth = Math.max(width, maxWidth);
                    }
                } else {
                    String string = value.toString();
                    int width = fontMetrics.stringWidth(string);
                    maxWidth = Math.max(width, maxWidth);
                }
            }

            // 追加で計算対象にする文字列による計算.
            for (String string : stringList) {
                int width = fontMetrics.stringWidth(string);
                maxWidth = Math.max(width, maxWidth);
            }
        }

        // 文字列の最大長が取得された場合は, 列幅を設定. (最大長 + COLUMN_ADDITIONAL_WIDTH) とする.
        if (maxWidth != -1) {
            tableColumn.setPreferredWidth(maxWidth + COLUMN_ADDITIONAL_WIDTH);
        }
    }
}
