/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.service.policyset.impl;

import java.util.List;
import java.util.Set;

import javax.validation.ConstraintViolation;
import javax.validation.Validator;

import jp.co.ogis_ri.citk.policytool.domain.policy.PolicyRepository;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.validation.PolicyValidModel;
import jp.co.ogis_ri.citk.policytool.domain.realm.RealmRepository;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Group;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Realm;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Referral;
import jp.co.ogis_ri.citk.policytool.service.common.validation.PolicyValidModelFactory;
import jp.co.ogis_ri.citk.policytool.service.policyset.PolicySetService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * ポリシー設定サービスの実装.
 */
@Service
class PolicySetServiceImpl implements PolicySetService {
    /**
     * RealmRepository.
     */
    @Autowired
    private RealmRepository realmRepository = null;

    /**
     * PolicyRepository.
     */
    @Autowired
    private PolicyRepository policyRepository = null;

    /**
     * バリデーター.
     */
    @Autowired
    private Validator validator;

    /**
     * 新しい PolicySetServiceImpl を構築する.
     */
    private PolicySetServiceImpl() {
    }

    @Override
    public List<Realm> findRealms() {
        return realmRepository.findAll();
    }

    @Override
    public Realm findRealmByName(String realmName) {
        return realmRepository.findRealmByName(realmName);
    }

    @Override
    public List<Group> findGroupByRealm(Realm realm) {
        return realmRepository.findGroupByRealm(realm);
    }

    @Override
    public List<Referral> findReferrals(Realm realmModel) {
        return realmRepository.findReferrals(realmModel);
    }

    @Override
    public ConstraintViolation<?> addPolicy(Policy policy) {
        // バリデーション
        Set<ConstraintViolation<Policy>> result = validatePolicy(policy);
        if (!result.isEmpty()) {
            return result.iterator().next();
        }

        Set<ConstraintViolation<PolicyValidModel>> correlationResult =
                validateCreatePolicyCorrelation(policy);
        if (!correlationResult.isEmpty()) {
            return correlationResult.iterator().next();
        }

        policyRepository.persist(policy);

        return null;
    }

    @Override
    public ConstraintViolation<?> modifyPolicy(Policy policy) {
        // バリデーション
        Set<ConstraintViolation<Policy>> result = validatePolicy(policy);
        if (!result.isEmpty()) {
            return result.iterator().next();
        }

        Set<ConstraintViolation<PolicyValidModel>> correlationResult =
                validateUpdatePolicyCorrelation(policy);
        if (!correlationResult.isEmpty()) {
            return correlationResult.iterator().next();
        }

        policyRepository.merge(policy);

        return null;
    }

    /**
     * ポリシーの単項目チェックを行う.
     * 
     * @param target ポリシー.
     * @return 単項目チェックの結果.
     */
    private Set<ConstraintViolation<Policy>> validatePolicy(Policy target) {
        Set<ConstraintViolation<Policy>> result;

        // 単項目チェック. エラーがある場合は打ち切り.
        result = validator.validate(target);
        if (!result.isEmpty()) {
            return result;
        }

        return result;
    }

    /**
     * ポリシー作成時の, ポリシーの相関項目チェックを行う.
     * 
     * @param target ポリシー.
     * @return 相関項目チェックの結果.
     */
    private Set<ConstraintViolation<PolicyValidModel>> validateCreatePolicyCorrelation(
            Policy target) {
        Set<ConstraintViolation<PolicyValidModel>> result;

        // 相関チェック
        PolicyValidModel model = PolicyValidModelFactory.create(target);
        model.setTarget(target);
        result = validator.validate(model, PolicyValidModel.Create.class);
        if (!result.isEmpty()) {
            return result;
        }

        return result;
    }

    /**
     * ポリシー更新時の, ポリシーの相関項目チェックを行う.
     * 
     * @param target ポリシー.
     * @return 相関項目チェックの結果.
     */
    private Set<ConstraintViolation<PolicyValidModel>> validateUpdatePolicyCorrelation(
            Policy target) {
        Set<ConstraintViolation<PolicyValidModel>> result;

        // 相関チェック
        PolicyValidModel model = PolicyValidModelFactory.create(target);
        model.setTarget(target);
        result = validator.validate(model, PolicyValidModel.Update.class);
        if (!result.isEmpty()) {
            return result;
        }

        return result;
    }
}
