/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.policytool.service.importpolicy.impl;

import java.util.List;
import java.util.Set;

import javax.validation.ConstraintViolation;
import javax.validation.Validator;

import jp.co.ogis_ri.citk.policytool.common.excel.ExcelPolicyReader;
import jp.co.ogis_ri.citk.policytool.common.exception.ExcelPolicyImportException;
import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;
import jp.co.ogis_ri.citk.policytool.common.util.ApplicationContextUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.PolicyRepository;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.validation.PolicyValidModel;
import jp.co.ogis_ri.citk.policytool.service.common.validation.PolicyValidModelFactory;
import jp.co.ogis_ri.citk.policytool.service.importpolicy.ImportPolicyService;

import org.apache.commons.logging.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.stereotype.Service;

/**
 * インポートメイン処理クラス
 */
@Service
public class ImportPolicyServiceImpl implements ImportPolicyService {
    /**
     * ロガー.
     */
    private Log logger =
            LogWrapperFactory.getLog(ImportPolicyServiceImpl.class);

    /**
     * ExcelPolicyをファイルから読み込むクラスのオブジェクト.
     */
    @Autowired
    ExcelPolicyReader excelPolicyReader;

    /**
     * バリデーター.
     */
    @Autowired
    private Validator validator;

    /**
     * ポリシーのエンティティに対する処理を行うクラスのオブジェクト.
     */
    @Autowired
    private PolicyRepository repository;

    /**
     * メッセージ リソース アクセスクラス.
     */
    private MessageSourceAccessor messageSourceAccessor = null;

    /**
     * デフォルトコンストラクター
     */
    public ImportPolicyServiceImpl() {
    }

    @Override
    public void importPolicy(String fileName) {
        List<Policy> policies = excelPolicyReader.read(fileName);

        singleValidate(policies, fileName);
        interrelateValidate(policies, fileName);

        repository.importPolicies(policies);
    }

    /**
     * メッセージを取得する.
     * 
     * @param res リソース名.
     * @param args メッセージ取得に使用する追加情報.
     * @return
     */
    private String getMessage(String res, Object... args) {
        if (messageSourceAccessor == null) {
            messageSourceAccessor = ApplicationContextUtil.getResources();
        }

        return messageSourceAccessor.getMessage(res, args);
    }

    /**
     * ポリシーに対する単項目チェックを行う.
     * 
     * @param policies ポリシーのリスト.
     * @param fileName ファイル名.
     */
    private void singleValidate(List<Policy> policies, String fileName) {
        boolean isBad = false;
        for (Policy policy : policies) {
            logger.info(getMessage(
                    "view.importService.singleValidationLoopStart",
                    policy.getRealmName(), policy.getPolicyName()));

            Set<ConstraintViolation<Policy>> result =
                    validator.validate(policy);
            if (!result.isEmpty()) {
                for (ConstraintViolation<Policy> violation : result) {
                    logger.error(getMessage(
                            "view.importService.singleValidationError",
                            violation.getMessage(),
                            violation.getInvalidValue(),
                            violation.getPropertyPath()));
                }
                isBad = true;
            } else {
                policy.resetPermit();
            }
            logger.info(getMessage("view.importService.singleValidationLoopEnd"));
        }

        if (isBad) {
            throw new ExcelPolicyImportException("E-0301", fileName);
        }
    }

    /**
     * ポリシーに対する相関項目チェックを行う.
     * 
     * @param policies ポリシーのリスト.
     * @param fileName ファイル名.
     */
    private void interrelateValidate(List<Policy> policies, String fileName) {
        boolean isBad = false;
        for (Policy policy : policies) {
            logger.info(getMessage(
                    "view.importService.interrelateValidationLoopStart",
                    policy.getRealmName(), policy.getPolicyName()));
            PolicyValidModel validModel =
                    PolicyValidModelFactory.create(policy);
            validModel.setTarget(policy);

            Set<ConstraintViolation<PolicyValidModel>> result =
                    validator.validate(validModel,
                            PolicyValidModel.Import.class);
            if (!result.isEmpty()) {
                for (ConstraintViolation<PolicyValidModel> violation : result) {
                    logger.error(getMessage(
                            "view.importService.interrelateValidationError",
                            violation.getMessage(), violation.getPropertyPath()));
                }
                isBad = true;
            }
            logger.info(getMessage("view.importService.interrelateValidationLoopEnd"));
        }

        if (isBad) {
            throw new ExcelPolicyImportException("E-0301", fileName);
        }
    }
}
