/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package jp.co.ogis_ri.citk.policytool.domain.policy.impl;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Join;
import javax.persistence.criteria.JoinType;
import javax.persistence.criteria.ParameterExpression;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.validation.ConstraintViolationException;

import jp.co.ogis_ri.citk.policytool.common.api.OpenAMAccess;
import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;
import jp.co.ogis_ri.citk.policytool.common.repository.JpaGenericRepository;
import jp.co.ogis_ri.citk.policytool.common.util.ExceptionUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.PolicyRepository;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Resource;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Subject;

import org.apache.commons.logging.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

/**
 * JPAによりポリシーのエンティティに対する処理を行うクラス.
 */
@Repository
public class JpaPolicyRepositoryImpl extends JpaGenericRepository<Policy, Long>
        implements PolicyRepository {
    /**
     * ロガー.
     */
    private Log logger =
            LogWrapperFactory.getLog(JpaPolicyRepositoryImpl.class);

    /**
     * OpenAM アクセスオブジェクト.
     */
    @Autowired(required = false)
    private OpenAMAccess amAccess = null;

    /**
     * ポリシーリスト同期処理
     * 
     * 指定されたレルム内に存在するポリシーのリストを、 アクセスマネージャ内の同データと同期する。
     * 
     * @param realmName 同期するレルムの名前
     */
    @Override
    public void sync(String realmName) {
        List<Policy> policies = findPoliciesByRealmName(realmName);
        for (Policy policy : policies) {
            em.remove(policy);
        }
        em.flush();

        List<Policy> listPolicies = amAccess.getPolicies(realmName, "*");
        for (Policy policy : listPolicies) {
            try {
                em.persist(policy);
            } catch (ConstraintViolationException e) {
                logger.debug(e.getMessage());
                throw ExceptionUtil.convertRuntimeException(e);
            }
        }
        em.flush();
    }

    /**
     * ポリシーリスト参照
     * 
     * データベース内の、指定されたレルム内に存在するポリシーのリストを参照する。
     * 
     * @param realmName 参照するレルムの名前
     * @return ポリシーのリスト
     */
    @Override
    @SuppressWarnings("unchecked")
    public List<Policy> findPoliciesByRealmName(String realmName) {

        return em.createQuery(
                "SELECT e FROM Policy e where e.realmName = :RealmName order by e.policyName")
                .setParameter("RealmName", realmName)
                .getResultList();
    }

    /**
     * ポリシー検索
     * 
     * 指定されたレルム内で、指定されたサブジェクト、リソースを持つポリシーを検索、参照する.
     * 
     * @param realmName 参照するレルム名
     * @param subjectName サブジェクト
     * @param resourceUrl リソース
     * @return ポリシーのリスト
     */
    public List<Policy> findPolicies(String realmName, String subjectName,
            String resourceUrl) {
        CriteriaBuilder cb = em.getCriteriaBuilder();
        CriteriaQuery<Policy> c = cb.createQuery(Policy.class);

        // Fromを構築する.
        Root<Policy> policy = c.from(Policy.class);
        // Selectを構築する.
        c.select(policy);
        c.distinct(true);

        // JOINを構築する
        Join<Policy, Subject> subject = policy.join("subjects", JoinType.LEFT);
        Join<Policy, Resource> resource =
                policy.join("resources", JoinType.LEFT);

        // Order Byを構築する
        c.orderBy(cb.asc(policy.get("realmName")),
                cb.asc(policy.get("policyName")));

        // Whereを構築する
        List<Predicate> criteria = new ArrayList<Predicate>();
        if (realmName != null) {
            ParameterExpression<String> p =
                    cb.parameter(String.class, "realmName");
            criteria.add(cb.equal(policy.get("realmName"), p));
        }
        if (subjectName != null) {
            ParameterExpression<String> p =
                    cb.parameter(String.class, "subjectName");
            criteria.add(cb.equal(subject.get("subjectName"), p));
        }
        if (resourceUrl != null) {
            ParameterExpression<String> p =
                    cb.parameter(String.class, "resourceUrl");
            criteria.add(cb.like(resource.get("resourceUrl").as(String.class),
                    p));
        }

        if (criteria.size() == 1) {
            c.where(criteria.get(0));
        } else if (criteria.size() > 1) {
            c.where(cb.and(criteria.toArray(new Predicate[0])));
        }

        // クエリーを作成する
        TypedQuery<Policy> q = em.createQuery(c);
        if (realmName != null) {
            q.setParameter("realmName", realmName);
        }
        if (subjectName != null) {
            q.setParameter("subjectName", subjectName);
        }
        if (resourceUrl != null) {
            // 前方一致
            resourceUrl = resourceUrl.replaceAll("[%*?_]", "") + "%";
            q.setParameter("resourceUrl", resourceUrl);
        }
        return q.getResultList();
    }

    @Override
    public void persist(Policy policy) {
        em.persist(policy);
        amAccess.createPolicy(policy);
    }

    @Override
    public Policy merge(Policy policy) {
        policy = em.merge(policy);
        amAccess.updatePolicy(policy);
        return policy;
    }

    @Override
    public void remove(Policy policy) {
        em.remove(em.merge(policy));
        amAccess.deletePolicy(policy);
    }

    @Override
    public void importPolicies(List<Policy> policies) {
        // 全ポリシーのリスト
        List<Policy> existedPolicies = findAll();

        // 既に同ポリシー名, 同レルム名で存在するポリシーは, 更新, 存在しないポリシーは追加
        for (Policy importedPolicy : policies) {
            // 現在の同ポリシー名, 同レルム名のポリシー名があるか確認
            boolean importedPolicyExists = false;
            for (Policy existedPolicy : existedPolicies) {
                if (importedPolicy.getPolicyName().equals(
                        existedPolicy.getPolicyName())
                        && importedPolicy.getRealmName().equals(
                                existedPolicy.getRealmName())) {
                    importedPolicyExists = true;
                    break;
                }
            }

            // 同ポリシー名, 同レルム名で存在する場合は更新, 存在しない場合は追加.
            if (importedPolicyExists) {
                amAccess.updatePolicy(importedPolicy);
            } else {
                amAccess.createPolicy(importedPolicy);
            }
        }
    }
}
