#ifndef __HADDOCK__
-- $Id: RubyString.hs,v 1.6 2007/01/14 12:57:58 desumasu Exp $
#endif
module Cinnamon.RubyString (
  capitalize,
  center,
  chomp,
  chomp',
  chop,
  rbConcat,
  downcase,
  dump,
  each,
  toLines,
  hex,
  include,
  index,
  ljust,
  oct,
  rindex,
  rjust,
  sliceFromTo,
  sliceFromLen,
  strip,
  succStr,
  swapcase,
  upcase,
  upto
) where

import Data.Char (isSpace, isPrint, isDigit, isHexDigit, isOctDigit, 
                  isUpper, isLower, toUpper, toLower,
                  showLitChar, digitToInt)
import Data.List (isPrefixOf, isSuffixOf)
import Data.Maybe (isJust)

-- 参照: <http://wiki.fdiary.net/desumasu/>

-- | RubyのString#capitalize相当の関数です。
capitalize :: String -> String
capitalize []       = []
capitalize (x : xs) = toUpper x : map toLower xs

-- | RubyのString#center相当の関数です。
center :: Int -> String -> String
center width str
  | width <= strWidth = str
  | otherwise         = replicate leftSp ' ' ++ str ++ replicate rightSp ' '
  where
    strWidth = length str
    leftSp = (width - strWidth) `div` 2
    rightSp = width - (leftSp + strWidth)

-- | RubyのString#chomp相当の関数です。
chomp :: String -> String
chomp str
  | null str              = str
  | isSuffixOf "\r\n" str = init $ init str
  | isSuffixOf "\n" str || isSuffixOf "\r" str
                          = init str
  | otherwise             = str

-- | RubyのString#chomp相当の関数です。
-- 行を区切る文字列を指定することができます。
chomp' :: (Eq a) => [a] -> [a] -> [a]
chomp' rs str
  | end == rs = main
  | otherwise = str
  where 
    (main, end) = splitAt (length str - length rs) str

-- | RubyのString#chop相当の関数です。
chop :: String -> String
chop str
  | null str = str
  | isSuffixOf "\r\n" str = init $ init str
  | otherwise = init str

-- | RubyのString#concat相当の関数です。
rbConcat :: (Show a) => String -> a -> String
rbConcat = (. show) . (++)

-- | RubyのString#downcase相当の関数です。
downcase :: String -> String
downcase = map toLower

-- | RubyのString#dump相当の関数です。
dump :: String -> String
dump = foldr f []
  where
    f ch converted
      | isPrint ch = ch : converted
      | otherwise  = showLitChar ch converted

-- | RubyのString#each相当の関数です。
each :: (Monad m, Eq a) => [a] -> ([a] -> m b) -> [a] -> m [b]
each rs f str = mapM f (toLines rs str)

-- | 与えた文字列を行ごとに分解したリストを返します。
-- この関数は文字列だけではなくリスト一般に適用可能です。
toLines :: (Eq a) => [a] -> [a] -> [[a]]
toLines rs str =
  case index str rs of
    Just idx -> s1 : if (null s2) then [] else toLines rs s2
      where
        (s1, s2) = splitAt (idx + length rs) str
    Nothing  -> str : []

-- | RubyのString#hex相当の関数です。
hex :: String -> Int
hex = filtrated . filter (/= '_')
  where
    filtrated []  = 0
    filtrated s @ (c : cs)
      | c == '-'  = -1 * afterSign cs
      | otherwise = afterSign s

    afterSign s
      | isPrefixOf0x = afterPrefix (drop 2 s) 0
      | otherwise    = afterPrefix s 0
      where 
        isPrefixOf0x = isPrefixOf "0x" s || isPrefixOf "0X" s

    afterPrefix [] acc = acc
    afterPrefix (c : cs) acc
      | isHexDigit c = afterPrefix cs $ digitToInt c + acc * 16
      | otherwise    = acc

-- | RubyのString#include相当の関数です。
include :: (Eq a) => [a] -> [a] -> Bool
include str substr = isJust $ index str substr

-- | RubyのString#index相当の関数です。
index str substr
  | isPrefixOf substr str       = Just 0
  | length str <= length substr = Nothing
  | otherwise                   = index (drop 1 str) substr >>= return . (+1)

-- | RubyのString#ljust相当の関数です。
ljust :: Int -> String -> String
ljust width str = str ++ spaces
  where
    spaces 
      | strWidth < width = replicate (width - strWidth) ' '
      | otherwise        = []

    strWidth = length str

-- | RubyのString#oct相当の関数です。
oct :: String -> Int
oct = filtrated . filter (/= '_')
  where
    filtrated [] = 0
    filtrated s @ (c : cs)
      | c == '-'  = -1 * afterSign cs
      | otherwise = afterSign s

    afterSign s
      | isPrefixOf0x = hex s
      | isPrefixOf0b = bin (drop 2 s) 0
      | otherwise    = afterPrefix s 0
      where
        isPrefixOf0x = isPrefixOf "0x" s || isPrefixOf "0X" s
        isPrefixOf0b = isPrefixOf "0b" s || isPrefixOf "0B" s

    afterPrefix [] acc = acc
    afterPrefix (c : cs) acc
      | isOctDigit c = afterPrefix cs $ digitToInt c + acc * 8
      | otherwise    = acc

    bin [] acc = acc
    bin (c : cs) acc
      | c `elem` "01" = bin cs $ digitToInt c + acc * 2
      | otherwise = acc

-- | RubyのString#rindex相当の関数です。
rindex :: (Eq a) => [a] -> [a] -> Maybe Int
rindex str substr = do
  len <- index (reverse str) (reverse substr)
  return $ length str - len - length substr

-- | RubyのString#rjust相当の関数です。
rjust :: Int -> String -> String
rjust width str = spaces ++ str
  where
    spaces
      | strWidth < width = replicate (width - strWidth) ' '
      | otherwise        = []

    strWidth = length str

-- | RubyのString#slice相当の関数です。
-- 開始インデックス、終了インデックスを指定して、指定した文字列から
-- 部分文字列を抽出します。
sliceFromTo :: Int -> Int -> [a] -> [a]
sliceFromTo from to str
  | from' <= to' + 1 && 0 <= from' = drop from' $ take (to' + 1) str
  | otherwise = error "sliceFromTo: invaid index."
  where
    to'   = if to   < 0 then length str + to   else to
    from' = if from < 0 then length str + from else from

-- | RubyのString#slice相当の関数です。
-- 開始インデックス、部分文字列の長さを指定します。
sliceFromLen :: Int -> Int -> [a] -> [a]
sliceFromLen from len = sliceFromTo from (from + len - 1)

-- | RubyのString#strip相当の関数です。
strip :: String -> String
strip = stripR . stripL
  where
    stripL = dropWhile isSpace
    stripR = reverse . stripL . reverse

-- | RubyのString#succ相当の関数です。
succStr :: String -> String
succStr str
  | carry     = addNewFig resStr
  | otherwise = resStr
  where
    addNewFig [] = []
    addNewFig str@(x : xs)
      | isDigit x = '1' : str
      | isLower x = 'a' : str
      | isUpper x = 'A' : str
      | otherwise = x : addNewFig xs

    (carry, resStr) = succStr' str

    succStr' [] = (True, [])
    succStr' (x : xs)
      | carry'     = (carrying x, next x : resStr')
      | otherwise = (False, x : resStr')
      where
        carrying x
          | isDigit x = x == '9'
          | isLower x = x == 'z'
          | isUpper x = x == 'Z'
          | otherwise = carry'
        next x
          | isDigit x = if x == '9' then '0' else succ x
          | isLower x = if x == 'z' then 'a' else succ x
          | isUpper x = if x == 'Z' then 'A' else succ x
          | otherwise = x

        (carry', resStr') = succStr' xs

-- | RubyのString#swapcase相当の関数です。
swapcase :: String -> String
swapcase = map $ \c -> if isLower c then toUpper c else toLower c

-- | RubyのString#upcase相当の関数です。
upcase :: String -> String
upcase = map toUpper

-- | RubyのString#upto相当の関数です。
upto :: (Monad m) => String -> String -> (String -> m a) -> m ()
upto from to f
  | from == to = f from >> return ()
  | otherwise  = f from >> upto (succStr from) to f >> return ()
