#ifndef __HADDOCK__
-- $Id: Nkf.hs,v 1.10 2007/01/14 11:31:01 ha-tan Exp $
#endif
module Cinnamon.Nkf (
  nkf, nkf',
  toJis, toEuc, toSjis, toUtf8, toUtf16
) where

import Cinnamon.RubyString (strip)
import Foreign.C.String (CString, withCStringLen, peekCString, newCString)
import Foreign.Marshal.Alloc (free)
import Foreign.Marshal.Array (withArrayLen)
import Foreign.Ptr (Ptr, nullPtr)
import System.IO.Unsafe (unsafePerformIO)

foreign import ccall unsafe "nkf" cNkf :: 
  Ptr CString -> Int -> CString -> Int -> IO CString

splitOpts :: String -> [String]
splitOpts = normal "" . strip
  where
    normal [] []              = []
    normal a  []              = [reverse a]
    normal a  (' '  : xs)     = reverse a : inSpace xs
    normal a  ('\'' : xs)     = inQuote a xs
    normal a  ('"'  : xs)     = inDQuote a xs
    normal a  ('\\' : x : xs) = normal (x : a) xs
    normal a  (x    : xs)     = normal (x : a) xs

    inSpace []          = []
    inSpace (' '  : xs) = inSpace xs
    inSpace xs          = normal "" xs
    
    inQuote [] []              = []
    inQuote a  []              = [reverse a]
    inQuote a  ('\'' : xs)     = normal a xs
    inQuote a  ('\\' : x : xs) = inQuote (x : a) xs
    inQuote a  (x    : xs)     = inQuote (x : a) xs
    
    inDQuote [] []              = []
    inDQuote a  []              = [reverse a]
    inDQuote a  ('"'  : xs)     = normal a xs
    inDQuote a  ('\\' : x : xs) = inDQuote (x : a) xs
    inDQuote a  (x    : xs)     = inDQuote (x : a) xs

nkfIO :: String -> String -> IO String
nkfIO = nkfIO' . splitOpts

nkfIO' :: [String] -> String -> IO String
nkfIO' opts src = do
  copts <- mapM newCString opts
  dst <- withArrayLen copts $ \ coptsAryLen coptsAry ->
           withCStringLen src $ \ (csrc, csrcLen) -> 
             nkfIO' coptsAry coptsAryLen csrc csrcLen
  mapM_ free copts
  return dst
  where
    nkfIO' coptsAry coptsAryLen csrc csrcLen = do
      cdst <- cNkf coptsAry coptsAryLen csrc csrcLen
      if (cdst == nullPtr)
        then error "nkf: failed."
        else do
          dst <- peekCString cdst
          free cdst
          return dst

-- | NKFを使って文字エンコーディングを変換します。
-- この関数はスレッドセーフではありません。
nkf :: String -> String -> String
nkf = (unsafePerformIO .) . nkfIO

-- | NKFを使って文字エンコーディングを変換します。
-- この関数はスレッドセーフではありません。
nkf' :: [String] -> String -> String
nkf' = (unsafePerformIO .) . nkfIO'

-- | NKFを使ってJISコードの文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toJis :: String -> String
toJis = nkf "-m0 -j"

-- | NKFを使ってEUCコードの文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toEuc :: String -> String
toEuc = nkf "-m0 -e"

-- | NKFを使ってShift-JISコードの文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toSjis :: String -> String
toSjis = nkf "-m0 -s"

-- | NKFを使ってUTF8の文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toUtf8 :: String -> String
toUtf8 = nkf "-m0 -w"

-- | NKFを使ってUTF16の文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toUtf16 :: String -> String
toUtf16 = nkf "-m0 -w16"
