#ifndef __HADDOCK__
-- $Id: Nkf.hs,v 1.6 2007/01/02 12:13:41 ha-tan Exp $
#endif
module Cinnamon.Nkf (
  nkf, 
  toJis, toEuc, toSjis, toUtf8, toUtf16,
  jisToUcs4, eucToUcs4, sjisToUcs4, utf16ToUcs4,
  ucs4ToJis, ucs4ToEuc, ucs4ToSjis, ucs4ToUtf16
) where

import Cinnamon.Misc (strip)
import Cinnamon.Ucs (ucs4ToUtf8, utf8ToUcs4)
import Foreign.C.String (CString, withCStringLen, peekCString, newCString)
import Foreign.Marshal.Alloc (free)
import Foreign.Marshal.Array (withArrayLen)
import Foreign.Ptr (Ptr, nullPtr)
import System.IO.Unsafe (unsafePerformIO)

foreign import ccall unsafe "nkf" cNkf :: 
  Ptr CString -> Int -> CString -> Int -> IO CString

splitOpts :: String -> [String]
splitOpts = normal "" . strip
  where
    normal [] []              = []
    normal a  []              = [reverse a]
    normal a  (' '  : xs)     = reverse a : inSpace xs
    normal a  ('\'' : xs)     = inQuote a xs
    normal a  ('"'  : xs)     = inDQuote a xs
    normal a  ('\\' : x : xs) = normal (x : a) xs
    normal a  (x    : xs)     = normal (x : a) xs

    inSpace []          = []
    inSpace (' '  : xs) = inSpace xs
    inSpace xs          = normal "" xs
    
    inQuote [] []              = []
    inQuote a  []              = [reverse a]
    inQuote a  ('\'' : xs)     = normal a xs
    inQuote a  ('\\' : x : xs) = inQuote (x : a) xs
    inQuote a  (x    : xs)     = inQuote (x : a) xs
    
    inDQuote [] []              = []
    inDQuote a  []              = [reverse a]
    inDQuote a  ('"'  : xs)     = normal a xs
    inDQuote a  ('\\' : x : xs) = inDQuote (x : a) xs
    inDQuote a  (x    : xs)     = inDQuote (x : a) xs

nkfIO :: String -> String -> IO String
nkfIO opt src = do
  copts <- mapM newCString $ splitOpts opt
  dst <- withArrayLen copts $ \ coptsAryLen coptsAry ->
           withCStringLen src $ \ (csrc, csrcLen) -> 
             nkfIO' coptsAry coptsAryLen csrc csrcLen
  mapM_ free copts
  return dst
  where
    nkfIO' coptsAry coptsAryLen csrc csrcLen = do
      cdst <- cNkf coptsAry coptsAryLen csrc csrcLen
      if (cdst == nullPtr)
        then error "nkf: failed."
        else do
          dst <- peekCString cdst
          free cdst
          return dst

-- | NKFを使って文字エンコーディングを変換します。
-- この関数はスレッドセーフではありません。
nkf :: String -> String -> String
nkf = (unsafePerformIO .) . nkfIO

-- | NKFを使ってJISコードの文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toJis :: String -> String
toJis = nkf "-m0 -j"

-- | NKFを使ってEUCコードの文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toEuc :: String -> String
toEuc = nkf "-m0 -e"

-- | NKFを使ってShift-JISコードの文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toSjis :: String -> String
toSjis = nkf "-m0 -s"

-- | NKFを使ってUTF8の文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toUtf8 :: String -> String
toUtf8 = nkf "-m0 -w"

-- | NKFを使ってUTF16の文字列に変換します。
-- RubyのKconv.tojisと違って、MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
toUtf16 :: String -> String
toUtf16 = nkf "-m0 -w16"

-- | JISコードの文字列をUCS4の文字列に変換します。
-- MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
jisToUcs4 :: String -> String
jisToUcs4 = utf8ToUcs4 . nkf "-m0 -J -w"

-- | EUCコードの文字列をUCS4の文字列に変換します。
-- MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
eucToUcs4 :: String -> String
eucToUcs4 = utf8ToUcs4 . nkf "-m0 -E -w"

-- | Shift-JISコードの文字列をUCS4の文字列に変換します。
-- MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
sjisToUcs4 :: String -> String
sjisToUcs4 = utf8ToUcs4 . nkf "-m0 -S -w"

-- | UTF16の文字列をUCS4の文字列に変換します。
-- MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
utf16ToUcs4 :: String -> String
utf16ToUcs4 = utf8ToUcs4 . nkf "-m0 -W16 -w"

-- | UCS4の文字列をJISコードの文字列に変換します。
-- MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
ucs4ToJis :: String -> String
ucs4ToJis = nkf "-m0 -W -j" . ucs4ToUtf8

-- | UCS4の文字列をEUCコードの文字列に変換します。
-- MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
ucs4ToEuc :: String -> String
ucs4ToEuc = nkf "-m0 -W -e" . ucs4ToUtf8

-- | UCS4の文字列をShift-JISコードの文字列に変換します。
-- MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
ucs4ToSjis :: String -> String
ucs4ToSjis = nkf "-m0 -W -s" . ucs4ToUtf8

-- | UCS4の文字列をUTF16の文字列に変換します。
-- MIMEの変換は行いません。
-- この関数はスレッドセーフではありません。
ucs4ToUtf16 :: String -> String
ucs4ToUtf16 = nkf "-m0 -W -w16" . ucs4ToUtf8
