#ifndef __HADDOCK__
-- $Id: Base64.hs,v 1.4 2007/01/02 09:41:56 ha-tan Exp $
#endif
module Cinnamon.Base64 (
  base64Encode,
  base64EncodeSafe,
  base64EncodeLn,
  base64EncodeSafeLn
) where

import Cinnamon.Misc (groupn)
import Data.Array (Array, (!), listArray)
import Data.Bits ((.&.), (.|.), shiftR, shiftL)
import Data.Char (ord)

table :: Array Int Char
table = 
    listArray (0, 63) $ ['A' .. 'Z'] ++ ['a' .. 'z'] ++ ['0' .. '9'] ++ "+/"

tableSafe :: Array Int Char
tableSafe = 
    listArray (0, 63) $ ['A' .. 'Z'] ++ ['a' .. 'z'] ++ ['0' .. '9'] ++ "-_"

conv :: Array Int Char -> Int -> Int -> Int -> String
conv table c1 c2 c3 =
  let i1 =  (c1 .&. 0xfc) `shiftR` 2
      i2 = ((c1 .&. 0x03) `shiftL` 4) .|. ((c2 .&. 0xf0) `shiftR` 4)
      i3 = ((c2 .&. 0x0f) `shiftL` 2) .|. ((c3 .&. 0xc0) `shiftR` 6)
      i4 =   c3 .&. 0x3f
  in [table ! i1, table ! i2, table ! i3, table ! i4]

base64EncodeBase :: Array Int Char -> String -> String
base64EncodeBase _ [] = ""

base64EncodeBase table (x1 : []) = 
  (take 2 $ conv table (ord x1) 0 0) ++ "=="

base64EncodeBase table (x1 : x2 : []) =
  (take 3 $ conv table (ord x1) (ord x2) 0) ++ "="

base64EncodeBase table (x1 : x2 : x3 : xs) =
   conv table (ord x1) (ord x2) (ord x3) ++ base64EncodeBase table xs

-- | 指定した文字列をBASE64変換します。
-- 参照: <http://www.ietf.org/rfc/rfc3548.txt>
base64Encode :: String -> String
base64Encode = base64EncodeBase table

-- | 指定した文字列をBASE64変換します()。
-- BASE64変換の結果には、URLに使用してよい文字のみ使用します。
-- 参照: <http://www.ietf.org/rfc/rfc3548.txt>
base64EncodeSafe :: String -> String
base64EncodeSafe = base64EncodeBase tableSafe

sepLn :: Int -> String -> String
sepLn n = concatMap (++ "\n") . groupn n

-- | 指定した文字列をBASE64変換します。
-- その際に指定した桁数ごとに改行を入れます。
-- BASE64変換の一番最後にも改行を入れます。
-- 参照: <http://www.ietf.org/rfc/rfc3548.txt>
base64EncodeLn :: Int -> String -> String
base64EncodeLn n = sepLn n . base64Encode

-- | 指定した文字列をBASE64変換します。
-- その際に指定した桁数ごとに改行を入れます。
-- BASE64変換の一番最後にも改行を入れます。
-- BASE64変換の結果には、URLに使用してよい文字のみ使用します。
-- 参照: <http://www.ietf.org/rfc/rfc3548.txt>
base64EncodeSafeLn :: Int -> String -> String
base64EncodeSafeLn n = sepLn n . base64EncodeSafe
