/*
    ChibiOS - Copyright (C) 2006..2018 Giovanni Di Sirio

    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

/**
 * @file    common/I2C/hal_i2c_lld.h
 * @brief   SW I2C subsystem low level driver header.
 *
 * @addtogroup I2C
 * @{
 */

#ifndef HAL_I2C_LLD_H
#define HAL_I2C_LLD_H

#if HAL_USE_I2C || defined(__DOXYGEN__)

/*===========================================================================*/
/* Driver constants.                                                         */
/*===========================================================================*/

/*===========================================================================*/
/* Driver pre-compile time settings.                                         */
/*===========================================================================*/

/**
 * @name    Configuration options
 * @{
 */
/**
 * @brief   Use OSAL delays.
 * @details If set to @p TRUE then delays are implemented using the
 *          thread-friendy delay function else a delay function must
 *          be provided extenally.
 */
#if !defined(SW_I2C_USE_OSAL_DELAY) || defined(__DOXYGEN__)
#define SW_I2C_USE_OSAL_DELAY               TRUE
#endif

/**
 * @brief   I2C1 driver enable switch.
 * @details If set to @p TRUE the support for I2C1 is included.
 * @note    The default is @p FALSE.
 */
#if !defined(SW_I2C_USE_I2C1) || defined(__DOXYGEN__)
#define SW_I2C_USE_I2C1                     FALSE
#endif

/**
 * @brief   I2C2 driver enable switch.
 * @details If set to @p TRUE the support for I2C2 is included.
 * @note    The default is @p FALSE.
 */
#if !defined(SW_I2C_USE_I2C2) || defined(__DOXYGEN__)
#define SW_I2C_USE_I2C2                     FALSE
#endif

/**
 * @brief   I2C3 driver enable switch.
 * @details If set to @p TRUE the support for I2C3 is included.
 * @note    The default is @p FALSE.
 */
#if !defined(SW_I2C_USE_I2C3) || defined(__DOXYGEN__)
#define SW_I2C_USE_I2C3                     FALSE
#endif

/**
 * @brief   I2C4 driver enable switch.
 * @details If set to @p TRUE the support for I2C4 is included.
 * @note    The default is @p FALSE.
 */
#if !defined(SW_I2C_USE_I2C4) || defined(__DOXYGEN__)
#define SW_I2C_USE_I2C4                     FALSE
#endif
/** @} */

/*===========================================================================*/
/* Derived constants and error checks.                                       */
/*===========================================================================*/

/*===========================================================================*/
/* Driver data structures and types.                                         */
/*===========================================================================*/

/**
 * @brief   Type representing an I2C address.
 */
typedef uint16_t i2caddr_t;

/**
 * @brief   Type of I2C driver condition flags.
 */
typedef uint8_t i2cflags_t;

/**
 * @brief   Type of a delay function.
 */
typedef void (*i2c_delay_t)(void);

/**
 * @brief   Type of I2C driver configuration structure.
 */
typedef struct {
  /**
   * @brief   10 bits addressing switch.
   */
  bool                      addr10;
  /**
   * @brief   I2C clock line.
   */
  ioline_t                  scl;
  /**
   * @brief   I2C data line.
   */
  ioline_t                  sda;
#if SW_I2C_USE_OSAL_DELAY || defined(__DOXYGEN__)
  /**
   * @brief   Delay of an half bit time in system ticks.
   */
  systime_t                 ticks;
#else
  /**
   * @brief   Pointer to an externally defined delay function.
   */
  i2c_delay_t               delay;
#endif
} I2CConfig;

/**
 * @brief   Type of a structure representing an I2C driver.
 */
typedef struct I2CDriver I2CDriver;

/**
 * @brief   Structure representing an I2C driver.
 */
struct I2CDriver {
  /**
   * @brief   Driver state.
   */
  i2cstate_t                state;
  /**
   * @brief   Current configuration data.
   */
  const I2CConfig           *config;
  /**
   * @brief   Error flags.
   */
  i2cflags_t                errors;
#if I2C_USE_MUTUAL_EXCLUSION || defined(__DOXYGEN__)
  mutex_t                   mutex;
#endif /* I2C_USE_MUTUAL_EXCLUSION */
#if defined(I2C_DRIVER_EXT_FIELDS)
  I2C_DRIVER_EXT_FIELDS
#endif
  /* End of the mandatory fields.*/
  /**
   * @brief   Time of operation begin.
   */
  systime_t                 start;
  /**
   * @brief   Time of operation timeout.
   */
  systime_t                 end;
};

/*===========================================================================*/
/* Driver macros.                                                            */
/*===========================================================================*/

/**
 * @brief   Get errors from I2C driver.
 *
 * @param[in] i2cp      pointer to the @p I2CDriver object
 *
 * @notapi
 */
#define i2c_lld_get_errors(i2cp) ((i2cp)->errors)

/*===========================================================================*/
/* External declarations.                                                    */
/*===========================================================================*/

#if !defined(__DOXYGEN__)
#if SW_I2C_USE_I2C1
extern I2CDriver I2CD1;
#endif
#if SW_I2C_USE_I2C2
extern I2CDriver I2CD2;
#endif
#if SW_I2C_USE_I2C3
extern I2CDriver I2CD3;
#endif
#if SW_I2C_USE_I2C4
extern I2CDriver I2CD4;
#endif
#endif /* !defined(__DOXYGEN__) */

#ifdef __cplusplus
extern "C" {
#endif
  void i2c_lld_init(void);
  void i2c_lld_start(I2CDriver *i2cp);
  void i2c_lld_stop(I2CDriver *i2cp);
  msg_t i2c_lld_master_transmit_timeout(I2CDriver *i2cp, i2caddr_t addr,
                                        const uint8_t *txbuf, size_t txbytes,
                                        uint8_t *rxbuf, size_t rxbytes,
                                        systime_t timeout);
  msg_t i2c_lld_master_receive_timeout(I2CDriver *i2cp, i2caddr_t addr,
                                       uint8_t *rxbuf, size_t rxbytes,
                                       systime_t timeout);
#ifdef __cplusplus
}
#endif

#endif /* HAL_USE_I2C  */

#endif /* HAL_I2C_LLD_H */

/** @} */
