/*
 * lisp.c - utility functions like LISP
 *
 * Copyright (c) 1996,1997 Nara Institute of Science and Technology
 *
 * Author: 1990/11/12/Mon Yutaka MYOKI(Nagao Lab., KUEE)
 *         1990/12/16/Mon Last Modified
 *         special thanks to Itsuki NODA
 *         A.Kitauchi <akira-k@is.aist-nara.ac.jp>, Apr. 1997
 */

#include "chadic.h"

#define COMMENTCHAR	';'
#define BPARENTHESIS	'('
#define BPARENTHESIS2	'<'
#define BPARENTHESIS3	'['
#define EPARENTHESIS	')'
#define EPARENTHESIS2	'>'
#define EPARENTHESIS3	']'
#define SCANATOM	"%[^(;) \n\t]"
#define NILSYMBOL	"NIL"
#define CELLALLOCSTEP	1024

#define _Car(cell)	(((cell_t *)(cell))->value.cons.car)
#define _Cdr(cell)	(((cell_t *)(cell))->value.cons.cdr)
#define new_cell()	(cons(NIL, NIL))
#define eq(x, y)	(x == y)

static int (*cha_getc)(), (*cha_ungetc)();
static char *s_tostr();

static int is_bol = 1;
static int c_stacked = EOF;

static int cha_getc_server(fp)
    FILE *fp;
{
    int c;

    if (c_stacked != EOF) {
	c = c_stacked;
	c_stacked = EOF;
    } else
      c = getc(fp);

    /* skip '\r' */
    if (c == '\r')
      c = getc(fp);

    if (c == '.' && is_bol) {
	/* skip '\r' */
	if ((c = getc(fp)) == '\r')
	  c = getc(fp);
	if (c == '\n')
	  c = EOF;
    }

    is_bol = c == '\n' ? 1 : 0;

#if 0
    putc(c,stdout);fflush(stdout);
#endif

    return c;
}

static int cha_ungetc_server(c, fp)
    int c;
    FILE *fp;
{
    c_stacked = c;
}

void set_cha_getc_alone()
{
    int fgetc(), ungetc();

    cha_getc = fgetc;
    cha_ungetc = ungetc;
}

void set_cha_getc_server()
{
    cha_getc = cha_getc_server;
    cha_ungetc = cha_ungetc_server;
}


/*
------------------------------------------------------------------------------
	local error processing
------------------------------------------------------------------------------
*/

static cell_t *error_in_lisp()
{
    cha_exit_file(1, "premature end of file or string\n");
    return NIL;
}

/*
------------------------------------------------------------------------------
	FUNCTION
	<ifnextchar>: if next char is <c> return 1, otherwise return 0
------------------------------------------------------------------------------
*/

static int ifnextchar(fp, i)
    FILE *fp;
    int i;
{
    int c;

    do {
	c = cha_getc(fp);
	if (c == '\n') LineNo++;
    } while (c == ' ' || c == '\t' || c == '\n');

    if (c == EOF)
      return EOF;

    if (i == c) 
      return TRUE;
    else {
	cha_ungetc(c, fp);
	return FALSE;
    }
}

/*
 * skip comment lines
 */
static int skip_comment(fp)
    FILE *fp;
{
    int n, c;

    while ((n = ifnextchar(fp, (int)COMMENTCHAR)) == TRUE) {
	while ((c = cha_getc(fp)) != '\n')
	  if (c == EOF)
	    return c;
	LineNo++;
    }

    return n;
}

int s_feof(fp)
    FILE *fp;
{
    int c;

    /* init the pointer to output functions */
    if (cha_getc == NULL)
      set_cha_getc_alone();

    for (;;) {
	if (skip_comment(fp) == EOF)
	  return TRUE;
	if ((c = cha_getc(fp)) == '\n')
	    LineNo++;
	else if (c == ' ' || c == '\t')
	  ;
	else {
	    cha_ungetc(c, fp);
	    return FALSE;
	}	
    }
}

/*
------------------------------------------------------------------------------
	FUNCTION
	<make_cell>: make a new cell
------------------------------------------------------------------------------
*/

static cell_t *cell_malloc_free(isfree)
    int isfree;
{
    static cell_t *ptr[1024*16];
    static int ptr_num = 0;
    static int idx = CELLALLOCSTEP;

    if (isfree) {
	/* free */
	if (ptr_num > 0) {
	    while (ptr_num > 1)
	      free(ptr[--ptr_num]);
	    idx = 0;
	}
	return NULL;
    } else {
	if (idx == CELLALLOCSTEP) {
	    if (ptr_num == 1024*16)
	      cha_exit(1, "Can't allocate memory");
	    ptr[ptr_num++] = cha_malloc(sizeof(cell_t) * idx);
	    idx = 0;
	}
	return ptr[ptr_num - 1] + idx++;
    }

#if 0
    return malloc(sizeof(cell_t));
#endif
}

#define CHA_MALLOC_SIZE (1024 * 64)
static void *malloc_char(size)
    int size;
{
    static char *ptr[128];
    static int ptr_num = 0;
    static int idx = CHA_MALLOC_SIZE;

    if (size < 0) {
	/* free */
	if (ptr_num > 0) {
	    while (ptr_num > 1)
	      free(ptr[--ptr_num]);
	    idx = 0;
	}
	return NULL;
    } else {
	if (idx + size >= CHA_MALLOC_SIZE) {
	    if (ptr_num == 128)
	      cha_exit(1, "Can't allocate memory");
	    ptr[ptr_num++] = cha_malloc(CHA_MALLOC_SIZE);
	    idx = 0;
	}
	idx += size;
	return ptr[ptr_num - 1] + idx - size;
    }
}

static char *lisp_strdup(str)
    char *str;
{
    char *newstr;

    newstr = malloc_char(strlen(str) + 1);
    strcpy(newstr, str);

    return newstr;
}

void s_free(cell)
    cell_t *cell;
{
#if 1
    cell_malloc_free(1);
    malloc_char(-1);
#else
    if (atomp(cell)) {
	free(cell->value.atom);
    } else if (consp(cell)) {
	s_free(car(cell));
	s_free(cdr(cell));
    } else {
	return;
    }

    free(cell);
#endif
}

/*
------------------------------------------------------------------------------
	FUNCTION
	<tmp_atom>: use <TmpCell>
------------------------------------------------------------------------------
*/

cell_t *tmp_atom(atom)
    char *atom;
{
    static cell_t _TmpCell;
    static cell_t *TmpCell = &_TmpCell;

    s_tag(TmpCell) = ATOM;
    s_atom_val(TmpCell) = atom;

    return TmpCell;
}

/*
------------------------------------------------------------------------------
	FUNCTION
	<cons>: make <cons> from <car> & <cdr>
------------------------------------------------------------------------------
*/

cell_t *cons(car, cdr)
    void *car, *cdr;
{
    cell_t *cell;

    cell = cell_malloc_free(0);
    s_tag(cell) = CONS;
    _Car(cell) = car;
    _Cdr(cell) = cdr;

    return cell;
}

/*
------------------------------------------------------------------------------
	FUNCTION
	<car>: take <car> from <cons>
------------------------------------------------------------------------------
*/

cell_t *car(cell)
    cell_t *cell;
{
    if (consp(cell))
      return _Car(cell);

    if (nullp(cell))
      return NIL;

    /* error */
    cha_exit_file(1, "%s is not list", s_tostr(cell));
    Cha_errno = 1;
    return NIL;
}

/*
------------------------------------------------------------------------------
	FUNCTION
	<cdr>: take <cdr> from <cons>
------------------------------------------------------------------------------
*/

cell_t *cdr(cell)
    cell_t *cell;
{
    if (consp(cell))
      return _Cdr(cell);

    if (nullp(cell))
      return NIL;

    /* error */
    cha_exit_file(1, "%s is not list\n", s_tostr(cell));
    return NIL;
}

char *s_atom(cell)
    cell_t *cell;
{
    if (atomp(cell))
      return s_atom_val(cell);

    /* error */
    cha_exit_file(1, "%s is not atom\n", s_tostr(cell));
    return NILSYMBOL;
}

/*
------------------------------------------------------------------------------
	FUNCTION
	<equal>:
------------------------------------------------------------------------------
*/

int equal(x, y)
    void *x, *y;
{
    if (eq(x, y)) return TRUE;
    if (nullp(x) || nullp(y)) return FALSE;
    if (s_tag(x) != s_tag(y)) return FALSE;
    if (s_tag(x) == ATOM) return !strcmp(s_atom_val(x), s_atom_val(y));
    if (s_tag(x) == CONS)
      return (equal(_Car(x), _Car(y)) && equal(_Cdr(x), _Cdr(y)));
    return FALSE;
}

int s_length(list)
    cell_t *list;
{
    int i;

    for (i = 0; consp(list); i++)
      list = _Cdr(list);

    return i;
}

static int dividing_code_p(code)
    int code;
{
    switch (code) {
      case '\n': case '\t': case ';': 
      case '(': case ')': case ' ':      
	return 1;
      default:
	return 0;
    }
}

static int myscanf(fp, cp)
    FILE *fp;
    char *cp;
{
    int code;

    code = cha_getc(fp);
    if (dividing_code_p(code) || code == EOF)
      return 0;

    if (code == '"') {
	while (1) {
	    if ((code = cha_getc(fp)) == EOF)
	      return 0;
	    else if (code == '"')
	      break;
	    else if (code == '\\') {
		*cp++ = code;
		if ((code = cha_getc(fp)) == EOF) 
		  return 0;
		*cp++ = code;
	    }	       
	    else
	      *cp++ = code;
	}
    }
    else {
	while (1) {
	    *cp++ = code;
	    code = cha_getc(fp);
	    if (dividing_code_p(code) || code == EOF) {
		cha_ungetc(code, fp);
		break;
	    }
	}
    }

    *cp = '\0';
    return 1;
}

/*
------------------------------------------------------------------------------
	FUNCTION
	<s_read>: read S-expression
------------------------------------------------------------------------------
*/

static cell_t *s_read_atom(fp)
    FILE *fp;
{
    cell_t *cell;
    char buffer[BUFSIZ];

    skip_comment(fp);

    /* changed by kurohashi. */
    if (myscanf(fp, buffer) == 0)
      return error_in_lisp();

    if (!strcmp(buffer, NILSYMBOL))
      return NIL;

    cell = new_cell();
    s_tag(cell) = ATOM;
    s_atom_val(cell) = lisp_strdup(buffer);

    return cell;
}

static cell_t *s_read_cdr();
static cell_t *s_read_main();

static cell_t *s_read_car(fp)
    FILE *fp;
{
    cell_t *cell;

    skip_comment(fp);

    switch (ifnextchar(fp, (int)EPARENTHESIS)) {
      case TRUE:
	return NIL;
      case FALSE:
	cell = new_cell();
	_Car(cell) = s_read_main(fp);
	_Cdr(cell) = s_read_cdr(fp);
	return cell;
      default: /* EOF */
	return error_in_lisp();
    }
}

static cell_t *s_read_cdr(fp)
    FILE *fp;
{
    skip_comment(fp);

    switch (ifnextchar(fp, (int)EPARENTHESIS)) {
      case TRUE:
	return NIL;
      case FALSE:
	return s_read_car(fp);
      default: /* EOF */
	return error_in_lisp();
    }
}

static cell_t *s_read_main(fp)
    FILE *fp;
{
    switch (ifnextchar(fp, (int)BPARENTHESIS)) {
      case TRUE:
	return s_read_car(fp);
      case FALSE:
	return s_read_atom(fp);
      default: /* EOF */
	return error_in_lisp();
    }
}

cell_t *s_read(fp)
    FILE *fp;
{
    /* init the pointer to output functions */
    if (cha_getc == NULL)
      set_cha_getc_alone();

    if (LineNo == 0)
      LineNo = 1;
    LineNoForError = LineNo;

    return s_read_main(fp);
}

/*
------------------------------------------------------------------------------
	FUNCTION
	<assoc>:
------------------------------------------------------------------------------
*/

cell_t *assoc(item, alist)
    cell_t *item, *alist;
{
    while (!nullp(alist) && !equal(item, (car(car(alist)))))
      alist = cdr(alist);
    return car(alist);
}

/*
------------------------------------------------------------------------------
	PROCEDURE
	<s_print>: pretty print S-expression
------------------------------------------------------------------------------
*/

static char cell_buffer_for_print[8192];

static void s_puts_to_buffer(str)
    char *str;
{
    static int idx = 0;
    
    /* initialization */
    if (str == NULL) {
	idx = 0;
	return;
    }

    if (idx + strlen(str) >= sizeof(cell_buffer_for_print))
      idx = sizeof(cell_buffer_for_print);
    else
      strcpy(cell_buffer_for_print + idx, str);
}

static void s_tostr_cdr(cell)
    cell_t *cell;
{
    if (!nullp(cell)) {
	if (consp(cell)) {
	    s_puts_to_buffer(" ");
	    s_tostr(_Car(cell));
	    s_tostr_cdr(_Cdr(cell));
	} else {
	    s_puts_to_buffer(" ");
	    s_tostr(cell);
	}
    }
}

static char *s_tostr(cell)
    cell_t *cell;
{
    /* initialization */
    s_puts_to_buffer(NULL);

    if (nullp(cell))
      s_puts_to_buffer(NILSYMBOL);
    else {
	switch (s_tag(cell)) {
	  case CONS:
	    s_puts_to_buffer("(");
	    s_tostr(_Car(cell));
	    s_tostr_cdr(_Cdr(cell));
	    s_puts_to_buffer(")");
	    break;
	  case ATOM:
	    s_puts_to_buffer(s_atom_val(cell));
	    break;
	  default:
	    s_puts_to_buffer("INVALID_CELL");
	}
    }

    return cell_buffer_for_print;
}

cell_t *s_print(fp, cell)
    FILE *fp;
    cell_t *cell;
{
    fputs(s_tostr(cell), fp);
    return cell;
}

