/*
 * parse.c - parse a sentence
 *
 * Copyright (C) 1996,1997 Nara Institute of Science and Technology
 *
 * Modified by: A.Kitauchi <akira-k@is.aist-nara.ac.jp>, Oct. 1996
 */

#include "chalib.h"
#include "pat.h"

#define MRPH_NUM	        1024
#define PATH1_NUM		256

#define HANKAKU            	0x80
#define PRIOD            	0xa1a5
#define CHOON            	0xa1bc
#define KIGOU            	0xa3b0
#define SUJI           	        0xa3c0
#define ALPH            	0xa4a0
#define HIRAGANA                0xa5a0
#define KATAKANA                0xa6a0
#define GR                      0xb0a0
#define KANJI                   0xffff
#define ILLEGAL                 1

mrph2_t *Mrph;
path_t *Path;
int Path_num;

/***********************************************************************
 * malloc_chars
 ***********************************************************************/
#define CHA_MALLOC_SIZE (1024 * 64)
#define malloc_char(n)     malloc_chars(1, n)
#define malloc_short(n)    malloc_chars(2, n)
#define malloc_int(n)      malloc_chars(4, n)
#define free_chars()       malloc_chars(0, 0)
static void *malloc_chars(size, nitems)
    int nitems, size;
{
    static char *buffer_ptr[128];
    static int buffer_ptr_num = 0;
    static int buffer_idx = CHA_MALLOC_SIZE;

    if (nitems == 0) {
	/* free */
	if (buffer_ptr_num > 0) {
	    while (buffer_ptr_num > 1)
	      free(buffer_ptr[--buffer_ptr_num]);
	    buffer_idx = 0;
	}
	return NULL;
    } else {
	if (size > 1) {
	    /* size ǳ꤭ͤ */
	    buffer_idx+= size - (buffer_idx & (size - 1));
	    nitems *= size;
	}

	if (buffer_idx + nitems >= CHA_MALLOC_SIZE) {
	    if (buffer_ptr_num == 128)
	      cha_exit(1, "Can't allocate memory");
	    buffer_ptr[buffer_ptr_num++] = cha_malloc(CHA_MALLOC_SIZE);
	    buffer_idx = 0;
	}

	buffer_idx += nitems;
	return buffer_ptr[buffer_ptr_num - 1] + buffer_idx - nitems;
    }
}

static void *malloc_free_block(ptr, nblockp, size, do_free)
    void *ptr;
    int *nblockp, size, do_free;
{
    if (do_free) {
	/* free and malloc one block */
	if (*nblockp > 1) {
#if 0
	    printf("# free block (%d)\n",size); fflush(stdout);
#endif
	    free(ptr);
	    *nblockp = 0;
	}
	if (*nblockp == 0)
	  ptr = malloc_free_block(ptr, nblockp, size, 0);
    } else {
	/* realloc one block larger */
	if (*nblockp == 0)
	  ptr = malloc(size * ++*nblockp);
	else {
#if 1
	    ptr = realloc(ptr, size * ++*nblockp);
#else
	    {
		char *ptr2;
		ptr2 = cha_malloc(size * (*nblockp + 1));
		memcpy(ptr2, ptr, size * *nblockp);
		(*nblockp)++;
		free(ptr);
		ptr = ptr2;
	    }
#endif
#if 0
	printf("# %s block (%d*%d)\n",*nblockp?"realloc":"malloc",size,*nblockp); 
	fflush(stdout);
#endif
	}
#if 0
	if (ptr == NULL)
	  printf("# Can't allocate memory"); fflush(stdout);
#endif
    }

    return ptr;
}

#define malloc_path()  malloc_free_path(0)
#define free_path()    malloc_free_path(1)
static int malloc_free_path(do_free)
    int do_free;
{
    static int nblock = 0;

#if 0
    printf("# path %d:%d ", nblock, Path_num);
#endif

    Path = malloc_free_block((void *)Path, &nblock,
			     sizeof(path_t) * PATH_NUM, do_free);

    return Path == NULL;
}

#define malloc_mrph()  malloc_free_mrph(0)
#define free_mrph()    malloc_free_mrph(1)
static int malloc_free_mrph(do_free)
    int do_free;
{
    static int nblock = 0;

#if 0
    printf("# mrph %d ", nblock);
#endif
    Mrph = malloc_free_block((void *)Mrph, &nblock,
			     sizeof(mrph2_t) * MRPH_NUM, do_free);

    return Mrph == NULL;
}

/***********************************************************************
 * check_code()
 ***********************************************************************/
static int check_code(str)
    char *str;
{
    int	code;
    unsigned char *s = (unsigned char *)str;

    /* nyuuryoku chuuni hankaku space wo yurusu, by. T.U. '96.01.10 */
#if 1
    if (*s == '\0' || *s == ' ')
      return 0;
#else
    if (*s == '\0')
      return 0;
#endif
    else if (*s < HANKAKU)
      return HANKAKU;
    else if (*(s+1) < HANKAKU)
      return ILLEGAL;

    code = *s * 256 + *(s + 1);

    if (code == PRIOD)        return PRIOD;
    else if (code == CHOON)   return CHOON;
    else if (code < KIGOU)    return KIGOU;
#if 0
    else if (code < SUJI)     return SUJI;
#endif
    else if (code < ALPH)     return ALPH;
    else if (code < HIRAGANA) return HIRAGANA;
    else if (code < KATAKANA) return KATAKANA;
    else if (code < GR)       return GR;
    else return KANJI;
}

/***********************************************************************
 * undef_mrph_len - ̤ĹĴ٤
 *
 * Ҥ餬ʡ: 1ʸ
 * ʡ޻ʤ: Ϣ³ʸ
 ************************************************************************/
static int undef_mrph_len(target)
    char *target;
{
    int	code, next_code;
    int len = 0;

    code = check_code(target);

    if (code == HIRAGANA || code == KANJI)
      return 2;

    do {
	if (code == HANKAKU || code == ILLEGAL)
	  len++;
	else
	  len += 2;
	next_code = check_code(target + len);
    } while (next_code == code
	     || (code == KATAKANA && next_code == CHOON)
	     || (code == ALPH     && next_code == PRIOD));

    return len;
}

/***********************************************************************
 *  ̤Хåեɲ
 ***********************************************************************/
static int register_undef_mrph(target, mrph_idx, undef_len)
    char *target;
    int mrph_idx, undef_len;
{
#if 0
    int undef_len;
#endif
    mrph2_t *mrph = &Mrph[mrph_idx];

#if 0
    undef_len = undef_mrph_len(target);
#endif

    mrph->midasi = target;
    mrph->base_length = mrph->length = undef_len;

    mrph->yomi = (char *)malloc_char(undef_len + 1);
    strncpy(mrph->yomi, target, undef_len);
    mrph->yomi[undef_len] = '\0';

    mrph->hinsi = Undef_hinsi;
#ifndef VGRAM
    mrph->bunrui = Undef_bunrui;
#endif
    mrph->con_tbl = Undef_con_tbl;
    mrph->ktype = 0;
    mrph->kform = 0;
    mrph->is_undef = 1; /* ̤ */
    mrph->weight = MRPH_DEFAULT_WEIGHT;
    mrph->info = "NIL"; /* ̣ʸ NIL Ȥ롥 */

    if (++mrph_idx % MRPH_NUM == 0 && malloc_mrph())
      return FALSE;

    return TRUE;
}

/*
------------------------------------------------------------------------------
        PROCEDURE: <take_data>                  >>> added by T.Nakamura <<<
------------------------------------------------------------------------------
*/
char *take_data(mrph, pbuf, target)
    mrph2_t *mrph;
    char *pbuf, *target;
{
    char *ret;
    char *p, *s;
    int  i;

    p = pbuf;

    /* ФĹ */
    for (i = 0; *p++ != '\t'; i++);
    mrph->base_length = mrph->length = i;
    /* ʻʬ No. */
    for (i = 0; *p != ' '; i = i * 10 + *p++ - '0');
    mrph->hinsi = i; p++;
#ifndef VGRAM
    /* ʻʬ No. */
    for (i = 0; *p != ' '; i = i * 10 + *p++ - '0');
    mrph->bunrui = i; p++;
#endif
    /* ѷ No. */
    for (i = 0; *p != ' '; i = i * 10 + *p++ - '0');
    mrph->ktype = i; p++;
    /* ѷ No. */
    for (i = 0; *p != ' '; i = i * 10 + *p++ - '0');
    mrph->kform = i; p++;
    /* Ť */
#ifndef VGRAM
    if (*p < '0' || *p > '9')
      cha_exit(1, "The format of the patricia dictionary is invalid.");
#endif
    for (i = 0; *p != ' '; i = i * 10 + *p++ - '0');
    mrph->weight = i; p++;
    /* ɤ */
    s = strchr(p, ' ');
    *s = '\0';
    mrph->yomi = (char *)malloc_char(s - p + 1);
    strcpy(mrph->yomi, p);
    p = s + 1;
    /* ³ơֹ֥ */
#ifdef VGRAM
    if (*p < '0' || *p > '9')
      cha_exit(1, "The format of the patricia dictionary is invalid.");
#endif
    for (i = 0; *p != ' '; i = i * 10 + *p++ - '0');
    mrph->con_tbl = i; p++;
    /* ̣ */
    s = strchr(p, '\n');
    *s = '\0';
    mrph->info = (char *)malloc_char(s - p + 1);
    strcpy(mrph->info, p);

    ret = s + 1;

    mrph->midasi = target;
    mrph->is_undef = 0;

    return ret;
}

/***********************************************************************
 * compare_top_str1 - compare strings to prefix of s1
 ***********************************************************************/
static int compare_top_str1(s1, s2)
    char *s1, *s2;
{
    /* 1996.07.28 by akira-k */
    for (;;) {
	if (*s1 == '\0') return TRUE;
	if (*s1++ != *s2++) return FALSE;
    }
}

/***********************************************************************
 * katuyou_process - find all forms which match the follow string
 ***********************************************************************/
static int *katuyou_process(follows, ktype)
    char *follows;
    int ktype;
{
    static int formstr[FORM_NO];
    int *fs = formstr;
    int f;

    for (f = 1; Form[ktype][f].name; f++)
      if (compare_top_str1(Form[ktype][f].gobi, follows))
	*fs++ = f;
    *fs = 0;

    return formstr;
}

/***********************************************************************
 * register_mrph - ѤĴ٤ʤ1ĤηǤХåեɲ
 *
 * mrph_nump: ХåեɲäǤο
 ************************************************************************/
static int register_mrph(mrph_idx, nmrphp)
    int mrph_idx, *nmrphp;
{
    int new_mrph_idx = mrph_idx;
    mrph2_t *new_mrph = &Mrph[mrph_idx];

    if (
#ifdef VGRAM
	Hinsi[new_mrph->hinsi].kt == 1
#else
	Class[new_mrph->hinsi][new_mrph->bunrui].kt
#endif
	) {
	/* Ѥ */
	if (new_mrph->kform == 0) {
	    /* 촴 */
	    int *f;
	    int ktype = new_mrph->ktype;
	    int length = new_mrph->length;
	    int con_tbl = new_mrph->con_tbl;
	    char *follows = new_mrph->midasi + new_mrph->base_length;
	    mrph2_t *mrph0 = new_mrph;
	    for (f = katuyou_process(follows, ktype); *f; f++) {
		if (new_mrph != mrph0)
		  *new_mrph = *mrph0;
		new_mrph->kform = *f;
		new_mrph->length = length + strlen(Form[ktype][*f].gobi);
		new_mrph->con_tbl = con_tbl + *f - 1;
		if (++new_mrph_idx % MRPH_NUM == 0 && malloc_mrph())
		  return FALSE;
		new_mrph = &Mrph[new_mrph_idx];
	    }
	} else {
	    /* 촴ʤ */
	    new_mrph->base_length = 0;
	    new_mrph->yomi = "";
	    if (++new_mrph_idx % MRPH_NUM == 0 && malloc_mrph())
	      return FALSE;
	}
    } else {                         /* Ѥʤ */
	if (++new_mrph_idx % MRPH_NUM == 0 && malloc_mrph())
	  return FALSE;
    }

    *nmrphp = new_mrph_idx - mrph_idx;

    return TRUE;
}

/***********************************************************************
 * convert_mrphs - ѤĴ٤ʤǤХåեɲ
 *
 * mrph_nump: ХåեɲäǤο
 ***********************************************************************/
static int convert_mrphs(target, pat_buffer, mrph_idx, mrph_nump)
    char *target, *pat_buffer;
    int mrph_idx, *mrph_nump;
{
    int nmrph;
    int new_mrph_idx = mrph_idx;
    char *pbuf;

    for (pbuf = pat_buffer; *pbuf; ) {
	pbuf = take_data(&Mrph[new_mrph_idx], pbuf, target);
#if 0
	printf("%4d: ", new_mrph_idx);
	print_mrph(0, &Mrph[new_mrph_idx], 'f', NULL);
#endif
	if (register_mrph(new_mrph_idx, &nmrph) == FALSE)
	  return FALSE;
	new_mrph_idx += nmrph;
    }

    *mrph_nump = new_mrph_idx - mrph_idx;

    return TRUE;
}

/*
------------------------------------------------------------------------------
  PROCEDURE: <trim_space> ڡκ (ȾѥڡΤߤѹ by T.U
                                                                   '96.01.11)
------------------------------------------------------------------------------
*/
static int trim_space(str, pos)
    char *str;
    int pos;
{
    for (;;) {
	/* Ⱦѥڡ */
	if (str[pos] == ' ')
	  pos++;
#if 0
	/* ѥڡ */
	else if ((unsigned char)str[pos  ] == 0xA1 &&
		 (unsigned char)str[pos+1] == 0xA1)
	  pos += 2;
#endif
	else
	  break;
    }
    return pos;
}

/*
------------------------------------------------------------------------------
  PROCEDURE: <pos_match_process>
------------------------------------------------------------------------------
*/
static void pos_match_process(pos, p_idx)
    int pos, *p_idx;
{
    static int p_start;
    int i, j;

    j = 0;
    if (pos == 0) {
	/* new sentence */
	p_idx[j++] = 0;
	p_start = 1;
    } else {
	for (i = p_start; i < Path_num; i++) {
	    if (Path[i].end <= pos) {
		if (i == p_start)
		  p_start++;
		if (Path[i].end == pos)
		  p_idx[j++] = i;
	    }
	}
    }
    p_idx[j] = -1;
}

/*
------------------------------------------------------------------------------
  PROCEDURE: <check_connect>
------------------------------------------------------------------------------
*/
static int check_connect(pos, m_num, p_idx)
    int pos, m_num, *p_idx;
{
    /* ֤ͤǥѥʬह */
    typedef struct _path_cost_t {
	int min_cost;
	short min_cost_no;
	short state;
	short num;
	int   cost[PATH1_NUM];
	int   pno[PATH1_NUM];
    } path_cost_t;

    static path_cost_t pcost[PATH1_NUM];
    int pcost_num;
    int path[PATH1_NUM], *new_path;
    mrph2_t *new_mrph;
    int i, pno, pcostno, npath;
    int	haba_cost, con_cost, cost, mrph_cost;
    int con_tbl, next_state;

#ifdef DEBUG
    printf("[m:%d] ", m_num);
#endif
    new_mrph = &Mrph[m_num];
    con_tbl = new_mrph->con_tbl;

    pcost[0].state = -1;
    pcost_num = 0;

    for (i = 0; (pno = p_idx[i]) >= 0; i++) {
	/* ȥޥȥĴ٤Ƽ֤³ȤФ */
	next_state = check_automaton
	  (Path[pno].state,
#ifdef KOCHA
	   Path[Path[pno].path[0]].state,
	   Mrph[Path[pno].mrph_p].midasi,
	   Mrph[Path[pno].mrph_p].is_undef,
#endif
	   con_tbl, Undef_con_cost, &con_cost);

#ifdef DEBUG
	printf("[m1:%d,m2:%d,cost:%d,state:%d]\n",
	       Path[pno].mrph_p,m_num,con_cost,next_state);
#endif
	if (con_cost == 0)
	  continue;

	/* cost ׻ */
	cost = Path[pno].cost + con_cost * Con_cost_weight;

	/* ɤ pcost °뤫Ĵ٤ */
	for (pcostno = 0; pcostno < pcost_num; pcostno++)
	  if (next_state == pcost[pcostno].state)
	    break;
	if (pcostno < pcost_num) {
	    if (pcost[pcostno].min_cost < cost - Cost_width)
	      continue;
	} else {
	    /*  pcost  */
	    pcost_num++;
	    pcost[pcostno].num = 0;
	    pcost[pcostno].state = next_state;
	    pcost[pcostno].min_cost = INT_MAX;
	}

	/* pcost Ͽ */
	pcost[pcostno].cost[pcost[pcostno].num] = cost;
	pcost[pcostno].pno[pcost[pcostno].num] = pno;
	if (cost < pcost[pcostno].min_cost) {
	    pcost[pcostno].min_cost = cost;
	    pcost[pcostno].min_cost_no = pcost[pcostno].num;
	}
	pcost[pcostno].num++;
    }

    if (pcost_num == 0)
      return TRUE;

    /* ǥ */
    if (new_mrph->is_undef)
      mrph_cost = Undef_cost /* *new_mrph->length */ ;
    else {
#ifdef VGRAM
	mrph_cost = Hinsi[new_mrph->hinsi].cost;
#else
	mrph_cost = Class[new_mrph->hinsi][new_mrph->bunrui].cost;
#endif
    }
    mrph_cost *= new_mrph->weight * Mrph_cost_weight;

#ifdef KOCHA
    if (new_mrph->midasi &&
	(unsigned char)new_mrph->midasi[0] == 0xA1 &&
	(unsigned char)new_mrph->midasi[1] == 0xA1) {
	for (i = 0; i < pcost[0].num; i++) {
	    new_path = malloc_int(2);
	    new_path[0] = pcost[0].pno[i];
	    new_path[1] = -1;

	    Path[Path_num].cost = pcost[0].cost[i] + mrph_cost;
	    Path[Path_num].mrph_p = m_num;
	    Path[Path_num].state = pcost[0].state;
	    Path[Path_num].start = pos;
	    Path[Path_num].end = pos + new_mrph->length;
	    Path[Path_num].path = new_path;
#ifdef KOCHA_DEBUG
	    printf("[Sp:%d,prev:%d,m:%d,c:%d,pc:%d]\n",
		   Path_num,Path[Path_num].path[0],m_num,pcost[0].cost[i],Path[Path_num].cost);
#endif
	    if (++Path_num % PATH_NUM == 0 && malloc_path())
	      return FALSE;
	}
    } else {
#endif /* KOCHA */
	for (pcostno = 0; pcostno < pcost_num; pcostno++) {
	    /* ˤޤäƤѥȴФ */
	    haba_cost = pcost[pcostno].min_cost + Cost_width;
	    npath = 0;
	    path[npath++] = pcost[pcostno].pno[pcost[pcostno].min_cost_no];
	    for (i = 0; i < pcost[pcostno].num; i++)
	      if (pcost[pcostno].cost[i] <= haba_cost && i != pcost[pcostno].min_cost_no)
		path[npath++] = pcost[pcostno].pno[i];
	    path[npath++] = -1;

	    new_path = malloc_int(npath);
	    memcpy(new_path, path, sizeof(int) * npath);

	    /* Path Ͽ */
	    Path[Path_num].cost = pcost[pcostno].min_cost + mrph_cost;
	    Path[Path_num].mrph_p = m_num;
	    Path[Path_num].state = pcost[pcostno].state;
	    Path[Path_num].start = pos;
	    Path[Path_num].end = pos + new_mrph->length;
	    Path[Path_num].path = new_path;
#ifdef DEBUG
	    printf("[p:%d,prev:%d,m:%d,c:%d,pc:%d]\n",
		   Path_num,Path[Path_num].path[0],m_num,pcost[0].cost[i],Path[Path_num].cost);
#endif
	    if (++Path_num % PATH_NUM == 0 && malloc_path())
	      return FALSE;
	}
#ifdef KOCHA
    }
#endif
    return TRUE;
}

static void set_mrph_end(mrph)
    mrph2_t *mrph;
{
    mrph->midasi = mrph->yomi = mrph->info = "NIL";
    mrph->base_length = mrph->length = 3;

    mrph->hinsi = 0;
#ifndef VGRAM
    mrph->bunrui = 0;
#endif

    mrph->con_tbl = 0;
    mrph->ktype = 0;
    mrph->kform = 0;

    mrph->is_undef = 0;
    mrph->weight = MRPH_DEFAULT_WEIGHT;
}

/***********************************************************************
 * chasen_sent() - ʸǲϤ
 *
 * return value:
 *     0 - ok
 *     1 - no result / too many morphs
 ***********************************************************************/
int chasen_sent(target)
    char *target;
{
    int  i, dic_no;
    int  pos, pos_end, length;
    int  path_idx[PATH1_NUM];
    int  mrph_idx, new_mrph_idx;
    int  is8bit;
    int  undef_len, need_undef;
    char pat_buffer[50000];
    static int path0 = -1;
#ifdef SJIS
    char euc_target[CHA_INPUT_SIZE];
#endif

#ifdef SJIS
    strcpy(euc_target,target);
    sjis2euc(euc_target);
    hankana2zenkana(target);
#endif

    /* Ⱦѥڡ, by. T.U. '96.01.11 */
    target += trim_space(target, 0);

    /* chomp */
    length = strlen(target);
    if (target[length - 1] == '\n')
      target[--length] = '\0';
    if (target[length - 1] == '\r')
      target[--length] = '\0';

    /* ԡιԤϥå */
    if (!target[0])
      return 1;

    free_chars();
    free_path();
    free_mrph();

    /* ʸƬ */
    Path[0].end = 0;
    Path[0].path = &path0;
    Path[0].cost = 0;
    Path[0].mrph_p = 0;
    Path[0].state = 0;

    Path_num = 1;
    set_mrph_end(&Mrph[0]);
    new_mrph_idx = mrph_idx = 1;

    for (pos = 0; pos < length;
	 pos += is8bit ? 2 : Mrph[mrph_idx-1].length) {
#if 0
	printf("# mrph %d\n", mrph_idx);
#endif
	pos_end = pos;
	/*  Ⱦѥڡ, by. T.U. '96.01.11 */
	if ((pos = trim_space(target, pos_end)) == length) {
	    length = pos_end;
	    break;
	}

	is8bit =
	  ((unsigned char)target[pos  ] & 0x80) && 
	  ((unsigned char)target[pos+1] & 0x80);

	pos_match_process(pos_end, path_idx);
	if (path_idx[0] < 0)
	  continue;

	/* (ʸΤ߸) */
	if (is8bit) {
#if 0
	    if (!((unsigned char)target[pos + 1] & 0x80)) {
		printf("Error: Illegal code in: %s\n", target);
		return 1;
	    }
#endif
	    for(dic_no = 0; dic_no < number_of_tree; dic_no++) {
		int nmrph;
		/* ѥȥꥷڤǤ򸡺 */
		pat_buffer[0] = '\0';
#ifdef SJIS
		pat_search(dic_file[dic_no], euc_target + pos,
			   &tree_top[dic_no], pat_buffer);
		euc2sjis((unsigned char *)pat_buffer);
#else
		pat_search(dic_file[dic_no], target + pos,
			   &tree_top[dic_no], pat_buffer);
#endif
		/* ѤĤķǤ Mrph ɲ */
		if (convert_mrphs(target + pos, pat_buffer, new_mrph_idx, &nmrph)
		    == FALSE)
		  goto error_end;
		new_mrph_idx += nmrph;
	    }
	}

#if 0
	/*
	 * ̤ϢܥȤ 0 ΤȤñ줬1Ĥ⼭Ǥʤä
	 * ̤ Mrph ɲ
	 */
	if (Undef_con_cost == 0 || mrph_idx == new_mrph_idx) {
	    if (register_undef_mrph(target + pos, new_mrph_idx) == FALSE)
	      goto error_end;
	    new_mrph_idx++;
	}
#endif

	/* ̤ĹĴ٤Ƥ */
#ifdef SJIS
	undef_len = undef_mrph_len(euc_target + pos);
#else
	undef_len = undef_mrph_len(target + pos);
#endif
	/* ̤νɬפɤ */
	need_undef = 1;

	/* ľΥѥȤ³å */
	for (i = mrph_idx; i < new_mrph_idx; i++) {
	    /* ̤ƱĹñ줬ˤ̤ɲäʤ */
	    if (Mrph[i].length == undef_len)
	      need_undef = 0;
	    if (check_connect(pos, i, path_idx) == FALSE)
	      goto error_end;
	}

	/* ̤ */
	if (Undef_con_cost == 0 || need_undef) {
	    if (register_undef_mrph(target + pos, new_mrph_idx, undef_len) == FALSE)
	      goto error_end;
	    if (check_connect(pos, new_mrph_idx, path_idx) == FALSE)
	      goto error_end;
	    new_mrph_idx++;
	}

	mrph_idx = new_mrph_idx;
    }

    /* ʸ */
    set_mrph_end(&Mrph[mrph_idx]);
    if (++mrph_idx % MRPH_NUM == 0 && malloc_mrph())
      goto error_end;

    pos_match_process(length, path_idx);
    if (check_connect(length, mrph_idx - 1, path_idx) == FALSE)
      goto error_end;

#ifdef DEBUG
    for (i = 1; i < mrph_idx - 1; i++) {
	printf("%4d: %4d ", i, Mrph[i].con_tbl);
	print_mrph(0, &Mrph[i], 'F', "%-11m %-11y %-11P3- %-14T  %F \n");
    }
#endif

#if 0
    printf("# num - mrph:%d path:%d, size - mrph:%d path:%d \n",
	   mrph_idx, Path_num, sizeof(mrph2_t), sizeof(path_t));
#endif

    return 0;

  error_end:
    printf("Error: Too many morphs: %s\n", target);
#if 0
    printf("# num - mrph:%d path:%d, size - mrph:%d path:%d \n",
	   mrph_idx, Path_num, sizeof(mrph2_t), sizeof(path_t));
#endif
    return 1;
}

