/*
 *	makemat.c - make table file and matrix file
 *
 *	last modified by A.Kitauchi <akira-k@is.aist-nara.ac.jp>, Nov. 1996
 *
 */

#include "chadic.h"

#define KANKEI_MAX       500
#define REN_TBL_MAX      3000
#define RENSETU_CELL_MAX 8192

#ifdef VGRAM
#define HINSI_ID_MAX USHRT_MAX
#else
#define HINSI_ID_MAX UCHAR_MAX
#endif

typedef struct _kankei_t {
#ifdef VGRAM
    unsigned short hinsi;
#else
    unsigned char hinsi;
    unsigned char bunrui;
#endif
    unsigned char type;
} kankei_t;

typedef struct _rensetu_pair2_t {
    short  i_pos;
    short  j_pos;
#ifdef VGRAM
    cell_t *hinsi;
#else
    unsigned char hinsi;
    unsigned char bunrui;
#endif
    unsigned char type;
    char   *form;
    char   *goi;
} rensetu_pair2_t;

typedef struct _rensetu_cell_t {
    short tbl;
    short prev;
    short has_rule;
} rensetu_cell_t;

static kankei_t       kankei_tbl[KANKEI_MAX];
static int            tbl_num;
static rensetu_pair_t rensetu_tbl[RENSETU_CELL_MAX];
static int            i_num;
static int            j_num;

static connect_rule_t **connect_mtr1;
static connect_rule_t **connect_mtr2;
#ifdef KOCHA
typedef unsigned short rensetu_mtr_t;
#else
typedef unsigned char  rensetu_mtr_t;
#endif
static rensetu_mtr_t **rensetu_mtr1;
static rensetu_mtr_t **rensetu_mtr2;

/***********************************************************************
 * read_kankei - read chasen's kankei file
 ***********************************************************************/
static void read_kankei()
{
    FILE *fp;
    cell_t *cell1, *cell2;
    int j = 0;
    int hinsi, bunrui, type;
    char *filepath;

    /* read only from current directory */
#ifdef VGRAM
    fp = cha_fopen(KANKEIFILE, "r", 1);
#else
    fp = cha_fopen2(KANKEIFILE, JM_KANKEIFILE, "r", 1);
#endif

    fprintf(stderr, "parsing %s\n", KANKEIFILE);

    while (! s_feof(fp)) {
	cell1 = s_read(fp);

#ifdef VGRAM
	hinsi = get_nhinsi_id(car(cell1));
#else
	hinsi = get_hinsi_id(s_atom(car(car(cell1))));
	if (nullp(cell2 = car(cdr(car(cell1)))) ||
	    !strcmp(s_atom(cell2), "*"))
	  bunrui = 0;
	else
	  bunrui = get_bunrui_id(s_atom(cell2), hinsi);
#endif
	cell1 = car(cdr(cell1));

	while (!nullp(cell2 = car(cell1))) {
	    type = get_type_id(s_atom(cell2));
	    kankei_tbl[j].hinsi  = hinsi;
#ifndef VGRAM
	    kankei_tbl[j].bunrui = bunrui;
#endif
	    kankei_tbl[j].type   = type;
	    /* 
	     * bug fixed by kurohashi on 92/03/16
	     */
	    if (++j >= KANKEI_MAX) 
	      cha_exit(1, "not enough size for KANKEI_MAX");
	    cell1 = cdr(cell1);
	}
	kankei_tbl[j].hinsi = HINSI_ID_MAX;
    }
}

/***********************************************************************
 * get_pair1
 ***********************************************************************/
static void get_pair1(cell, pair)
    cell_t *cell;
    rensetu_pair_t *pair;
{
    cell_t *cell_p;

    pair->hinsi  = 0;
#ifndef VGRAM
    pair->bunrui = 0;
#endif
    pair->type   = 0;
    pair->form   = 0;
    pair->goi    = NULL;

    if (nullp(cell_p = car(cell)))
      return;
#ifdef VGRAM
    pair->hinsi = get_nhinsi_id(cell_p);
#else
    pair->hinsi = get_hinsi_id(s_atom(cell_p));
    if (nullp(cell_p = car(cell = cdr(cell))))
      return;
    pair->bunrui = get_bunrui_id(s_atom(cell_p), pair->hinsi);
#endif

    if (nullp(cell_p = car(cell = cdr(cell))))
      return;
    pair->type = get_type_id(s_atom(cell_p));

    if (nullp(cell_p = car(cell = cdr(cell))))
      return;
    pair->form = get_form_id(s_atom(cell_p), pair->type);

    if (nullp(cell_p = car(cell = cdr(cell))))
      return;
    pair->goi = cha_strdup(s_atom(cell_p));
}

/***********************************************************************
 * get_pair2
 ***********************************************************************/
static void get_pair2(cell, pair)
    cell_t *cell;
    rensetu_pair2_t *pair;
{
    cell_t *cell_p;
    char *s;

#ifdef VGRAM
    pair->hinsi  = NULL;
#else
    pair->hinsi  = 0;
    pair->bunrui = 0;
#endif
    pair->type   = 0;
    pair->form   = NULL;
    pair->goi    = NULL;

    if (nullp(cell_p = car(cell)))
      return;

#ifdef VGRAM
    s = s_atom(car(cell_p));
    if (strcmp(JSTR_BUNTO, s) == 0 || strcmp(JSTR_BUNMATSU, s) == 0) {
	pair->hinsi = NULL;
	return;
    }
    pair->hinsi = cell_p;
#else
    s = s_atom(cell_p);
    if (strcmp(JSTR_BUNTO, s) == 0 || strcmp(JSTR_BUNMATSU, s) == 0) {
	pair->hinsi = HINSI_ID_MAX;
	return;
    }
    pair->hinsi = get_hinsi_id(s);

    if (nullp(cell_p = car(cell = cdr(cell))))
      return;
    pair->bunrui = get_bunrui_id(s_atom(cell_p), pair->hinsi);
#endif

    if (nullp(cell_p = car(cell = cdr(cell))))
      return;
    pair->type = get_type_id(s_atom(cell_p));

    if (nullp(cell_p = car(cell = cdr(cell))))
      return;

    if (strcmp(s = s_atom(cell_p), "*"))
      pair->form = cha_strdup(s);

    if (nullp(cell_p = car(cell = cdr(cell))))
      return;
    pair->goi = cha_strdup(s_atom(cell_p));
}

/***********************************************************************
 * match_pair1
 ***********************************************************************/
static int match_pair1(pair1, pair2)
    rensetu_pair_t *pair1, *pair2;
{
    if (pair1->hinsi == pair2->hinsi &&
#ifndef VGRAM
	pair1->bunrui == pair2->bunrui &&
#endif
	pair1->type == pair2->type &&
	(!pair2->form || pair1->form == pair2->form) &&
	!strcmp(pair1->goi, pair2->goi))
      return 1;

    return 0;
}

/***********************************************************************
 * match_pair2
 ***********************************************************************/
static int match_pair2(pair, tbl)
    rensetu_pair2_t *pair;
    rensetu_pair_t *tbl;
{
    /* ʸƬ, ʸ */
#ifdef VGRAM
    if (pair->hinsi == NULL)
      return tbl->hinsi == 0;
#else
    if (pair->hinsi == HINSI_ID_MAX)
      return tbl->hinsi == 0;
#endif

    if (
#ifdef VGRAM
	match_nhinsi(pair->hinsi, (int)tbl->hinsi) &&
#else
	(!pair->hinsi || pair->hinsi == tbl->hinsi) &&
	(!pair->bunrui || pair->bunrui == tbl->bunrui) &&
#endif
	(!pair->type || pair->type == tbl->type) &&
	(!pair->form ||
	 (tbl->form && !strcmp(pair->form, Form[tbl->type][tbl->form].name))) &&
	(!pair->goi || (tbl->goi && !strcmp(pair->goi, tbl->goi))))
      return 1;

    return 0;
}

/***********************************************************************
 * make_rensetu_tbl1 - register hinsi with goi
 ***********************************************************************/
void make_rensetu_tbl1(cell1, cnt)
    cell_t *cell1;
    int *cnt;
{
    int i;
    rensetu_pair_t r_pair;
    cell_t *cell11;

    for (; !nullp(cell11 = car(cell1)); cell1 = cdr(cell1)) {
#ifdef VGRAM
	if (nullp(car(cdr(cdr(cdr(cell11))))))
	  continue;
#else
	if (nullp(car(cdr(cdr(cdr(cdr(cell11)))))))
	  continue;
#endif

	get_pair1(cell11, &r_pair);

	for (i = 1; i < *cnt; i++)
	  if (match_pair1(&rensetu_tbl[i], &r_pair))
	    break;
	if (i < *cnt)
	  continue;

	if (r_pair.type) {
	    for (i = 1; Form[r_pair.type][i].name != NULL; i++) {
		rensetu_tbl[*cnt].hinsi  = r_pair.hinsi;
#ifndef VGRAM
		rensetu_tbl[*cnt].bunrui = r_pair.bunrui;
#endif
		rensetu_tbl[*cnt].type   = r_pair.type;
		rensetu_tbl[*cnt].form   = i;
		rensetu_tbl[*cnt].goi    = r_pair.goi;
		if (++*cnt >= REN_TBL_MAX)
		  cha_exit(1, "not enough size for table");
	    }
	} else {
	    rensetu_tbl[*cnt].hinsi  = r_pair.hinsi;
#ifndef VGRAM
	    rensetu_tbl[*cnt].bunrui = r_pair.bunrui;
#endif
	    rensetu_tbl[*cnt].type   = r_pair.type;
	    rensetu_tbl[*cnt].form   = r_pair.form;
	    rensetu_tbl[*cnt].goi    = r_pair.goi;
	    if (++*cnt >= REN_TBL_MAX)
	      cha_exit(1, "not enough size for table");
	}
    }
}

/***********************************************************************
 * make_rensetu_tbl2 - register hinsi
 ***********************************************************************/
static void make_rensetu_tbl2(hinsi, bunrui, cnt)
    int hinsi, bunrui, *cnt;
{
    int i, j;

    if (
#ifdef VGRAM
	Hinsi[hinsi].kt == 1
#else
	Class[hinsi][bunrui].kt
#endif
	) {
	/* Ѥ */
	for (i = 0; kankei_tbl[i].hinsi != HINSI_ID_MAX; i++) {
	    if (kankei_tbl[i].hinsi == hinsi
#ifndef VGRAM
		&& kankei_tbl[i].bunrui == bunrui
#endif
		) {
		for (j = 1; Form[kankei_tbl[i].type][j].name != NULL; j++) {
		    rensetu_tbl[*cnt].hinsi  = hinsi;
#ifndef VGRAM
		    rensetu_tbl[*cnt].bunrui = bunrui;
#endif
		    rensetu_tbl[*cnt].type   = kankei_tbl[i].type;
		    rensetu_tbl[*cnt].form   = j;
		    rensetu_tbl[*cnt].goi    = NULL;
		    if (++*cnt >= REN_TBL_MAX) 
		      cha_exit(1, "not enough size for table");
		}
	    }
	}
#ifdef KOCHA2
    } else if (Hinsi[hinsi].kt == 2) {
	extern char *Gomatsu[];
	for (i = 0; Gomatsu[i]; i++, (*cnt)++) {
	    rensetu_tbl[*cnt].hinsi  = hinsi;
	    rensetu_tbl[*cnt].type   = get_type_id(Gomatsu[i]);
	    rensetu_tbl[*cnt].form   = 0;
	    rensetu_tbl[*cnt].goi    = NULL;
	}
	if (*cnt >= REN_TBL_MAX) 
	    cha_exit(1, "not enough size for table");
#endif

    } else {
	/* Ѥʤ */
	rensetu_tbl[*cnt].hinsi  = hinsi;
#ifndef VGRAM
	rensetu_tbl[*cnt].bunrui = bunrui;
#endif
	rensetu_tbl[*cnt].type   = 0;
	rensetu_tbl[*cnt].form   = 0;
	rensetu_tbl[*cnt].goi    = NULL;
	/* bug fixed by kurohashi on 92/03/16 */
	if (++*cnt >= REN_TBL_MAX) 
	  cha_exit(1, "not enough size for table");
    }
}

/***********************************************************************
 * make_rensetu_tbl - register hinsi into table
 ***********************************************************************/
static int make_rensetu_tbl(fp)
    FILE *fp;
{
    cell_t *cell;
    int i, j, lines;
    int tbl_count = 1; /* 0 ʸƬʸ */

    /* äꤷƤΤơ֥Ͽ */
    for (lines = 0; !s_feof(fp); lines++) {
#ifdef VGRAM
	for (cell = car(s_read(fp)); !nullp(cell); cell = cdr(cell))
	  make_rensetu_tbl1(car(cell), &tbl_count);
#else
	cell = s_read(fp);
	make_rensetu_tbl1(car(cell), &tbl_count);
	make_rensetu_tbl1(car(cdr(cell)), &tbl_count);      
#endif
    }

    /* ѤŸƥơ֥Ͽ */
#ifdef VGRAM
    for (i = 1; Hinsi[i].name; i++)
      /* second argument is dummy for compatibility */
      make_rensetu_tbl2(i, 0, &tbl_count);
#else
    for (i = 1; Class[i][0].id; i++)
      for (j = 0; Class[i][j].id; j++)
	make_rensetu_tbl2(i, j, &tbl_count);
#endif
    tbl_num = tbl_count;

    /* print for check */
    fprintf(stderr, "table size: %d\n", tbl_num);

    return lines;
}

/***********************************************************************
 * variables and functions for rensetu_cell
 ***********************************************************************/
static rensetu_cell_t rensetu_cell[RENSETU_CELL_MAX];
static int cell_num;

static int new_cell1[3000], new_cell2[3000];
static int new_cell1_num, new_cell2_num;

static int search_rensetu_cell(tbl, prev)
    int tbl, prev;
{
    int i;

    for (i = 0; i < cell_num; i++)
      if (rensetu_cell[i].tbl == tbl)
	if (rensetu_cell[i].prev == prev)
	  return i;

    return -1;
}

/* c2  c1  suffix ɤ */
static int match_rensetu_cell_suffix(c1, c2)
    int c1, c2;
{
    int n;

    for (n = 0; c2 >= 0; n++) {
	if (rensetu_cell[c1].tbl != rensetu_cell[c2].tbl)
	  return 0;
	c1 = rensetu_cell[c1].prev;
	c2 = rensetu_cell[c2].prev;
    }

    return n;
}

static void match_rensetu_cell_tbl(tbl, cells)
    int tbl, *cells;
{
    int *c, i;

    c = cells;
    *c++ = tbl;
    for (i = tbl_num; i < cell_num; i++)
      if (tbl == rensetu_cell[i].tbl)
	*c++ = i;
    *c = -1;
}

/***********************************************************************
 * add_connect_rule
 ***********************************************************************/
static void add_connect_rule(in, prev, cost, is_last, in_cells, cur_cells)
    int in, prev, cost, is_last, *in_cells, *cur_cells;
{
    int cur, next, *curp, *cellp;
    int suffix_len, suffix_len_max;

    next = 0; /* to avoid warning */
    match_rensetu_cell_tbl(rensetu_cell[prev].tbl, cur_cells);

    /* cell 椫 cur()򸡺 */
    for (curp = cur_cells; (cur = *curp) >= 0; curp++) {
	/* prev  cur  suffix ˤʤäƤ ok */
	if (!match_rensetu_cell_suffix(cur, prev))
	  continue;
	/* ǸʻǤʤϵ§񤭤ʤ */
	if (!is_last && connect_mtr1[cur][in].cost)
	  continue;
	suffix_len_max = 0;
	/* cell 椫 next()򸡺 */
	for (cellp = in_cells; *cellp >= 0; cellp++) {
	    /* cur+in  suffix ΤǤĹΤõ */
	    suffix_len = match_rensetu_cell_suffix(cur, rensetu_cell[*cellp].prev) + 1;
	    if (suffix_len_max < suffix_len) {
		suffix_len_max = suffix_len;
		next = *cellp;
	    }
	}
#ifdef DEBUG
	if (suffix_len_max>1) {
	    printf("suffix_len:%d,prev:%d,cur:%d,in:%d,next:%d,cost:%d\n",
		   suffix_len_max,prev,cur,in,next,cost);
	}
#endif
	/* §ɲ */
	if (suffix_len_max) {
	    connect_mtr1[cur][in].next = next - in;
	    connect_mtr1[cur][in].cost = cost;
	}
    }
}

#ifdef VGRAM

/***********************************************************************
 * read_rensetu
 ***********************************************************************/
static void read_rensetu(fp, lines)
    FILE *fp;
    int lines;
{
    cell_t **rule;
    int *rule_len;
    rensetu_pair2_t pair;
    cell_t *cell, *cell1;
    int  rule_len_max, rlen;
    int  prev, in, c1, ln, linenum, linecnt;
    int  cost, is_last;
    int  *in_cells, *cur_cells;
    connect_rule_t *ptr;

    rule = (cell_t **)cha_malloc(sizeof(cell_t *) * lines);
    rule_len = (int *)cha_malloc(sizeof(int) * lines);

    /* rensetu_cell ν */
    if (cell_num >= RENSETU_CELL_MAX)
      cha_exit(1, "not enough size for cell");
    for (cell_num = 0; cell_num < tbl_num; cell_num++) {
#if 0
	rensetu_cell[cell_num] = cha_malloc_rensetu_cell();
#endif
	rensetu_cell[cell_num].tbl = cell_num;
	rensetu_cell[cell_num].prev = -1;
    }

    rule_len_max = 0;
    for (ln = 0; !s_feof(fp); ln++) {
	rule[ln] = s_read(fp);
	/* ǤĹ§򸫤Ĥ */
	rule_len[ln] = s_length(car(rule[ln]));
	if (rule_len[ln] < 2)
	  cha_exit_file(1, "too few morphemes");
	if (rule_len_max < rule_len[ln])
	  rule_len_max = rule_len[ln];

	/* new_cell2: ʻϿ rensetu_cell */
	new_cell2[0] = -1; /* ʸƬʸ */
	new_cell2_num = 1;
	/* §ΥΡɤ */
	/* cell: ʻ췲Υꥹ */
	for (cell = car(rule[ln]); !nullp(cdr(cell)); cell = cdr(cell)) {
	    /* new_cell2  new_cell1 ˥ԡ */
	    memcpy(new_cell1, new_cell2, sizeof(int) * new_cell2_num);
	    new_cell1_num = new_cell2_num;
	    new_cell2_num = 0;
	    /* cell1: ʻ췲 */
	    for (cell1 = car(cell); !nullp(cell1); cell1 = cdr(cell1)) {
		int tbl;
		/* pair: 磻ɥɤĤʻ */
		get_pair2(car(cell1), &pair);
		/* pair  tbl(ʻ11)Фƽ */
		for (tbl = 0; tbl < tbl_num; tbl++) {
		    if (!match_pair2(&pair, &rensetu_tbl[tbl]))
		      continue;
		    /* c1, prev: 1ʻϿ줿cell */
		    for (c1 = 0; c1 < new_cell1_num; c1++) {
			int prev = new_cell1[c1], cellno;
			if ((cellno = search_rensetu_cell(tbl, prev)) < 0) {
			    cellno = cell_num;
			    if (++cell_num >= RENSETU_CELL_MAX)
			      cha_exit_file(1, "not enough size for cell");
			    rensetu_cell[cellno].tbl = tbl;
			    rensetu_cell[cellno].prev = prev;
#ifdef DEBUG
			    printf("cellno:%d,tbl:%d,prev:%d\n",cellno,tbl,prev);
#endif
			}
			new_cell2[new_cell2_num++] = cellno;
		    }
		}
	    }
	}
    }

    fprintf(stderr, "number of states: %d\n", cell_num);

    ptr = (connect_rule_t *)cha_malloc(sizeof(connect_rule_t) * cell_num * tbl_num);
    memset(ptr, 0, sizeof(connect_rule_t) * cell_num * tbl_num);
    connect_mtr1 = (connect_rule_t **)cha_malloc(sizeof(connect_rule_t *) * cell_num);
    for (c1 = 0; c1 < cell_num; c1++)
      connect_mtr1[c1] = ptr + c1 * tbl_num;

    ptr = (connect_rule_t *)cha_malloc(sizeof(connect_rule_t) * cell_num * tbl_num);
    connect_mtr2 = (connect_rule_t **)cha_malloc(sizeof(connect_rule_t *) * cell_num);
    for (c1 = 0; c1 < cell_num; c1++)
      connect_mtr2[c1] = ptr + c1 * tbl_num;

    in_cells = cha_malloc(sizeof(int) * cell_num);
    cur_cells = cha_malloc(sizeof(int) * cell_num);

    linenum = ln;
    linecnt = 0;
    printf("lines: %d\n", linenum);

    /* û§ν˽ */
    for (rlen = 2; rlen <= rule_len_max; rlen++) {
	if (rlen > 2)
	  printf("%d-gram:\n", rlen);
	for (ln = 0; ln < linenum; ln++) {
	    if (rule_len[ln] != rlen)
	      continue;
	    LineNoForError = LineNo = ln + 1;
#ifdef DEBUG
	    printf("Line: %d/%d\n",ln+1,linenum);
#endif
	    if (rlen > 2) {
		if ((linecnt % 100) == 0)
		  fputc('.', stdout);
		if ((linecnt % 5000) == 0)
		  printf(" %d\n", linecnt);
		fflush(stdout);
	    }
	    linecnt++;

	    cell = car(cdr(rule[ln]));
	    cost = nullp(cell) ? DEFAULT_C_WEIGHT : atoi(s_atom(cell));
	    is_last = 0;
	    /* new_cell2: ʻϿ rensetu_cell */
	    new_cell2[0] = -1; /* ʸƬʸ */
	    new_cell2_num = 1;
	    /* cell: ʻ췲Υꥹ */
	    for (cell = car(rule[ln]); !is_last; cell = cdr(cell)) {
		is_last = nullp(cdr(cell));
		/* new_cell2  new_cell1 ˥ԡ */
		memcpy(new_cell1, new_cell2, sizeof(int) * new_cell2_num);
		new_cell1_num = new_cell2_num;
		new_cell2_num = 0;
		/* cell1: ʻ췲 */
		for (cell1 = car(cell); !nullp(cell1); cell1 = cdr(cell1)) {
		    /* pair: 磻ɥɤĤʻ */
		    get_pair2(car(cell1), &pair);
		    /* pair  in(ʻ11)Фƽ */
		    for (in = 0; in < tbl_num; in++) {
			if (!match_pair2(&pair, &rensetu_tbl[in]))
			  continue;
			match_rensetu_cell_tbl(in, in_cells);
			/* c1, prev: 1ʻϿ줿cell */
			for (c1 = 0; c1 < new_cell1_num; c1++) {
			    prev = new_cell1[c1];
			    if (!is_last) {
				int cellno = search_rensetu_cell(in, prev);
				new_cell2[new_cell2_num++] = cellno;
			    }
			    /* §ɲ */
			    add_connect_rule(in, prev, cost, is_last,
					     in_cells, cur_cells);
			}
		    }
		}
	    }
	}
    }
    if (rule_len_max > 2)
      printf(" %d\n", linecnt);
}

#else /* !VGRAM */

/* modified by T.Utsuro for weight of rensetu matrix */
static void fill_matrix(pair1, pair2, c_weight)
    rensetu_pair2_t *pair1, *pair2;
    int c_weight;
{
    int i, j;
    int *tmp_tbl;
    int flag = 0;

    tmp_tbl = (int *)cha_malloc(sizeof(int) * tbl_num);

    for (j = 0; j < tbl_num; j++) {
	if (match_pair2(pair2, &rensetu_tbl[j]))
	  tmp_tbl[j] = 1;
	else
	  tmp_tbl[j] = 0;
    }

    for (i = 0; i < tbl_num; i++) {
	if (!match_pair2(pair1, &rensetu_tbl[i]))
	  continue;
	for (j = 0; j < tbl_num; j++) {
	    if (tmp_tbl[j]) { /* ˤ롼νŤߤͥ */
#ifdef KOCHA
		if (c_weight >= 100) 
		  rensetu_mtr1[i][j] =
		    rensetu_mtr1[i][j] % 100 + c_weight;
		else
		  rensetu_mtr1[i][j] =
		    rensetu_mtr1[i][j] - rensetu_mtr1[i][j] % 100 + c_weight;
#ifdef _KOCHA_DEBUG
		printf("[c:%d,rm:%d]\n",c_weight,rensetu_mtr1[i][j]);
#endif
#else		
		rensetu_mtr1[i][j] = c_weight;
#endif
		flag = 1;
	    }
	}
    }

    free(tmp_tbl);

    if (!flag) {
	cha_exit_file(-1, "invalid katsuyou form `%s' or `%s'",
		      pair1->form ? pair1->form : "(null)",
		      pair2->form ? pair2->form : "(null)");
    }
}

/***********************************************************************
 * read_rensetu
 ***********************************************************************/
static void read_rensetu(fp, lines)
    FILE *fp;
    int lines; /* dummy for compatibility with VGRAM */
{
    int i, j;
    cell_t *cell, *cell1, *cell11, *cell2, *cell22, *cell2_stock;
    rensetu_pair2_t pair1, pair2;
    int c_weight;
    rensetu_mtr_t *ptr;

    ptr = (rensetu_mtr_t *)cha_malloc(sizeof(rensetu_mtr_t) * tbl_num * tbl_num);
    memset(ptr, 0, sizeof(rensetu_mtr_t) * tbl_num * tbl_num);
    rensetu_mtr1 = (rensetu_mtr_t **)cha_malloc(sizeof(rensetu_mtr_t *) * tbl_num);
    for (i = 0; i < tbl_num; i++)
      rensetu_mtr1[i] = ptr + i * tbl_num;

    ptr = (rensetu_mtr_t *)cha_malloc(sizeof(rensetu_mtr_t) * tbl_num * tbl_num);
    rensetu_mtr2 = (rensetu_mtr_t **)cha_malloc(sizeof(rensetu_mtr_t *) * tbl_num);
    for (i = 0; i < tbl_num; i++)
      rensetu_mtr2[i] = ptr + i * tbl_num;

    while (! s_feof(fp)) {
	cell = s_read(fp);
	cell1 = car(cell);
	cell2 = car(cdr(cell));

	/* added by T.Utsuro for weight of rensetu matrix */
	if (nullp(cdr(cdr(cell))))
	  c_weight = DEFAULT_C_WEIGHT;
	else 
	  c_weight = atoi(s_atom(car(cdr(cdr(cell)))));

	cell2_stock = cell2;
	while (!nullp(cell11 = car(cell1))) {
	    get_pair2(cell11, &pair1);
	    cell2 = cell2_stock;

	    while (!nullp(cell22 = car(cell2))) {
		get_pair2(cell22, &pair2);
		fill_matrix(&pair1, &pair2, c_weight);
		cell2 = cdr(cell2);
	    }
	    cell1 = cdr(cell1);
	}
    }
}

#endif /* !VGRAM */

static int compare_vector1(k, j, num)
    int k, j, num;
{
    int i;
     
#ifdef VGRAM
    for (i = 0; i < num; i++)
      if (connect_mtr2[i][k].next != connect_mtr1[i][j].next ||
	  connect_mtr2[i][k].cost != connect_mtr1[i][j].cost)
	return 0;
#else
    for (i = 0; i < num; i++)
      if (rensetu_mtr2[i][k] != rensetu_mtr1[i][j])
	return 0;
#endif

    return 1;
}

static void copy_vector1(j, j_n, num)
    int j, j_n, num;
{
    int i;

#ifdef VGRAM
    for (i = 0; i < num; i++) {
	connect_mtr2[i][j_n].next = connect_mtr1[i][j].next;
	connect_mtr2[i][j_n].cost = connect_mtr1[i][j].cost;
    }
#else
    for (i = 0; i < num; i++)
      rensetu_mtr2[i][j_n] = rensetu_mtr1[i][j];
#endif
}

static int compare_vector2(k, i, num)
    int k, i, num;
{
    int j;

#ifdef VGRAM
    for (j = 0; j < num; j++)
      if (connect_mtr2[i][j].next != connect_mtr1[k][j].next ||
	  connect_mtr2[i][j].cost != connect_mtr1[k][j].cost)
	return 0;
#else
    for (j = 0; j < num; j++)
      if (rensetu_mtr2[i][j] != rensetu_mtr1[k][j])
	return 0;
#endif

    return 1;
}

static void copy_vector2(i, i_n, num)
    int i, i_n, num;
{
    int j;

#ifdef VGRAM
    for (j = 0; j < num; j++) {
	connect_mtr1[i_n][j].next = connect_mtr2[i][j].next;
	connect_mtr1[i_n][j].cost = connect_mtr2[i][j].cost;
    }
#else
    for (j = 0; j < num; j++)
      rensetu_mtr1[i_n][j] = rensetu_mtr2[i][j];
#endif
}

/***********************************************************************
 * condense_matrix
 ***********************************************************************/
static void condense_matrix()
{
    int i, j, k;
    int i_n = 0;
    int j_n = 0;

#ifndef VGRAM
    cell_num = tbl_num;
#endif

    for (i = 0; i < cell_num; i++) {
	rensetu_tbl[i].i_pos = i;
	rensetu_tbl[i].j_pos = i;
    }

    for (j = 0; j < tbl_num; j++)  {
	for (k = 0; k < j_n; k++) {
	    if (compare_vector1(k, j, cell_num)) {
		rensetu_tbl[j].j_pos = k;
		break;
	    }
	}
	if (rensetu_tbl[j].j_pos == j) {
	    copy_vector1(j, j_n, cell_num);
	    rensetu_tbl[j].j_pos = j_n++;
	}
    }

    for (i = 0; i < cell_num; i++)  {
	for (k = 0; k < i_n; k++) {
	    if (compare_vector2(k, i, j_n)) {
		rensetu_tbl[i].i_pos = k;
		break;
	    }
	}
	if (rensetu_tbl[i].i_pos == i) {
	    copy_vector2(i, i_n, j_n);
	    rensetu_tbl[i].i_pos = i_n++;
	}
    }

    i_num = i_n;
    j_num = j_n;

    /* print for check */
    fprintf(stderr, "matrix size: %d, %d\n", i_num, j_num);
}

/***********************************************************************
 * write_trble, write_matrix
 ***********************************************************************/
static void write_table(fp)
    FILE *fp;
{
    rensetu_pair_t *tbl;
    int i;

    fprintf(fp, "%d\n", cell_num);
    for (i = 0, tbl = &rensetu_tbl[0]; i < tbl_num; i++, tbl++) {
#ifdef VGRAM
	/* comment */
	fprintf(fp, "%s %s %s %s\n",
		Hinsi[tbl->hinsi].name ?
		Hinsi[tbl->hinsi].name : "(null)",
		tbl->type ? Type[tbl->type].name : "",
		tbl->form ? Form[tbl->type][tbl->form].name : "",
		tbl->goi ? tbl->goi : "");
	/* data */
	fprintf(fp, "%d %d %d %d %d %s\n",
		tbl->i_pos, tbl->j_pos, tbl->hinsi,
		tbl->type, tbl->form,
		tbl->goi ? tbl->goi : "*");
#else
	/* comment */
	fprintf(fp, "%s %s %s %s\n",
		Class[tbl->hinsi][tbl->bunrui].id ?
		Class[tbl->hinsi][tbl->bunrui].id : "(null) ",
		tbl->type ? Type[tbl->type].name : "",
		tbl->form ? Form[tbl->type][tbl->form].name : "",
		tbl->goi ? tbl->goi : "");
	/* data */
	fprintf(fp, "%d %d %d %d %d %d %s\n",
		tbl->i_pos, tbl->j_pos, tbl->hinsi, tbl->bunrui,
		tbl->type, tbl->form,
		tbl->goi ? tbl->goi : "*");
#endif
    }
#ifdef VGRAM
    for (; i < cell_num; i++, tbl++)
      fprintf(fp, ";\n%d -1 0 0 0 *\n", tbl->i_pos);
#endif
}

static void write_matrix(fp)
    FILE *fp;
{
    int i, j;
     
    fprintf(fp, "%d %d\n", i_num, j_num);

    for (i = 0; i < i_num; i++) {
	for (j = 0; j < j_num; j++) {
#ifdef VGRAM
	    fprintf(fp, "%d,%d ", (int)connect_mtr1[i][j].next,
		    (int)connect_mtr1[i][j].cost);
#else
	    fprintf(fp, "%d ", (int)rensetu_mtr1[i][j]);
#endif
	}
	fprintf(fp, "\n");
    }
}

/***********************************************************************
 * main
 ***********************************************************************/
int main(argc, argv)
    int argc;
    char **argv;
{
    FILE *fpt, *fpm, *fpk, *fpc;
    int lines;

    if (argv[1] && !strcmp(argv[1], "-i")) {
#ifdef VGRAM
	fputs("cha\n", stdout);
#else
	fputs("juman\n", stdout);
#endif
	exit(0);
    }

    set_progpath(argv[0]);

    /* .chasenrc ɤ߹ɬפʤ */

    /* ʸˡѡطե */
    read_grammar(stderr, 0, 0);
    read_katuyou(stderr, 0);
    read_kankei();

    /* Ϣܵ§եΥץ */
#ifdef VGRAM
    fpc = cha_fopen(CONNECTFILE, "r", 1);
#else
    fpc = cha_fopen2(CONNECTFILE, JM_CONNECTFILE, "r", 1);
#endif
    /* ϥեΥץ */
    fpt = cha_fopen(TABLEFILE, "w", 1);
    fpm = cha_fopen(MATRIXFILE, "w", 1);

    /* Ϣܵ§եν */
    fprintf(stderr, "parsing %s\n", CONNECTFILE);
    lines = make_rensetu_tbl(fpc);
    rewind(fpc);
    LineNo = 0;
    read_rensetu(fpc, lines);
    fclose(fpc);

    /* Ϣܹΰ */
    condense_matrix();
    write_table(fpt);
    write_matrix(fpm);

    fclose(fpt);
    fclose(fpm);

    return 0;
}
