<?php
/**
*
* cache driver for memcached
*
* PHP version 5
*
* @package    config_providers
* @author     CharcoalPHP Development Team
* @copyright  2008 - 2013 CharcoalPHP Development Team
*/
class Charcoal_FileCacheDriver extends Charcoal_CharcoalObject implements Charcoal_ICacheDriver
{
	private $_cache_root;
	private $_cache_root_dir;
	private $_default_duration;

	/*
	 *	Construct object
	 */
	public function __construct()
	{
		parent::__construct();
	}

	/*
	 * Configure component
	 *
	 * @param Charcoal_Config $config   configuration data
	 */
	public function configure( Charcoal_Config $config )
	{
		parent::configure( $config );

		$default_cache_root = Charcoal_ResourceLocator::getApplicationPath( s('cache') );

		$this->_cache_root        = $config->getString( s('cache_root'), s($default_cache_root) );
		$this->_default_duration  = $config->getInteger( s('default_duration'), i(0) );

		$this->_cache_root_dir = new Charcoal_File( $this->_cache_root );

		log_debug( "system, debug, cache", "cache", "cache_root=[{$this->_cache_root}]" );
		log_debug( "system, debug, cache", "cache", "default_duration=[{$this->_default_duration}]" );
	}

	/**
	 * Get cache data file object
	 */
	public function getCacheDataFile( Charcoal_String $key )
	{
		return new Charcoal_File( s("{$key}.data"), $this->_cache_root_dir );
	}

	/**
	 * Get cache meta file object
	 */
	public function getCacheMetaFile( Charcoal_String $key )
	{
		return new Charcoal_File( s("{$key}.meta"), $this->_cache_root_dir );
	}

	/**
	 * Get non-typed data which is associated with a string key
	 */
	public function get( Charcoal_String $key )
	{
		$meta_file = $this->getCacheMetaFile($key);
		$data_file = $this->getCacheDataFile($key);

		// read cache meta file
//		if ( !$meta_file->exists() || !$meta_file->canRead() ){
		if ( !$meta_file->exists() ){
			log_debug("system,debug,cache", "cache", "Can not read meta file[$meta_file]" );
			return FALSE;
		}

		$meta_data = parse_ini_file($meta_file->getPath());

		if ( $meta_data === FALSE || !is_array($meta_data) ){
			log_debug("system,debug,cache", "cache", "Can not parse meta file[$meta_file]" );
			return FALSE;
		}

		// check expire date
		if ( !isset($meta_data['expire_date']) ){
			log_warning("system,debug,cache", "cache", "Not found mandatory field[expire_date] in meta file[$meta_file]" );
			return FALSE;
		}

		$expire_date = $meta_data['expire_date'];
		$expire_date = ($expire_date == 0) ? 0 : strtotime($expire_date);

		if ( $expire_date === FALSE ){
			log_warning("system,debug,cache", "cache", "field[expire_date] must be date value or zero in meta file[$meta_file]" );
			return FALSE;
		}

		if ( $expire_date > 0 && $expire_date < time() ){
			log_debug("system,debug,cache", "cache", "Cache expired: [$meta_file]" );
			return FALSE;
		}

		// read cache data file
		if ( !$data_file->exists() || !$data_file->canRead() ){
			log_debug("system,debug,cache"," cache", "Can not read data file[$meta_file]" );
			return FALSE;
		}

		$serialized_data = $data_file->getContents();

		// check SHA1 digest
		if ( !isset($meta_data['sha1_digest']) ){
			log_warning("system,debug,cache", "cache", "Not found mandatory field[sha1_digest] in meta file[$meta_file]" );
			return FALSE;
		}

		$sha1_digest1 = sha1($serialized_data);
		$sha1_digest2 = $meta_data['sha1_digest'];
		if ( $sha1_digest1 != $sha1_digest2 ){
			log_warning("system,debug,cache", "cache", "Data file SHA1 digest does not match the value in meta file[$meta_file] digest1=[$sha1_digest1] digest2=[$sha1_digest2]" );
			return FALSE;
		}

		log_info("system,debug,cache", "cache", "cache[$key] HIT!" );

		return unserialize($serialized_data);
	}

	/**
	 * Save a value to cache
	 */
	public function set( Charcoal_String $key, Charcoal_Object $value, Charcoal_Integer $duration = NULL )
	{
		$meta_file = $this->getCacheMetaFile($key);
		$data_file = $this->getCacheDataFile($key);

		$duration = $duration ? ui($duration) : ui($this->_default_duration);

		$serialized_data = serialize($value);
		$sha1_digest = sha1($serialized_data);

		// save meta file
		$meta_data['expire_date'] = ($duration == 0) ? 0 : date('Y-m-d H:i:s', strtotime("+{$duration} seconds"));
		$meta_data['sha1_digest'] = $sha1_digest;

		$fp = @fopen( $meta_file->getPath(), 'w' );
		if ( $fp === FALSE ){
			_throw( new Charcoal_CacheDriverException( s('file'), s("cache set failed. Saving meta file failed: [$meta_file]") ) );
		}

		foreach( $meta_data as $meta_key => $meta_value ){
			fwrite( $fp, "{$meta_key} = {$meta_value}\n" );
		}

		fclose( $fp );
	
		// save data file
		$res = $data_file->putContents( s($serialized_data) );
		if ( $res === FALSE ){
			_throw( new Charcoal_CacheDriverException( s('file'), s("cache set failed. Saving data file failed: [$data_file]") ) );
		}
	}

	/**
	 * Delete a cache data
	 */
	public function delete( Charcoal_String $key )
	{
		$data_file = $this->getCacheDataFile($key);

		if ( $data_file->exists() && $data_file->isFile() ){
			$data_file->delete();
		}
	}

}
return __FILE__;
