<?php
/**
* 設定値を保持するクラス
*
* PHP version 5
*
* @package    core
* @author     CharcoalPHP Development Team
* @copyright  2008 - 2013 CharcoalPHP Development Team
*/

class Charcoal_Config extends Charcoal_HashMap
{
	/*
	 *    コンストラクタ
	 */
	public function __construct( array $data = NULL )
	{
		parent::__construct( $data );
	}

	/*
	 * 他の設定ファイルから設定値を上書きする
	 */
	public function import( Charcoal_ObjectPath $object_path, Charcoal_String $type_name )
	{
		Charcoal_ConfigLoader::loadConfig( $object_path, $type_name, $this );
	}

	/*
	 * 設定値を文字列として取得する
	 */
	public function getString( Charcoal_String $key, Charcoal_String $default_value = NULL )
	{
		$value = parent::get( us($key) );

		// 未設定なら空の文字列を返す
		if ( NULL === $value ){
			return $default_value;
		}

		// フォーマット確認
		if ( !is_string($value) && !($value instanceof Charcoal_String) ){
			_throw( new Charcoal_StringFormatException( $value, "key=[$key]" ) );
		}

		// マクロ展開
		$value = Charcoal_ConfigUtil::processMacro( s($value) );

		return s($value);
	}

	/*
	 * 設定値を配列として取得する
	 */
	public function getArray( Charcoal_String $key, Charcoal_Vector $default_value = NULL )
	{
		$value = parent::get( us($key) );

		// 未設定なら空の配列を返す
		if ( NULL === $value ){
			return $default_value;
		}

		// カンマで分割
		$array = explode( ',', $value );

		// 要素内の空白を削除
		foreach( $array as $_key => $value ){
			$value = trim($value);
			if ( strlen($value) == 0 ){
				unset( $array[$_key] );
			}
			else{
				$array[$_key] = us( $value );
			}
		}

		// 要素が１つで空ならデフォルト値を返す
		if ( count($array) == 1 && $array[0] === '' ){
			return $default_value !== NULL ? $default_value : new Charcoal_Vector();
		}

		// フォーマット確認
		if ( !is_array($array) ){
			_throw( new Charcoal_ArrayFormatException( $value, "key=[$key]" ) );
		}

		// マクロ展開
		foreach( $array as $_key => $value ){
			$value = Charcoal_ConfigUtil::processMacro(s($value));
			$array[$_key] = us($value);
		}

		// 配列を返却
		return  v($array);
	}

	/*
	 * 設定値をブール値として取得する
	 */
	public function getBoolean( Charcoal_String $key, Charcoal_Boolean $default_value = NULL )
	{
		$value = parent::get( us($key) );

		// 未設定ならFALSEを返す
		if ( NULL === $value ){
			return $default_value;
		}

		if ( is_string($value) ){
			$value = (strlen($value) > 0 );
		}

		// フォーマット確認
		if ( !is_bool($value) && !($value instanceof Charcoal_Boolean) ){
			_throw( new BooleanFormatException( $value, "key=[$key]" ) );
		}

		// ブール型にして返却
		return b($value);
	}

	/*
	 * 設定値を整数値として取得する
	 */
	public function getInteger( Charcoal_String $key, Charcoal_Integer $default_value = NULL )
	{
		$value = parent::get( us($key) );

		// 未設定ならNULLを返す
		if ( NULL === $value ){
			return $default_value;
		}

		// フォーマット確認
		if ( $value instanceof Charcoal_Integer ){
			return $value;
		}
		if ( !is_numeric($value) ){
			_throw( new Charcoal_IntegerFormatException( $value, "key=[$key]" ) );
		}

		// 整数型にして返却
		return i($value);
	}

}
return __FILE__;