require 'digest/md5'

unless Hash.method_defined?(:key)   # Ruby 1.8.x
  class Hash
    alias key index
  end
end

module CGIKit

  # A class for session management. Session objects have a hash of
  # arbitary objects and information about browser name, IP address, etc.
  # However, you can't set objects that can't be marshal ( IO, Proc, etc. )
  # to the session with default database manager FileSessionStore.
  class Session
    include Logging

    DEFAULT_SESSION_ID_FIGURES = 16
    DEFAULT_TIMEOUT = 60 * 60 * 24 * 7

    class << self
      def session_id?( session_id )
        session_id and (/\A([0-9A-Za-z]+)\Z/ === session_id)
      end

      def create_session_id
        md5 = Digest::MD5::new
        md5.update Time.now.to_s
        md5.update rand(0).to_s
        md5.update $$.to_s
        md5.hexdigest[0, session_id_figures]
      end

      def session_id_figures
        DEFAULT_SESSION_ID_FIGURES
      end
    end


    # A hash of arbitary objects.
    attr_accessor :values

    # Session ID.
    attr_accessor :session_id

    # Seconds until the session has timed out.
    attr_accessor :timeout

    # Name of browser.
    attr_accessor :user_agent

    # IP address.
    attr_accessor :remote_addr

    attr_accessor :context, :application, :session_store, :last_accessed_time, \
    :caches, :context_ids, :frame_components, :permanent_caches, :session_key, \
    :cookie_expires, :editing_context
    alias ec editing_context

    def initialize( session_id = nil )
      unless Session.session_id? session_id then
        session_id = Session.create_session_id
      end
      @session_id         = session_id
      @last_accessed_time = Time.new
      @context_ids        = {}
      @caches             = {}
      @permanent_caches   = {}
      @values             = {}
      @frame_components   = {}
      @user_agent         = nil
      @remote_addr        = nil
      @timeout            = DEFAULT_TIMEOUT
      @terminate          = false
      @last_component_id  = -1
      init
    end

    #
    # hook
    
    def init; end


    #
    # accessing
    #

    def []( key )
      @values[key]
    end

    def []=( key, value )
      @values[key] = value
    end

    def remove( key )
      @values.delete(key)
    end

    def terminate
      @terminate = true
    end

    def terminate?
      @terminate or timeout?
    end


    #
    # accessing cached components
    #

    def add_component( component, context = nil, permanently = false )
      unless component_id = component_id(component) then
        component_id = next_component_id()
        if context then
          context_id = context.context_id
        else
          context_id = "#{component_id}.0"
        end
        add_component_for_ids(component, component_id, \
                              context_id, permanently)
      end
      component_id
    end

    def next_component_id
      @last_component_id += 1
    end

    def add_component_for_ids( component,
                               component_id,
                               context_id,
                               permanently = false )
      if permanently then
        page_caches = @permanent_caches
      else
        page_caches = @caches
      end
      page_caches[component_id] = component
      @context_ids[context_id] = component_id
    end

    def component_id( component )
      if @caches.value?(component) then
        @caches.key(component)
      elsif @permanent_caches.value?(component) then
        @permanent_caches.index(component)
      else
        nil
      end
    end

    def component_for_component_id( component_id )
      @caches[component_id] || @permanent_caches[component_id]
    end

    def component_for_context_id( context_id )
      page = nil
      unless context_id.nil? then
        ids = context_id.split(Context::SEPARATOR)
        while ids.size > 1
          id = ids.join(Context::SEPARATOR)
          if page = _component(id) then break end
          ids.pop
          ids.push(0)
          id = ids.join(Context::SEPARATOR)
          if page = _component(id) then break end
          ids.pop
        end
      end
      page
    end

    alias component component_for_context_id

    private

    def _component( context_id )
      if id = @context_ids[context_id] then
        @caches[id] || @permanent_caches[id]
      end
    end

    public

    #
    # testing
    #

    # Enables or disables the use of URLs for storing session IDs.
    def store_in_url?
      @application.store_in_url
    end

    # Enables or disables the use of cookies for storing session IDs.
    def store_in_cookie?
      @application.store_in_cookie
    end

    # Enables or disables session authorization by browsers.
    def auth_by_user_agent?
      @application.auth_by_user_agent
    end

    # Enables or disables session authorization by IP addresses.
    def auth_by_remote_addr?
      @application.auth_by_remote_addr
    end

    # Returns true if the browser is equal to one when the session created.
    def user_agent?( user_agent )
      auth_by_user_agent? and (@user_agent == user_agent)
    end

    # Returns true if the IP address is equal to one when the session created.
    def remote_addr?( remote_addr )
      auth_by_remote_addr? and (@remote_addr == remote_addr)
    end

    # Returns true if the session isn't expired.
    def timeout?
      if @timeout then
        (@timeout != 0) and (timeout_expire <= Time.new)
      else
        false
      end
    end

    def timeout_expire
      @last_accessed_time + @timeout
    end


    #
    # Handling requests
    #

    def take_values_from_request( request, context )
      context.component.delegate_take_values_from_request(request, context)
    end

    def invoke_action( request, context )
      context.component.delegate_invoke_action(request, context)
    end

    def append_to_response( response, context )
      context.component.delegate_append_to_response(response, context)
    end


    #
    # Page management
    #

    def save_page( component, permanently = false )
      if permanently then
        caches = @permanent_caches
        size = @application.permanent_page_cache_size
      else
        caches = @caches
        size = @application.page_cache_size
      end
      component.add_all_components(self, permanently)
      if component.context then
        cid = component.context.context_id
        @context_ids[cid] = component_id(component)
      end
      _remove_old_cache(caches, size)
    end

    private

    def _remove_old_cache( caches, size )
      removes = caches.size - size
      if removes > 0 then
        if removed_keys = caches.keys.sort.slice!(0..removes-1) then
          caches.delete_if do |key, value|
            if removed_keys.include?(key) then
              true
            end
          end
        end
      end
    end

    public

    def restore_page( context_id )
      component(context_id)
    end


    #
    # marshaling
    #

    def marshal_dump
      dump                      = {}
      dump[:timeout]            = @timeout
      dump[:values]             = @values
      dump[:session_id]         = @session_id
      dump[:user_agent]         = @user_agent
      dump[:remote_addr]        = @remote_addr
      dump[:caches]             = @caches
      dump[:permanent_caches]   = @permanent_caches
      dump[:context_ids]        = @context_ids
      dump[:frame_components]   = @frame_components
      dump[:last_accessed_time] = @last_accessed_time
      dump[:last_component_id]  = @last_component_id
      dump[:cookie_expires]     = @cookie_expires
      dump
    end

    def marshal_load( object )
      @timeout            = object[:timeout]
      @values             = object[:values]
      @session_id         = object[:session_id]
      @user_agent         = object[:user_agent]
      @remote_addr        = object[:remote_addr]
      @context_ids        = object[:context_ids]
      @caches             = object[:caches]
      @permanent_caches   = object[:permanent_caches]
      @frame_components   = object[:frame_components]
      @last_accessed_time = object[:last_accessed_time]
      @last_component_id  = object[:last_component_id]
      @cookie_expires     = object[:cookie_expires]
    end

    def awake_from_restoration( application, request )
      @application         = application
      @timeout             = application.timeout
      @cookie_expires      = application.session_cookie_expires
      @session_key         = application.session_key
      @session_store       = application.session_store
      @last_accessed_time  = Time.new
      @user_agent          = request.user_agent
      @remote_addr         = request.remote_addr
      if defined?(TapKit::EditingContext) and @application.database then
        @editing_context = @application.database.create_editing_context
      end
    end


    #
    # validating
    #

    def validate
      validate_authorization
      validate_timeout
    end

    def validate_authorization
      if (auth_by_user_agent? and \
          !user_agent?(@context.request.user_agent)) or \
        (auth_by_remote_addr? and \
         !remote_addr?(@context.request.remote_addr)) then
        raise Application::SessionAuthorizationError, 'Your session is not authorized.'
      end
    end

    def validate_timeout
      if timeout? then
        flush
        raise Application::SessionTimeoutError, 'Your session has timed out.'
      end
    end

    def flush
      @session_store.remove(@session_id)
      if store_in_cookie? then
        remove_cookie(@context.response)
      end
    end


    #
    # managing cookie
    #

    def remove_cookie
      cookie         = Cookie.new(@session_key)
      cookie.expires = Time.new - 600
      @context.response.add_cookie(cookie)
    end

    def set_cookie
      cookie = Cookie.new(@session_key, @session_id)
      cookie.expires = Time.new + @cookie_expires
      @context.response.add_cookie(cookie)
    end

  end

end
