module CGIKit

# == Context ID Format
# * When generating a component, 0 appends to context ID.
# * When generating a dynamic element, last element ID is incremented.
#   If in form, "name" attribute will be setted value or the last element ID.
# * In form, 0 appends to context ID.
#   The 0 last element ID is deleted at end of the form.
class Context

  SEPARATOR = '.'

  attr_accessor :request, :response, :application, :sender_id, :in_form
  attr_accessor :component, :request_handler_key

  attr_writer :session

  alias in_form? in_form

  def initialize( request = nil, application = nil )
    @request = request
    @response = Response.new
    @application = application
    @contexts = []
    @counts = []
    @session = nil
    @sender_id = request.context_id if request
  end

  def context_id( digit = false )
    if @contexts.empty? then
      ''
    elsif digit == true
      copied = @contexts.dup
      copied.pop
      copied.push(@counts.last)
      copied.join(SEPARATOR)
    else
      @contexts.join(SEPARATOR)
    end
  end

  def context_id=( id )
    delete_all
    id.to_s.split(SEPARATOR).each do |item|
      if item =~ /\A\d+\Z/ then
        @contexts << item.to_i
        @counts << item.to_i
      else
        @contexts << item
      end
    end
  end

  def element_id
    @contexts.last.to_s
  end

  def component_id
    @contexts.first
  end

  def component_id=( cid )
    @contexts[0] = cid
  end 

  def has_session?
    !@session.nil?
  end

  # Returns the session. If the receiver doesn't have a session,
  # it creates and returns a new session.
  def session
    unless has_session? then
      session = @application.restore_session(@request.session_id, self)
      unless session then
        session = @application.create_session(@request)
      end
      @session = session
      init_component_id_with_session
    end
    @session
  end

  def init_component_id_with_session( component = @component )
    unless component_id = session.component_id(component) then
      # register current component with ID for creating new session
      component_id = component_id() || @session.next_component_id
      @session.add_component_for_ids(component, component_id, context_id())
      @session.save_page(component)
      component_id = session.component_id(component)
    end
    init_context_id_and_component_id(component, component_id)
  end

  def init_component_id_without_session( component )
    if @component_id then
      component_id = @component_id + 1
    else
      component_id = 0
    end
    init_context_id_and_component_id(component, component_id)
  end

  def init_context_id_and_component_id( component, component_id )
    @component = component
    self.component_id = component_id
  end

  def action?( request, context_id = nil )
    context_id ||= self.context_id
    context_id == request.context_id
  end

  def in_form=( flag )
    @in_form = flag
    if @in_form == true then
      @form_id = context_id
    else
      @form_id = nil
    end
  end

  def current_form?( request )
    @form_id and (@form_id == request.context_id)
  end


  #
  # manipulating element ID
  #

  def append( string )
    @contexts << string
    _append_zero_to_count
  end

  alias << append

  def append_zero
    @contexts << 0
    _append_zero_to_count
  end

  def delete_all
    @contexts.clear
    _delete_all_of_count
  end

  def delete
    @contexts.pop
    _delete_last_of_count
  end

  def increment( string = nil )
    if string then
      @contexts.pop
      @contexts << string
      _increment_count
    else
      _increment_count
      @contexts.pop
      @contexts << @counts.last
    end
  end

  private

  def _append_zero_to_count
    @counts << 0
  end

  def _delete_all_of_count
    @counts.clear
  end

  def _delete_last_of_count
    @counts.pop
  end

  def _increment_count
    index = @contexts.size - 1
    @counts[index] ||= 0
    @counts[index] += 1
  end

  public

  #
  # generating URLs
  #

  def direct_action_url( action_class, action_name, query = {}, sid = true )
    action_class ||= @application.direct_action_class || CGIKit::DirectAction
    app = @session || @application
    url = @application.direct_action_url(self, action_class, action_name, query, sid)
    url
  end

  def component_action_url( query = {}, is_secure = false )
    complete_url(@application.component_request_handler_key, nil, query, is_secure)
  end

  def complete_url( key, path, query, is_secure, port = nil )
    @application.url(key, self, path, query, is_secure, port)
  end

  def url( key, path = nil, query = {} )
    complete_url(key, path, query, false)
  end

end

end
