module CGIKit

  # ByteData objects manage bytes.
  class ByteData

    # Path of a file saving the bytes.
    attr_accessor :path

    # Content type of the bytes.
    attr_accessor :content_type

    class << self

      # Create an instance from specified IO object.
      # If you give this a File object, sets "path" attribute of the instance.
      def new_with_io( io, offset = nil, count = nil )
        io.pos = offset if offset
        bytes  = new io.read(count)
        if io.respond_to? 'path' then
          bytes.path = io.path
        end
        bytes
      end

      # Create an instance from specified file.
      def new_with_file( filename )
        bytes = nil
        open(filename) do |f|
          bytes      = new(f.read)
          bytes.path = f.path
          if ext = File.extname(filename) then
            ext.tr!('.', '')
            bytes.content_type = ResourceManager.mime(ext)
          end
        end
        bytes
      end

    end

    attr_accessor :tmp
    alias tmp? tmp

    def initialize( string = nil )
      @bytes = string || ''
    end

    def tempfile?
      false
    end

    # Returns bytes with spcified length or whole length if you omit it.
    def bytes( length = nil )
      if length then
        @bytes.slice(0, length)
      else
        @bytes.to_s
      end
    end

    # Executes the block for every byte.
    def each
      @bytes.each_byte do |byte|
        yield byte
      end
    end

    # Returns true if the bytes of each objects are equal.
    def ==( bytes )
      @bytes == bytes.bytes
    end

    # Returns length of the bytes.
    def length
      @bytes.size
    end
    alias size length

    # Appends bytes to the bytes.
    def <<( bytes )
      if bytes.is_a?(ByteData) then
        @bytes << bytes.bytes
      else
        @bytes << bytes
      end
      self
    end

    # Writes the bytes to a specified file.
    def write_to_file( filename, lock = true )
      if lock then
        FileLock.exclusive_lock(filename, 'w+b') do |file|
          file.write to_s
        end
      else
        File.open(filename, 'w+b') do |file|
          file.write to_s
        end
      end      
    end

    # Returns the object as a string.
    def to_s
      @bytes.to_s
    end

    def open; end
    def close; end

  end


  class TempfileByteData < ByteData

    attr_accessor :tempfile

    def initialize( tempfile )
      @tempfile = tempfile
      close
    end

    def tempfile?
      true
    end

    # Returns bytes with spcified length or whole length if you omit it.
    def bytes( length = nil )
      open do
        @tempfile.read(length)
      end
    end

    # Executes the block for every byte.
    def each
      open do
        @tempfile.each_byte do |byte|
          yield byte
        end
      end
    end

    def ==( bytes )
      @tempfile == bytes.tempfile
    end

    def length
      open do
        @tempfile.size
      end
    end

    def <<( bytes )
      open do
        @tempfile.seek(0, IO::SEEK_END)
        if ByteData === bytes then
          @tempfile << bytes.bytes
        else
          @tempfile << bytes
        end
        self
      end
    end

    def to_s
      bytes
    end

    def open( &block )
      @tempfile.open if @tempfile.closed?
      if block_given? then
        @tempfile.rewind
        value = block.call
        close
        return value
      end
    end

    def close
      @tempfile.close unless @tempfile.closed?
    end

    def _dump( limit )
      Marshal.dump(bytes(), limit)
    end

    def self._load( object )
      ByteData.new(Marshal.load(object))
    end

  end

end

